package io.fullview.fullview_sdk.ongoing

import android.annotation.SuppressLint
import android.content.res.ColorStateList
import android.content.res.Configuration
import android.os.Bundle
import android.util.DisplayMetrics
import android.view.KeyEvent
import android.view.LayoutInflater
import android.view.MotionEvent
import android.view.View
import android.view.ViewGroup
import android.view.animation.LinearInterpolator
import android.widget.ImageButton
import android.widget.TextView
import androidx.annotation.ColorRes
import androidx.annotation.DrawableRes
import androidx.appcompat.app.AlertDialog
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.core.content.ContextCompat
import androidx.core.view.isVisible
import androidx.fragment.app.DialogFragment
import androidx.fragment.app.viewModels
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import co.daily.model.CallState
import co.daily.model.Participant
import co.daily.model.ParticipantId
import co.daily.view.VideoView
import io.fullview.fullview_sdk.DailyState
import io.fullview.fullview_sdk.DailyStateListener
import io.fullview.fullview_sdk.HostType
import io.fullview.fullview_sdk.data.WebRTCConfig
import io.fullview.fullview_sdk.ui.ToggleImageButton
import io.fullview.fullview_sdk.R
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.filterNotNull
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.launch
import kotlin.time.Duration.Companion.seconds

const val ARG_AGENT_NAME = "fullview_agent_name"
const val ARG_COBROWSE_ID = "fullview_cobrowse_id"
const val ARG_WEB_RTC_CONFIG = "fullview_webRTC_config"
const val ARG_HOST = "fullview_host"

class OngoingCallFragment : DialogFragment(), DailyStateListener {

    companion object {

        const val TAG = "OngoingCallFragment"

        @JvmStatic
        fun newInstance(invitationId: String, agentName: String, config: WebRTCConfig, hostType: HostType) = OngoingCallFragment().apply {
            arguments = Bundle().apply {
                putString(ARG_AGENT_NAME, agentName)
                putString(ARG_COBROWSE_ID, invitationId)
                putParcelable(ARG_WEB_RTC_CONFIG, config)
                putParcelable(ARG_HOST, hostType)
            }
        }
    }

    private lateinit var agentName: String
    private lateinit var id: String
    private lateinit var webRTCConfig: WebRTCConfig
    private lateinit var hostType: HostType
    private var isPortraitOrientation: Boolean = true
    private var listener : OngoingCallInterface? = null
    private var localParticipant: Participant? = null
    private var hostOverlay = true
    private val displayMetrics = DisplayMetrics()
    private var initialSetupDone = false

    // Create map of video views
    private val videoViews = mutableMapOf<ParticipantId, VideoView>()

    private lateinit var hostVideoView : VideoView
    private lateinit var agentVideoView : VideoView
    private lateinit var overlay : ConstraintLayout
    private lateinit var overlayVideo : VideoView
    private lateinit var hubView : View
    private lateinit var root : View
    private lateinit var toggleMic : ToggleImageButton
    private lateinit var toggleCamera : ToggleImageButton
    private lateinit var screenShare : ToggleImageButton
    private lateinit var miniHub : View
    private lateinit var hub : View


    private val viewModel: OngoingCallViewModel by viewModels<OngoingCallViewModel>()

    fun setListener(listener: OngoingCallInterface?) {
        this.listener = listener
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        val currentOrientation = resources.configuration.orientation
        viewModel.updateOrientation(currentOrientation)

        arguments?.let {
            agentName = it.getString(ARG_AGENT_NAME) ?: "Unknown"
            id = it.getString(ARG_COBROWSE_ID) ?: "-"
            webRTCConfig = it.getParcelable(ARG_WEB_RTC_CONFIG)!!
            hostType = it.getParcelable<HostType>(ARG_HOST)!!
        }
    }

    override fun onResume() {
        super.onResume()
        listener?.attachService(TAG)
    }

    @SuppressLint("ClickableViewAccessibility")
    override fun onCreateView(inflater: LayoutInflater, container: ViewGroup?,
                              savedInstanceState: Bundle?): View {
        root =  inflater.inflate(R.layout.fragment_ongoing_call, container, false)
        hostVideoView = root.findViewById(R.id.host_video)
        agentVideoView = root.findViewById(R.id.agent_video)
        toggleMic = root.findViewById(R.id.toggle_mic)
        toggleCamera = root.findViewById(R.id.toggle_camera)
        miniHub = root.findViewById(R.id.mini_hub)
        hub = root.findViewById(R.id.fullsize_bar)
        hubView = root.findViewById(R.id.draggable_view)
        screenShare = root.findViewById(R.id.screen_share)

        requireActivity().windowManager.defaultDisplay.getMetrics(displayMetrics)

        if (viewModel.startOfCall.replayCache.isEmpty()) {
            viewModel.setStart()
        }

        hostVideoView.setOnClickListener {
            listener?.startFullscreen()
        }

        agentVideoView.setOnClickListener {
            listener?.startFullscreen()
        }

        root.findViewById<ImageButton>(R.id.end_call).setOnClickListener {
            viewModel.setTouchTimestamp(System.currentTimeMillis())

            AlertDialog.Builder(requireContext(), R.style.AlertDialog_AppCompat_Fullview)
                .setTitle(getString(R.string.end_cobrowse))
                .setMessage(getString(R.string.are_you_sure_you_want_to_end_the_current_cobrowsing_session))
                .setPositiveButton(android.R.string.yes
                ) { _, _ ->
                    viewModel.clearStart()
                    listener?.onEndCall()
                }
                .setNegativeButton(android.R.string.cancel) { _ , _ -> }
                .show()
        }

        root.findViewById<ImageButton>(R.id.hide_hub).setOnClickListener {
            miniHub.visibility = View.VISIBLE
            hub.visibility = View.GONE
        }

        root.findViewById<View>(R.id.mini_hub_click).setOnClickListener {
            miniHub.visibility = View.GONE
            hub.visibility = View.VISIBLE
        }


        val currentOrientation = resources.configuration.orientation
        isPortraitOrientation = currentOrientation == Configuration.ORIENTATION_PORTRAIT

        var dX = 0f
        var dY = 0f
        var lastAction = 0
        hubView.setOnTouchListener { v, ev ->
            viewModel.setTouchTimestamp(System.currentTimeMillis())
            when(ev.actionMasked){
                MotionEvent.ACTION_DOWN -> {
                    dX = v.x - ev.rawX
                    dY = v.y - ev.rawY
                    lastAction = ev.actionMasked
                    return@setOnTouchListener false
                }
                MotionEvent.ACTION_MOVE -> {
                    if((ev.rawX + dX) >= 0 && (ev.rawX + dX + v.measuredWidth) <= displayMetrics.widthPixels) {
                        v.x = (ev.rawX + dX)
                    }

                    if((ev.rawY + dY) >= 0 && (ev.rawY + dY + v.measuredHeight) <= displayMetrics.heightPixels) {
                        v.y = (ev.rawY + dY)
                    }
                    lastAction = ev.actionMasked
                }

                MotionEvent.ACTION_UP -> {
                    lastAction = ev.actionMasked
                    viewModel.updateHubCoordinates(v.x, v.y, isPortraitOrientation)
                }
                else -> return@setOnTouchListener false
            }

            return@setOnTouchListener true
        }

        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.RESUMED) {
                viewModel.tickerFlow(1.seconds)
                    .onEach {
                        viewModel.startOfCall.replayCache.lastOrNull()?.apply {
                            val secondsSinceStart = (System.currentTimeMillis() - this)/1000
                            requireActivity().runOnUiThread {
                                root.findViewById<TextView>(R.id.timer).text = getString(
                                    R.string.fullview_timer,
                                    (secondsSinceStart / 60).toString().padStart(2, '0'),
                                    (secondsSinceStart % 60).toString().padStart(2, '0')
                                )
                            }
                        }
                    }
                    .launchIn(this@launch)
            }
        }

        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.RESUMED) {
                viewModel.tickerFlow(1.seconds)
                    .combine(viewModel.touchTimestamp) { _, lastTouch ->
                        lastTouch
                    }
                    .collectLatest {
                        val diff = System.currentTimeMillis() - it
                        if (diff >= 5000) {

                            if ((hostVideoView.isVisible || agentVideoView.isVisible).not() && hubView.alpha >= 1f) {
                                hubView.animate().apply {
                                    interpolator = LinearInterpolator()
                                    duration = 300
                                    alpha(0.5f)
                                    start()
                                }
                            }
                        } else {
                            if (hubView.alpha < 1f) {
                                hubView.animate().apply {
                                    interpolator = LinearInterpolator()
                                    duration = 300
                                    alpha(1f)
                                    start()
                                }
                            }
                        }
                    }
            }
        }

        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.RESUMED) {
                viewModel
                    .dailyState
                    .filterNotNull()
                    .collectLatest {
                        when(it.status) {
                            CallState.initialized -> {}
                            CallState.joining -> {}
                            CallState.joined -> {
                                if (webRTCConfig.enableAudio) {
                                    toggleMic.visibility = View.VISIBLE
                                }

                                if (webRTCConfig.enableVideo) {
                                    toggleCamera.visibility = View.VISIBLE
                                }

                                screenShare.visibility = View.VISIBLE

                                val local = it.allParticipants.values.first { it.info.isLocal }
                                localParticipant = local
                                videoViews[local.id] = hostVideoView

                                if (it.localParticipantTrack != null) {
                                    hostVideoView.track = it.localParticipantTrack
                                    hostVideoView.visibility = View.VISIBLE
                                } else {
                                    hostVideoView.track = null
                                    hostVideoView.visibility = View.GONE
                                }

                                val agent = it.allParticipants.values.firstOrNull { it.info.isLocal.not() }
                                if(agent?.media?.camera?.track !== null) {
                                    agentVideoView.track = agent.media?.camera?.track
                                    agentVideoView.visibility = View.VISIBLE
                                    videoViews[agent.id] = agentVideoView
                                } else {
                                    agentVideoView.track = null
                                    agentVideoView.visibility = View.GONE
                                }

                                toggleButtonUI(
                                    toggleCamera,
                                    it.inputs.cameraEnabled,
                                    enabledDrawable = R.drawable.baseline_videocam_24,
                                    disabledDrawable = R.drawable.baseline_videocam_off_24
                                )

                                toggleButtonUI(
                                    toggleMic,
                                    it.inputs.micEnabled,
                                    enabledDrawable = R.drawable.baseline_mic_24,
                                    disabledDrawable = R.drawable.baseline_mic_off_24
                                )

                                toggleButtonUI(
                                    screenShare,
                                    it.inputs.screenVideoEnabled,
                                    enabledDrawable = R.drawable.baseline_mobile_screen_share_24,
                                    disabledBackgroundColor = R.color.fullview_ios_dark_gray,
                                    disabledDrawable = R.drawable.baseline_mobile_screen_share_24,
                                    disabledIconTint = R.color.offwhite
                                )


                                setupToggles()
                            }
                            CallState.leaving -> {}
                            CallState.left -> {}
                        }
                    }
            }
        }

        return root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        root.findViewById<TextView>(R.id.cobrowsing_with).text = agentName

        if (viewModel.hubCoordinates.replayCache.isEmpty() || (viewModel.hubCoordinates.replayCache.lastOrNull()?.portraitMode == isPortraitOrientation)) {
            root.findViewById<View>(R.id.draggable_view).apply {
                measure(0,0)
                this.x = 150f
                this.y = (rootView.measuredHeight - this.measuredHeight) - 300f

                viewModel.updateHubCoordinates(
                    150f,
                    (rootView.measuredHeight - this.measuredHeight) - 300f,
                    isPortraitOrientation
                )
            }
        } else {
            viewModel.hubCoordinates.replayCache.lastOrNull()?.let {
                root.findViewById<View>(R.id.draggable_view).apply {
                    x = it.x
                    y = it.y
                }
            }
        }
    }

    private fun toggleButtonUI(
        buttonView: ToggleImageButton?,
        isChecked: Boolean,
        @ColorRes enabledBackgroundColor: Int = R.color.fullview_ios_enabled_background,
        @DrawableRes enabledDrawable: Int,
        @ColorRes enabledIconTint: Int = R.color.offwhite,
        @ColorRes disabledBackgroundColor: Int = R.color.fullview_ios_disabled_background,
        @DrawableRes disabledDrawable: Int,
        @ColorRes disabledIconTint: Int = R.color.fullview_ios_disabled_foreground
    ) {
        if (isChecked) {
            buttonView?.backgroundTintList = ColorStateList.valueOf(ContextCompat.getColor(requireContext(),
                enabledBackgroundColor
            ))
            buttonView?.setImageResource(enabledDrawable)
            buttonView?.imageTintList = ColorStateList.valueOf(ContextCompat.getColor(requireContext(),
                enabledIconTint
            ))
        } else {
            buttonView?.backgroundTintList = ColorStateList.valueOf(ContextCompat.getColor(requireContext(),
                disabledBackgroundColor
            ))
            buttonView?.setImageResource(disabledDrawable)
            buttonView?.imageTintList = ColorStateList.valueOf(ContextCompat.getColor(requireContext(),
                disabledIconTint
            ))
        }
    }

    private fun closeOverlay() {
        localParticipant?.apply {
            if (hostOverlay) {
                hostVideoView.track = overlayVideo.track
                videoViews[id] = hostVideoView
                overlayVideo.track = null
            } else {
                val agentId = videoViews.keys.firstOrNull { it != id }
                agentId?.apply {
                    agentVideoView.track = overlayVideo.track
                    videoViews[id] = agentVideoView
                    overlayVideo.track = null
                }
            }
        }
        overlay.visibility = View.GONE
    }

    override fun onStateChanged(newState: DailyState) {
        if (isResumed) {
            viewModel.setDailyState(newState)
        }
    }

    override fun onError(msg: String) {

    }

    @SuppressLint("ClickableViewAccessibility")
    private fun setupToggles() {
        if (hostType == HostType.FLUTTER) {
            toggleCamera.setOnClickListener {
                viewModel.setTouchTimestamp(System.currentTimeMillis())
                listener?.toggleCamera()
            }

            toggleMic.setOnClickListener {
                viewModel.setTouchTimestamp(System.currentTimeMillis())
                listener?.toggleMicrophone()
            }

            screenShare.setOnClickListener {
                viewModel.setTouchTimestamp(System.currentTimeMillis())
                listener?.toggleScreenShare()
            }
        } else {
            toggleCamera.setOnTouchListener { v, event ->
                if (event.metaState != KeyEvent.META_META_ON) {
                    if (event.actionMasked == MotionEvent.ACTION_UP) {
                        viewModel.setTouchTimestamp(System.currentTimeMillis())
                        listener?.toggleCamera()

                        return@setOnTouchListener true
                    }
                }
                return@setOnTouchListener false
            }

            toggleMic.setOnTouchListener { v, event ->
                if (event.metaState != KeyEvent.META_META_ON) {
                    if (event.actionMasked == MotionEvent.ACTION_UP) {
                        viewModel.setTouchTimestamp(System.currentTimeMillis())
                        listener?.toggleMicrophone()

                        return@setOnTouchListener true
                    }
                }
                return@setOnTouchListener false
            }

            screenShare.setOnTouchListener { v, event ->
                if (event.metaState != KeyEvent.META_META_ON) {
                    if (event.actionMasked == MotionEvent.ACTION_UP) {
                        viewModel.setTouchTimestamp(System.currentTimeMillis())
                        listener?.toggleScreenShare()

                        return@setOnTouchListener true
                    }
                }
                return@setOnTouchListener false
            }
        }
    }
}