package io.fullview.fullview_sdk.services

import android.app.Service
import android.content.Context
import android.content.Intent
import android.os.IBinder
import androidx.core.content.ContextCompat
import co.daily.CallClient
import co.daily.CallClientListener
import co.daily.model.AvailableDevices
import co.daily.model.CallJoinData
import co.daily.model.CallState
import co.daily.model.NetworkStats
import co.daily.model.Participant
import co.daily.model.ParticipantCounts
import co.daily.model.ParticipantLeftReason
import co.daily.model.RequestListener
import co.daily.model.RequestListenerWithData
import co.daily.settings.BitRate
import co.daily.settings.CameraInputSettings
import co.daily.settings.CameraInputSettingsUpdate
import co.daily.settings.CameraPublishingSettingsUpdate
import co.daily.settings.ClientSettingsUpdate
import co.daily.settings.Enable
import co.daily.settings.FacingMode
import co.daily.settings.FacingModeUpdate
import co.daily.settings.FrameRate
import co.daily.settings.InputSettings
import co.daily.settings.InputSettingsUpdate
import co.daily.settings.MicrophoneInputSettingsUpdate
import co.daily.settings.PublishingSettings
import co.daily.settings.PublishingSettingsUpdate
import co.daily.settings.Scale
import co.daily.settings.ScreenVideoInputSettingsUpdate
import co.daily.settings.StateBoolean
import co.daily.settings.VideoEncodingSettingsUpdate
import co.daily.settings.VideoEncodingsSettingsUpdate
import co.daily.settings.VideoMaxQualityUpdate
import co.daily.settings.VideoMediaTrackSettingsUpdate
import co.daily.settings.VideoSendSettingsUpdate
import io.fullview.fullview_sdk.DailyState
import io.fullview.fullview_sdk.DailyStateListener
import timber.log.Timber
import java.util.concurrent.CopyOnWriteArrayList

private const val ACTION_LEAVE = "action_leave"

class BackgroundCallService : Service() {

    private var callClient: CallClient? = null
    private val listeners = CopyOnWriteArrayList<DailyStateListener>()
    private var state: DailyState = DailyState.default()
    private var pendingScreenShareIntent: Intent? = null


    inner class Binder : android.os.Binder() {
        fun addListener(listener: DailyStateListener) {
            listeners.add(listener)
            listener.onStateChanged(state)
        }

        fun removeListener(listener: DailyStateListener) {
            listeners.remove(listener)
        }

        fun join(url: String, callback: ((Participant) -> Unit)? = null) {
            callClient?.apply {
                if(this.callState() != CallState.joined) {
                    callClient?.join(url= url, clientSettings = createClientSettings()) {
                        it.error?.apply {
                            Timber.e( "Got error while joining call: $msg")
                            listeners.forEach { it.onError("Failed to join call: $msg") }
                        }
                        it.success?.apply {
                            callClient?.setInputsEnabled(camera = false, microphone = false)
                            callback?.let { cb -> cb(participants().local) }
                        }
                    }
                }
            }
        }

        fun startScreenShare(mediaProjectionPermissionResultData: Intent) {
            pendingScreenShareIntent = mediaProjectionPermissionResultData
            updateServiceState { it.with(newScreenShareActive = true) }
            ContextCompat.startForegroundService(
                this@BackgroundCallService,
                Intent(this@BackgroundCallService, ScreenShareService::class.java)
            )
        }


        fun screenShareForegroundServiceStarted() {
            pendingScreenShareIntent?.apply {
                callClient?.startScreenShare(this) {
                }
                pendingScreenShareIntent = null
            }
        }

        fun stopScreenShare() {
            callClient?.stopScreenShare()
            updateServiceState { it.with(newScreenShareActive = false) }
        }

        fun toggleCamera(enabled: Boolean? = null) {
            callClient?.setInputsEnabled(camera = enabled ?: callClient?.inputs()?.camera?.isEnabled?.not())
        }

        fun toggleMicrophone(enabled: Boolean? = null) {
            callClient?.setInputsEnabled(microphone = enabled ?: callClient?.inputs()?.microphone?.isEnabled?.not())
        }

        fun toggleTorch(enabled: Boolean? = null) {
            val isEnabled = enabled ?: (callClient?.inputs()?.camera?.settings?.torch ?: false).not()
            callClient?.setCameraTorch(isEnabled)
        }

        fun toggleActiveCamera() {
            val currentFacingCamera = callClient?.inputs()?.camera?.settings?.facingMode

            val updateFacingMode = if (currentFacingCamera == FacingMode.user) {
                FacingModeUpdate.environment
            } else {
                FacingModeUpdate.user
            }

            callClient?.updateInputs(inputSettings = InputSettingsUpdate(camera = CameraInputSettingsUpdate(settings = VideoMediaTrackSettingsUpdate(facingMode = updateFacingMode))))
        }

        fun leave(listener: RequestListener) {
            callClient?.leave(listener)
            updateServiceState { it.with(newScreenShareActive = false) }
        }
    }


    override fun onStartCommand(intent: Intent?, flags: Int, startId: Int): Int {
        Timber.d( "onStartCommand(${intent?.action})")
        if (intent?.action == ACTION_LEAVE) {
            callClient?.leave()
        }
        return START_STICKY
    }

    override fun onCreate() {
        super.onCreate()

        try {
            callClient = CallClient(appContext = applicationContext).apply {
                addListener(callClientListener)

                updateServiceState { it.with(newAllParticipants = participants().all) }
                state.allParticipants.values.firstOrNull { it.info.isLocal }?.apply {
                    updateLocalVideo(this)
                }
            }
        } catch (e: Exception) {
            Timber.e(e, "Got exception while creating CallClient", e)
            listeners.forEach { it.onError("Failed to initialize call client") }
        }
    }

    override fun onBind(intent: Intent?): IBinder {
        return Binder()
    }

    override fun onUnbind(intent: Intent?): Boolean {
        stopSelf()
        return false
    }

    private val callClientListener: CallClientListener = object : CallClientListener {
        override fun onCallStateUpdated(state: CallState) {
            updateServiceState { it.with(newStatus = state) }
        }

        override fun onInputsUpdated(inputSettings: InputSettings) {
            updateServiceState {
                it.with(
                    newInputs = DailyState.StreamsState(
                        cameraEnabled = inputSettings.camera.isEnabled,
                        micEnabled = inputSettings.microphone.isEnabled,
                        screenVideoEnabled = inputSettings.screenVideo.isEnabled
                    )
                )
            }
        }

        override fun onPublishingUpdated(publishingSettings: PublishingSettings) {
            /*
            updateServiceState {
                it.with(
                    newPublishing = DailyState.StreamsState(
                        cameraEnabled = publishingSettings.camera.isPublishing,
                        micEnabled = publishingSettings.microphone.isPublishing,
                        screenVideoEnabled = true
                    )
                )
            }
             */
        }

        override fun onParticipantLeft(
            participant: Participant,
            reason: ParticipantLeftReason
        ) {
            onParticipantsChanged()
            updateVideoForParticipant(participant)
        }

        override fun onParticipantJoined(participant: Participant) {
            onParticipantsChanged()
            updateVideoForParticipant(participant)
        }

        override fun onParticipantUpdated(participant: Participant) {
            onParticipantsChanged()
            updateVideoForParticipant(participant)
        }

        override fun onError(message: String) {
            listeners.forEach { it.onError(message) }
        }

        override fun onAvailableDevicesUpdated(availableDevices: AvailableDevices) {
            updateServiceState {
                it.with(
                    newAvailableDevices = availableDevices,
                    newActiveAudioDevice = callClient?.audioDevice()
                )
            }
        }

        override fun onParticipantCountsUpdated(newParticipantCounts: ParticipantCounts) {}
        override fun onNetworkStatsUpdated(newNetworkStatistics: NetworkStats) {}
    }


    /**
     * State updates
     */

    private fun updateLocalVideo(participant: Participant) {
        updateServiceState { it.with(newLocalParticipantTrack = participant.media?.camera?.track) }
    }

    private fun updateServiceState(stateUpdate: (DailyState) -> DailyState) {
        val newState = stateUpdate(state)
        state = newState
        listeners.forEach { it.onStateChanged(newState) }
    }

    private fun onParticipantsChanged() {
        updateServiceState { it.with(newAllParticipants = callClient?.participants()?.all ?: emptyMap()) }
    }

    private fun updateVideoForParticipant(participant: Participant) {
        if (participant.info.isLocal) {
            updateLocalVideo(participant)
        }
    }

    private fun createClientSettings(): ClientSettingsUpdate {
        return ClientSettingsUpdate(
            publishingSettings = PublishingSettingsUpdate(
                camera = CameraPublishingSettingsUpdate(
                    sendSettings = VideoSendSettingsUpdate(
                        allowAdaptiveLayers = Enable(),
                        encodings = VideoEncodingsSettingsUpdate(
                            settings = mapOf(
                                VideoMaxQualityUpdate.low to
                                    VideoEncodingSettingsUpdate(
                                        maxBitrate = BitRate(80000),
                                        maxFramerate = FrameRate(10),
                                        scaleResolutionDownBy = Scale(4F)
                                    ),
                                VideoMaxQualityUpdate.medium to
                                    VideoEncodingSettingsUpdate(
                                        maxBitrate = BitRate(680000),
                                        maxFramerate = FrameRate(30),
                                        scaleResolutionDownBy = Scale(1F)
                                    )
                            )
                        )
                    )
                )
            )
        )
    }

    override fun onDestroy() {
        super.onDestroy()
        stopSelf()
    }

    companion object {
        fun leaveIntent(context: Context): Intent =
            Intent(context, BackgroundCallService::class.java).apply {
                action = ACTION_LEAVE
            }
    }
}