package io.fullview.fullview_sdk

import android.app.Activity
import android.app.Application
import android.app.Application.ActivityLifecycleCallbacks
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.FrameLayout
import androidx.annotation.Keep
import androidx.appcompat.app.AppCompatActivity
import androidx.fragment.app.FragmentActivity
import androidx.fragment.app.FragmentManager
import kotlinx.coroutines.flow.SharedFlow
import timber.log.Timber

@Suppress("Unused")
class Fullview : FullviewInterface {

    @Keep
    companion object {
        @JvmStatic
        private lateinit var INSTANCE: Fullview

        @JvmStatic
        fun getInstance(): Fullview {
            if (!::INSTANCE.isInitialized) {
                INSTANCE = Fullview()
            }

            return INSTANCE
        }
    }

    private var fullviewFragment : FullviewFragment? = null
    private var oldFragment : FullviewFragment? = null
    private val sessionRepository = KoinApp.getSessionRepository()
    private var registered = false
    private lateinit var fm : FragmentManager
    private var oldFm : FragmentManager? = null
    private val activities : MutableList<String> = mutableListOf()

    init {
        if(BuildConfig.DEBUG) {
            Timber.plant(Timber.DebugTree())
        }
    }

    override fun attach(application: Application, hostType: HostType?) {
        application.registerActivityLifecycleCallbacks(object : ActivityLifecycleCallbacks {
            override fun onActivityCreated(activity: Activity, savedInstanceState: Bundle?) {
                val appRootView = (activity.window.decorView.rootView as ViewGroup)
                if (activities.contains(activity.localClassName).not()) {
                    try {
                        val v = LayoutInflater.from(activity).inflate(R.layout.fragment_injector, activity.window.decorView.rootView as ViewGroup, false)
                        v.layoutParams = ViewGroup.LayoutParams(
                            ViewGroup.LayoutParams.MATCH_PARENT,
                            ViewGroup.LayoutParams.MATCH_PARENT
                        )
                        if (appRootView.findViewById<View>(R.id.fullview_fragment_injector) == null) {
                            appRootView.addView(v)
                        }

                        oldFm = if(this@Fullview::fm.isInitialized) { fm } else { null }
                        try {
                            fm = (activity as AppCompatActivity).supportFragmentManager
                        } catch (e: Exception) {
                            try {
                                fm = (activity as FragmentActivity).supportFragmentManager
                            } catch (e: Exception) {
                                Timber.e(e, "Error when getting fragment manager")
                            }
                        }

                        oldFragment = fullviewFragment
                        fullviewFragment = FullviewFragment.newInstance(hostType ?: HostType.NATIVE)
                        fm.beginTransaction().replace(R.id.fullview_fragment_injector, fullviewFragment!!).commit()
                    } catch (e: Exception) {
                        Timber.e(e, "Error when attaching Fullview to activity")
                    }
                    activities.add(activity.localClassName)
                }
            }

            override fun onActivityStarted(activity: Activity) {
            }

            override fun onActivityResumed(activity: Activity) {

            }

            override fun onActivityPaused(activity: Activity) {
            }

            override fun onActivityStopped(activity: Activity) {
            }

            override fun onActivitySaveInstanceState(activity: Activity, outState: Bundle) {}

            override fun onActivityDestroyed(activity: Activity) {
                activities.remove(activity.localClassName)
                oldFragment?.apply {
                    oldFm?.beginTransaction()?.detach(this)?.commitAllowingStateLoss()
                }
            }

        })
    }

    override fun attach(activity: Activity, hostType: HostType?) {
        val rootView = (activity.window.decorView.rootView as ViewGroup)

        // If the fragment inject already exists then don't add another - duplicates occurs in expo / metro with hot reload
        if (rootView.findViewById<FrameLayout>(R.id.fullview_fragment_injector) != null) {
            return
        }

        if (activities.contains(activity.localClassName).not()) {
            try {
                val v = LayoutInflater.from(activity).inflate(R.layout.fragment_injector, activity.window.decorView.rootView as ViewGroup, false)
                v.layoutParams = ViewGroup.LayoutParams(
                    ViewGroup.LayoutParams.MATCH_PARENT,
                    ViewGroup.LayoutParams.MATCH_PARENT
                )
                (activity.window.decorView.rootView as ViewGroup).addView(v)
                oldFm = if(this@Fullview::fm.isInitialized) { fm } else { null }

                try {
                    fm = (activity as AppCompatActivity).supportFragmentManager
                } catch (e: Exception) {
                    try {
                        fm = (activity as FragmentActivity).supportFragmentManager
                    } catch (e: Exception) {
                        Timber.e(e, "Error when getting fragment manager")
                    }
                }

                oldFragment = fullviewFragment
                fullviewFragment = FullviewFragment.newInstance(hostType ?: HostType.NATIVE)
                fm.beginTransaction().replace(R.id.fullview_fragment_injector, fullviewFragment!!).commit()
            } catch (e: Exception) {
                Timber.e(e, "Error when attaching Fullview to activity")
            }
            activities.add(activity.localClassName)
        }
        if (hostType != HostType.FLUTTER) {
            attach(activity.application)
        }
    }

    override fun register(
        organisationId: String,
        userId: String,
        deviceId: String,
        name: String,
        email: String,
        region: Region,
        layout: HubLayout
    ) {
        if(registered.not()) {
            fullviewFragment?.register(organisationId, userId, deviceId, name, email, region, layout)
            registered = true
        }
    }

    override fun logout() {
        if (registered) {
            fullviewFragment?.logout()
        }
        registered = false
    }
    override fun requestCoBrowse() {
        fullviewFragment?.requestCoBrowse()
    }

    override fun cancelCoBrowseRequest() {
        fullviewFragment?.cancelCoBrowseRequest()
    }

    override val positionInCoBrowseQueue: SharedFlow<Int>
        get() = sessionRepository.queue

    override val sessionState: SharedFlow<FullviewSessionState>
        get() = sessionRepository.sessionState
}