package io.fullview.fullview_sdk

import access.models.CoBrowseUpdateParticipant
import access.models.CustomerCoBrowseModel
import access.models.CustomerTabStateSourceEvent
import access.models.OrganisationSDKConfig
import android.annotation.SuppressLint
import android.graphics.Bitmap
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import co.daily.model.Participant
import collector.models.CoBrowseClickEvent
import collector.models.CoBrowseCursorEvent
import collector.models.CoBrowseHighlightEvent
import collector.models.CustomerEvent
import collector.models.EventType
import com.google.gson.Gson
import io.fullview.fullview_sdk.data.DisplayInfo
import io.fullview.fullview_sdk.data.WebRTCConfig
import io.fullview.fullview_sdk.helpers.ScreenshotHelpers
import io.fullview.fullview_sdk.helpers.SurfaceViewHelpers
import io.fullview.fullview_sdk.ui.ClickRipple
import io.reactivex.rxjava3.schedulers.Schedulers
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.distinctUntilChanged
import kotlinx.coroutines.flow.filterNotNull
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import network.models.RRWebMeta
import repository.AccessRepository
import repository.HubRepository
import repository.SessionRepository
import timber.log.Timber
import java.io.ByteArrayOutputStream
import java.util.Calendar
import java.util.TimeZone
import java.util.zip.Deflater
import kotlin.time.Duration
import kotlin.time.Duration.Companion.seconds


class FullviewViewModel : ViewModel() {

    private val accessRepository: AccessRepository = KoinApp.getAccessRepository()
    private val hubRepository: HubRepository = KoinApp.getHub()
    private val sessionRepository : SessionRepository = KoinApp.getSessionRepository()
    internal val sessionState = sessionRepository.sessionState
    internal val currentLayout = sessionRepository.currentLayout
    private val orgConfig = accessRepository.organisationConfig

    private val _flow: MutableStateFlow<Pair<Boolean, Bitmap>?> = MutableStateFlow(null)
    private val screenshotFlow: StateFlow<Pair<Boolean, Bitmap>?> = _flow.asStateFlow()

    private val _highlighterFlow: MutableStateFlow<Long> =
        MutableStateFlow(System.currentTimeMillis())
    val highlighterFlow: StateFlow<Long> = _highlighterFlow.asStateFlow()

    private val _highlighterPointer: MutableStateFlow<CoBrowseHighlightEvent?> = MutableStateFlow(null)
    val highlighterPointer = _highlighterPointer.asStateFlow()

    private val _cursorFlow: MutableStateFlow<CoBrowseCursorEvent?> = MutableStateFlow(null)
    val cursorFlow = _cursorFlow.asStateFlow()

    private val _clickFlow: MutableStateFlow<List<ClickRipple>> = MutableStateFlow(emptyList())
    val clickFlow = _clickFlow.asStateFlow()

    private val _dailyState = MutableStateFlow<DailyState?>(null)
    val dailyState = _dailyState.asStateFlow()

    private var snapshotTemplate : String? = null


    fun tickerFlow(period: Duration, initialDelay: Duration = Duration.ZERO) = flow {
        delay(initialDelay)
        while (true) {
            emit(Unit)
            delay(period)
        }
    }

    fun setTemplate(template: String) {
        snapshotTemplate = template
    }

    fun setDailyState(state: DailyState) {
        _dailyState.update { state }
    }

    private var organisationSDKConfig : OrganisationSDKConfig? = null
    lateinit var displayInfo: DisplayInfo

    init {
        viewModelScope.launch {
            orgConfig.replayCache.lastOrNull()?.apply {
                organisationSDKConfig = this
            }
        }

        viewModelScope.launch {
            orgConfig.collectLatest {
                organisationSDKConfig = it
            }
        }

        viewModelScope.launch {
            accessRepository
                .currentUser
                .filterNotNull()
                .distinctUntilChanged()
                .collectLatest {
                    hubRepository.createCustomerAccessConnection(region = it.region, it.accessToken)
                    setupAccessConnection()
                    setState(FullviewSessionState.IDLE)
                }
        }

        CoroutineScope(Dispatchers.Default).launch {
            screenshotFlow
                .filterNotNull()
                .distinctUntilChanged { old, new -> old.second.sameAs(new.second) && new.first.not() }
                .collectLatest {
                    if (hubRepository.isCollectorConnected) {
                        sendMetaData {
                            sendSnapshot(ScreenshotHelpers.getEncodedScreenshot(it.second))
                        }
                    }
                }
        }
    }

    fun declineCobrowse(id: String) {
        viewModelScope.launch {
            accessRepository
                .currentUser
                .filterNotNull()
                .collectLatest {
                    accessRepository.declineCobrowse(id, it)
                }
        }
    }

    fun acceptCobrowse(id: String) {
        viewModelScope.launch {
            accessRepository
                .currentUser
                .filterNotNull()
                .collectLatest {
                    accessRepository.acceptCoBrowse(id, it)
                }
        }
    }

    fun endCall() {
        setupCalled = false
        connected = false
        hubRepository.closeCollector()
        viewModelScope.launch {
            accessRepository
                .currentUser
                .filterNotNull()
                .collectLatest {
                    accessRepository.endCoBrowse(it)
                }
        }
    }


    fun requestCoBrowse() {
        viewModelScope.launch {
            accessRepository
                .currentUser
                .filterNotNull()
                .collectLatest {
                    accessRepository.requestCoBrowse(it)
                }
        }
    }

    fun cancelCoBrowseRequest() {
        viewModelScope.launch {
            accessRepository
                .currentUser
                .filterNotNull()
                .collectLatest {
                    accessRepository.cancelRequestCoBrowse(it)
                }
        }
    }

    fun consumeScreenshot(forceUpdate: Boolean? = false, bitmap: Bitmap) {
        if (hubRepository.isCollectorConnected) {
            _flow.update { Pair(forceUpdate ?: false, bitmap) }
        }
    }

    fun updateClicks(list: List<ClickRipple>) {
        _clickFlow.update { list }
    }

    fun updateHighlightTimestamp(timestamp: Long) {
        _highlighterFlow.update { timestamp }
    }

    fun register(
        organisationId: String,
        userId: String,
        deviceId: String,
        name: String,
        email: String,
        tabId: String,
        region: Region
    ) {
        accessRepository.register(organisationId, userId, deviceId, name, email, tabId, region)
    }

    fun setLayout(layout: HubLayout) {
        sessionRepository.setLayout(layout)
    }

    fun setState(state : FullviewSessionState) {
        sessionRepository.updateSessionState(state)
    }

    fun logout() {
        endCall()
        accessRepository.clearUser()
        hubRepository.closeAccess()
        setState(FullviewSessionState.ATTACHED)
    }

    fun clearScreenshot() {
        _flow.update { null }
    }

    private var setupCalled = false
    private var connected = false
    fun setupCobrowse(state: FullviewSessionState.CO_BROWSE_ACTIVE) {
        accessRepository.currentUser.replayCache.lastOrNull()?.apply {
            if (setupCalled.not() && hubRepository.isCollectorConnected.not()){
                setupCalled = true
                hubRepository.createCustomerCollectorConnection(region, accessToken, state.id)
            }
        }
        if (connected.not() && hubRepository.isCollectorConnected.not()) {
            connected = true
            setupCollectorHubConnection()
        }
    }

    /**
     * Private functions
     */

    private fun setupCollectorHubConnection() {
        hubRepository.collectorConnection.on("click", { clickEvent ->
            val (x, y) = SurfaceViewHelpers.getDensityCoordinates(
                clickEvent.x,
                clickEvent.y,
                displayInfo.density
            )
            val list = clickFlow.value.toMutableList()
                .plus(ClickRipple(x = x, y = y, radius = 0f, alpha = 1f, sendClickToUI = true))
            _clickFlow.update { list.toList() }
        }, CoBrowseClickEvent::class.java)

        hubRepository.collectorConnection.on("highlight", { pointer ->
            _highlighterPointer.update { pointer }
        }, CoBrowseHighlightEvent::class.java)

        hubRepository.collectorConnection.on("cursor", { cursor ->
            _cursorFlow.update { cursor }
        }, CoBrowseCursorEvent::class.java)

        hubRepository.collectorConnection.on("probe", {}, String::class.java)

        hubRepository.collectorConnection.on("agentConnected") {
            trySendCachedScreenshot()
        }
        hubRepository
            .collectorConnection
            .start()
            .doOnComplete {
                trySendCachedScreenshot()
            }
            .subscribe()
    }

    @SuppressLint("CheckResult")
    private fun setupAccessConnection() {
        viewModelScope.launch {
            tickerFlow(15.seconds)
                .collectLatest {
                    sendHeartbeat()
                }
        }

        hubRepository.accessConnection.on("coBrowseRequestPositionUpdated", {})

        hubRepository.accessConnection.on("coBrowseStateUpdated", { data ->
            val event = data.sourceEvent
            Timber.d("Event received: $data")

            when (event) {
                CustomerTabStateSourceEvent.TabSwitch -> {}
                CustomerTabStateSourceEvent.TabOpen -> {}
                CustomerTabStateSourceEvent.TabReconnect -> {}
                CustomerTabStateSourceEvent.TabRefresh -> {}
                CustomerTabStateSourceEvent.TabDisconnect -> {}
                CustomerTabStateSourceEvent.TabBecameActive -> {}
                CustomerTabStateSourceEvent.TabBecameInactive -> {}
                CustomerTabStateSourceEvent.TabPageChanged -> {}
                CustomerTabStateSourceEvent.AgentConnected -> {}
                CustomerTabStateSourceEvent.AgentDisconnected -> {}
                CustomerTabStateSourceEvent.CustomerConnected -> {}
                CustomerTabStateSourceEvent.CustomerDisconnected -> {}
                CustomerTabStateSourceEvent.ParticipantUpdated -> {}
                CustomerTabStateSourceEvent.CoBrowseInvitation -> {
                    val agent = data.invitation?.agentName ?: "agent"
                    val id = data.invitation?.coBrowseId ?: "-"

                    sessionRepository.updateSessionState(
                        FullviewSessionState.CO_BROWSE_INVITATION(id, agent, organisationSDKConfig?.coBrowse?.privacyPolicyURL ?: "", organisationSDKConfig?.coBrowse?.privacyPolicyCustomText)
                    )
                }
                CustomerTabStateSourceEvent.CoBrowseInvitationDeclined -> {
                    setState(FullviewSessionState.IDLE)
                }
                CustomerTabStateSourceEvent.CoBrowseInvitationAccepted -> {}
                CustomerTabStateSourceEvent.CoBrowseEnded -> {
                    sessionRepository.updateSessionState(
                        FullviewSessionState.CO_BROWSE_ENDED
                    )
                }
                CustomerTabStateSourceEvent.SessionStopped -> {}
                CustomerTabStateSourceEvent.MeetingInvitationReceived -> {}
                CustomerTabStateSourceEvent.MeetingInvitationDeclined -> {}
                CustomerTabStateSourceEvent.MeetingInvitationAccepted -> {}
                CustomerTabStateSourceEvent.MeetingLeft -> {}
                CustomerTabStateSourceEvent.MeetingUpdated -> {}
                CustomerTabStateSourceEvent.SupportRequestPositionUpdated -> {}
            }


            if (data.ongoingCoBrowseId != null) {
                data.mirror?.apply {
                    val config = organisationSDKConfig?.coBrowse?.let {
                        WebRTCConfig(
                            it.enableVideo ?: false,
                            it.enableAudio ?: false,
                            it.startWithCameraOff ?: true,
                            it.startWithMicrophoneOff ?: true
                        )
                    } ?: WebRTCConfig()
                    sessionRepository.updateSessionState(
                        FullviewSessionState.CO_BROWSE_ACTIVE(
                            data.ongoingCoBrowseId!!,
                            agentName,
                            webRTCURL,
                            config
                        )
                    )
                }
            }

        }, CustomerCoBrowseModel::class.java)


        hubRepository.accessConnection.on("supportRequestCancelledByAgent") {
            sessionRepository.updateQueue(-1)
            sessionRepository.updateSessionState(FullviewSessionState.IDLE)
        }

        hubRepository.accessConnection.on("supportRequestPositionUpdated", { position ->
            sessionRepository.updateQueue(position)
            sessionRepository.updateSessionState(FullviewSessionState.CO_BROWSE_REQUESTED)
        }, Int::class.java)

        hubRepository.accessConnection.start()
            .doOnComplete {
                sendHeartbeat()
            }
            .subscribe({}, {
                Timber.e(it, "Error when starting")
            })
    }

    private fun getMetaData(height: Int, width: Int, href: String): String {
        val meta = RRWebMeta(
            4,
            RRWebMeta.RRWebData(
                href,
                width,
                height
            ),
            Calendar.getInstance(TimeZone.getTimeZone("UTC")).timeInMillis
        )
        val gson = Gson()
        return gson.toJson(meta)
    }

    private fun sendMetaData(callback: () -> Unit) {
        val metaData = getMetaData(
            displayInfo.correctedHeight.toInt(),
            displayInfo.correctedWidth.toInt(),
            ""
        )

        val deflater = Deflater(Deflater.BEST_COMPRESSION)
        deflater.setInput(metaData.toByteArray(Charsets.ISO_8859_1))
        deflater.finish()

        val outputStream = ByteArrayOutputStream()
        val buffer = ByteArray(1024)

        while (!deflater.finished()) {
            val compressedSize = deflater.deflate(buffer)
            outputStream.write(buffer, 0, compressedSize)
        }
        if (hubRepository.isCollectorConnected) {
            hubRepository.collectorConnection.invoke(
                "event", CustomerEvent(
                    type = EventType.RRWebMeta,
                    payload = outputStream.toByteArray().toString(Charsets.ISO_8859_1),
                    timestamp = Calendar.getInstance(TimeZone.getTimeZone("UTC")).timeInMillis,
                )
            ).blockingAwait()
        }
        callback()
    }

    @SuppressLint("CheckResult")
    private fun sendSnapshot(imageData: String) {
        viewModelScope.launch(Dispatchers.IO) {
            if (snapshotTemplate == null) {
                return@launch
            }

            val timestamp = Calendar.getInstance(TimeZone.getTimeZone("UTC")).timeInMillis
            val payload = snapshotTemplate!!
                .replace("BASE64_JPG_REPLACE", imageData)
                .replace("TIMESTAMP_REPLACE", timestamp.toString())
                .replace(Regex("(\"(?=$timestamp)|(?<=$timestamp)\")"), "")
                .replace("\n", "")


            val deflater = Deflater(Deflater.BEST_COMPRESSION)
            deflater.setInput(payload.toByteArray(Charsets.ISO_8859_1))
            deflater.finish()

            val outputStream = ByteArrayOutputStream()
            val buffer = ByteArray(1024)

            while (!deflater.finished()) {
                val compressedSize = deflater.deflate(buffer)
                outputStream.write(buffer, 0, compressedSize)
            }

            if (hubRepository.isCollectorConnected) {
                hubRepository.collectorConnection.invoke(
                    "event",
                    CustomerEvent(
                        type = EventType.RRWebFullSnapshot,
                        payload = outputStream.toByteArray().toString(Charsets.ISO_8859_1),
                        timestamp = Calendar.getInstance(TimeZone.getTimeZone("UTC")).timeInMillis,
                    )
                )
            }
        }
    }

    fun updateLocalParticipant(localParticipant: Participant) {
        updateParticipant(
            CoBrowseUpdateParticipant(
                webRTCId = localParticipant.id.uuid.toString(),
                webRTCStatus = "ongoing"
            )
        )
    }

    fun updateParticipant(participant: CoBrowseUpdateParticipant) {
        viewModelScope.launch {
            accessRepository
                .currentUser
                .filterNotNull()
                .collectLatest {
                    accessRepository.updateParticipant(participant, it)
                }
        }
    }

    @SuppressLint("CheckResult")
    private fun sendHeartbeat() {
        if (hubRepository.isAccessConnected) {
            hubRepository.accessConnection.invoke("heartbeat",   Calendar.getInstance(TimeZone.getTimeZone("UTC")).timeInMillis).subscribeOn(Schedulers.io())
                .subscribe({}, {
                    Timber.e(it, "Error when sending recurring heartbeat")
                })
        }
    }

    private fun trySendCachedScreenshot() {
        screenshotFlow.replayCache.lastOrNull()?.apply {
            sendMetaData {
                sendSnapshot(ScreenshotHelpers.getEncodedScreenshot(second))
            }
        }
    }
}