package io.fullview.fullview_sdk.helpers

import android.app.Activity
import android.graphics.Bitmap
import android.graphics.Canvas
import android.graphics.Rect
import android.os.Handler
import android.os.Looper
import android.util.Base64
import android.view.PixelCopy
import android.view.SurfaceView
import android.view.ViewGroup
import android.view.Window
import androidx.core.view.children
import io.fullview.fullview_sdk.HostType
import io.fullview.fullview_sdk.R
import timber.log.Timber
import java.io.ByteArrayOutputStream

object ScreenshotHelpers {
    private val FORMAT = Bitmap.CompressFormat.WEBP
    private const val QUALITY = 5

    fun takeScreenshot(hostType: HostType, activity: Activity, bitmapBuffer: Bitmap, surfaceView: SurfaceView?, onSuccess: (Bitmap) -> Unit) {
        when(hostType) {
            HostType.FLUTTER -> {
                surfaceView?.apply {
                    takeScreenshotFlutter(surfaceView, bitmapBuffer, onSuccess)
                }
            }
            else -> takeScreenshot(activity, onSuccess)
        }
    }


    /**
     * Screenshot helpers
     */
    private fun takeScreenshot(activity: Activity, onSuccess: (Bitmap) -> Unit) {
            try {
                val v1 = (activity.window.decorView.rootView as ViewGroup).children.first {  it.id != android.R.id.statusBarBackground && it.id != android.R.id.navigationBarBackground && it.id != R.id.fullview_fragment_injector }
                v1.isDrawingCacheEnabled = true
                val bitmap = Bitmap.createBitmap(
                    v1.measuredWidth,
                    v1.measuredHeight,
                    Bitmap.Config.ARGB_8888
                )
                val canvas = Canvas(bitmap)
                v1.draw(canvas)
                v1.isDrawingCacheEnabled = false
                onSuccess(bitmap)
            } catch (e: Exception) {
                Timber.e(e, "Error when saving bitmap")
            }
    }

    /**
     * Screenshot helpers
     */
    private fun takeScreenshotFlutter(surfaceView: SurfaceView, buffer: Bitmap, onSuccess: (Bitmap) -> Unit) {
        try {
            val bitmap = buffer.copy(Bitmap.Config.ARGB_8888, true)
            PixelCopy.request(surfaceView, bitmap, { onSuccess(bitmap)}, Handler(Looper.getMainLooper()))
        } catch (e: Exception) {
            Timber.e(e, "Error when saving bitmap for flutter")
        }
    }

    /**
     * Used to send video to cobrowse
     */
    fun takeScreenshotOfSurface(bitmapBuffer: Bitmap, surfaceView: SurfaceView, onSuccess: (Bitmap) -> Unit) {
        try {
            val bitmap = bitmapBuffer.copy(Bitmap.Config.ARGB_8888, true)
            val location = IntArray(2)
            PixelCopy.request(
                surfaceView,
                Rect(location[0], location[1], location[0] + surfaceView.width, location[1] + surfaceView.height),
                bitmap,
                {
                    onSuccess(bitmap)
                },
                Handler(Looper.getMainLooper())
            )
        } catch (e: Exception) {
            Timber.e(e, "Error when saving bitmap")
        }
    }

    /**
     * Used to send video to cobrowse
     */
    fun takeScreenshot(bitmapBuffer: Bitmap, window: Window, onSuccess: (Bitmap) -> Unit) {
        try {
            val bitmap = bitmapBuffer.copy(Bitmap.Config.ARGB_8888, true)
            val location = IntArray(2)
            PixelCopy.request(
                window,
                Rect(location[0], location[1], location[0] + window.decorView.width, location[1] + window.decorView.height),
                bitmap,
                {
                    onSuccess(bitmap)
                },
                Handler(Looper.getMainLooper())
            )
        } catch (e: Exception) {
            Timber.e(e, "Error when saving bitmap")
        }
    }

    fun getEncodedScreenshot(bitmap: Bitmap): String {
        val byteArrayOutputStream = ByteArrayOutputStream()
        bitmap.compress(FORMAT, QUALITY, byteArrayOutputStream)
        val byteArray = byteArrayOutputStream.toByteArray()

        return Base64.encodeToString(byteArray, Base64.DEFAULT)
    }
}