package io.fullview.fullview_sdk.services

import android.content.Context
import android.content.res.Resources
import android.graphics.Color
import android.graphics.Paint
import android.graphics.Path
import android.graphics.PixelFormat
import android.graphics.PorterDuff
import android.view.Gravity
import android.view.LayoutInflater
import android.view.SurfaceView
import android.view.View
import android.view.WindowManager
import collector.models.CoBrowseHighlightEvent
import io.fullview.fullview_sdk.R
import io.fullview.fullview_sdk.helpers.SurfaceViewHelpers
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.FlowPreview
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.debounce
import kotlinx.coroutines.flow.filterNotNull
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import timber.log.Timber
import kotlin.time.Duration.Companion.seconds


@OptIn(FlowPreview::class)
class DrawView(
    private val context: Context
) {

    private var windowManager: WindowManager? = null
    private var overlayView: View? = null
    private lateinit var surface: SurfaceView
    private var params: WindowManager.LayoutParams? = null

    private val _highlighterPointer: MutableStateFlow<CoBrowseHighlightEvent?> = MutableStateFlow(null)
    private val highlighterPointer = _highlighterPointer.asStateFlow()

    private val _highlighterTimestamp: MutableStateFlow<Long> =
        MutableStateFlow(System.currentTimeMillis())
    private val highlighterTimestamp: StateFlow<Long> = _highlighterTimestamp.asStateFlow()
    private var drawPaint: Paint = Paint()
    private var lastDrawn: Long? = null

    init {
        showOverlay()
        drawPaint.apply {
            isAntiAlias = true
            isDither = true
            setColor(context.resources.getColor(R.color.highlighter))
            alpha = 64
            style = Paint.Style.STROKE
            strokeJoin = Paint.Join.ROUND
            strokeCap = Paint.Cap.ROUND
            strokeWidth = 20f
        }

        val displayMetrics = Resources.getSystem().displayMetrics

        CoroutineScope(Dispatchers.Default).launch {
            highlighterPointer
                .filterNotNull()
                .collectLatest { pointer ->
                    val (x, y) = SurfaceViewHelpers.getDensityCoordinates(
                        pointer.x,
                        pointer.y,
                        displayMetrics.density
                    )

                    if (lastDrawn == null) {
                        highlightPath.moveTo(x, y)
                    }

                    // If last line was 300ms ago then move to the new spot, to prevent a line being drawn between the points
                    if (System.currentTimeMillis() - (lastDrawn
                            ?: System.currentTimeMillis()) > 300
                    ) {
                        highlightPath.moveTo(x, y)
                    }
                    val shouldClear = System.currentTimeMillis() - (lastDrawn
                        ?: System.currentTimeMillis()) > 3000
                    highlightPath.lineTo(x, y)
                    lastDrawn = System.currentTimeMillis()

                    _highlighterTimestamp.update { System.currentTimeMillis() }

                    val canvas = surface.holder.lockCanvas()
                    canvas?.apply {
                        if (shouldClear) {
                            drawColor(Color.TRANSPARENT, PorterDuff.Mode.CLEAR)
                        }
                        drawPath(highlightPath, drawPaint)
                        surface.holder.unlockCanvasAndPost(this)
                    }
                }
        }


        CoroutineScope(Dispatchers.Default).launch {
            highlighterTimestamp
                .debounce(3.seconds)
                .collectLatest {
                    highlightPath.reset()
                    try {
                        SurfaceViewHelpers.clear(surface)
                    } catch (e: Exception) {
                        Timber.e(e, "Error when clearing highlight surface")
                    }
                }
        }
    }

    fun updatePointer(highlight: CoBrowseHighlightEvent) {
        _highlighterPointer.update { highlight }
    }


    private var highlightPath = Path()

    private fun showOverlay() {
        val LAYOUT_FLAG =
            WindowManager.LayoutParams.TYPE_APPLICATION_OVERLAY
        params = WindowManager.LayoutParams(
            WindowManager.LayoutParams.MATCH_PARENT,
            WindowManager.LayoutParams.MATCH_PARENT,
            LAYOUT_FLAG,
            WindowManager.LayoutParams.FLAG_NOT_TOUCHABLE or WindowManager.LayoutParams.FLAG_LAYOUT_IN_SCREEN or WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE,
            PixelFormat.TRANSLUCENT
        );

        val layoutInflater =
            context.getSystemService(Context.LAYOUT_INFLATER_SERVICE) as LayoutInflater
        overlayView = layoutInflater.inflate(R.layout.layout_draw_view, null)
        surface = overlayView!!.findViewById(R.id.surface)
        surface.setZOrderOnTop(true)
        surface.holder.setFormat(PixelFormat.TRANSPARENT)
        params?.gravity = Gravity.START or Gravity.TOP
        windowManager = context.getSystemService(Context.WINDOW_SERVICE) as WindowManager
    }

    fun open() {
        try {
            if (overlayView?.windowToken == null) {
                windowManager?.addView(overlayView, params)
            }
        } catch (e: Exception) {
            Timber.e(e, "Error when opening window")
        }
    }

    fun removeOverLay() {
        try {
            overlayView?.let {
                windowManager?.removeView(it)
                overlayView = null
                params = null
            }
        } catch (e: Exception) {
            Timber.e(e, "Error when removing overlay")
        }
    }
}
