package io.fullview.fullview_sdk

import access.models.CoBrowseUpdateParticipant
import android.Manifest
import android.app.Activity
import android.content.ComponentName
import android.content.Context
import android.content.Intent
import android.content.ServiceConnection
import android.content.pm.PackageManager
import android.graphics.Bitmap
import android.graphics.Color
import android.graphics.Paint
import android.graphics.Path
import android.graphics.PixelFormat
import android.graphics.PorterDuff
import android.media.projection.MediaProjectionManager
import android.net.Uri
import android.os.Build
import android.os.Bundle
import android.os.IBinder
import android.provider.Settings
import android.util.DisplayMetrics
import android.util.Log
import android.view.LayoutInflater
import android.view.SurfaceView
import android.view.View
import android.view.ViewGroup
import android.widget.Toast
import androidx.activity.result.contract.ActivityResultContracts
import androidx.activity.result.contract.ActivityResultContracts.RequestMultiplePermissions
import androidx.core.app.ActivityOptionsCompat
import androidx.core.content.ContextCompat
import androidx.core.content.ContextCompat.getSystemService
import androidx.core.graphics.drawable.toBitmap
import androidx.core.view.children
import androidx.core.view.isVisible
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentManager
import androidx.fragment.app.viewModels
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import co.daily.model.CallState
import io.fullview.fullview_sdk.FullviewSessionState.IDLE.isActive
import io.fullview.fullview_sdk.camera.CameraFragment
import io.fullview.fullview_sdk.data.DisplayInfo
import io.fullview.fullview_sdk.fullscreen.FullscreenFragment
import io.fullview.fullview_sdk.helpers.ScreenshotHelpers
import io.fullview.fullview_sdk.helpers.SurfaceViewHelpers
import io.fullview.fullview_sdk.helpers.ViewHelpers
import io.fullview.fullview_sdk.helpers.throttleLatest
import io.fullview.fullview_sdk.incoming.IncomingCallFragment
import io.fullview.fullview_sdk.ongoing.OngoingCallFragment
import io.fullview.fullview_sdk.ongoing.OngoingCallInterface
import io.fullview.fullview_sdk.services.ActiveCallService
import io.fullview.fullview_sdk.services.BackgroundCallService
import io.fullview.fullview_sdk.ui.ClickRipple
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.FlowPreview
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.debounce
import kotlinx.coroutines.flow.distinctUntilChanged
import kotlinx.coroutines.flow.filterNotNull
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import timber.log.Timber
import kotlin.time.Duration.Companion.milliseconds
import kotlin.time.Duration.Companion.seconds
import kotlin.uuid.ExperimentalUuidApi
import kotlin.uuid.Uuid


@OptIn(FlowPreview::class)
class FullviewFragment : Fragment(), IncomingCallFragment.IncomingCallInterface,
    OngoingCallInterface, CameraFragment.CameraFragmentInterface, DailyStateListener, FullviewFragmentCoreInterface {

    companion object {

        const val ARG_HOST = "fullview.fragment.host"

        fun newInstance(hostType: HostType) = FullviewFragment().apply {
            arguments = Bundle().apply {
                putParcelable(ARG_HOST, hostType)
            }
        }
    }

    private val viewModel by viewModels<FullviewViewModel>()

    // Views
    lateinit var root : View

    private lateinit var incomingCallFragment : IncomingCallFragment
    private lateinit var ongoingCallFragment: OngoingCallFragment
    private lateinit var fullscreenFragment: FullscreenFragment
    private var cameraFragment: CameraFragment? = null
    private lateinit var fragmentManager: FragmentManager
    private lateinit var cursorAgentSurfaceView : SurfaceView
    private lateinit var highlightAgentSurfaceView: SurfaceView
    private lateinit var cursorBitmap : Bitmap
    private lateinit var bitmapBuffer : Bitmap
    private var windowInsetsTop: Int = 0
    private var lastDrawn : Long? = null
    private lateinit var hostType: HostType
    private var flutterContentView : SurfaceView? = null

    private var callService : BackgroundCallService.Binder? = null
    var triggeredForegroundService = false


    private lateinit var mediaProjectionManager: MediaProjectionManager
    private val requestMediaProjection =
        registerForActivityResult(ActivityResultContracts.StartActivityForResult()) { result ->
            if (result.resultCode == Activity.RESULT_OK) {
                // Handle the permission granted and media projection here
                val data: Intent? = result.data
                if (data != null) {
                    callService?.startScreenShare(data)
                }
            } else {
                Log.e(this::class.java.simpleName, "Denied permission to start media projection.")
            }
        }

    private val requestPermissionLauncher =
        registerForActivityResult(RequestMultiplePermissions()) { result ->
            if (result.values.any { !it }) {
                Toast.makeText(requireContext(), "Required permissions not granted", Toast.LENGTH_LONG).show()
            } else {
                // permission is granted, we can initialize
                initialize()
            }
        }

    private val serviceConnection = object : ServiceConnection {
        override fun onServiceConnected(name: ComponentName?, service: IBinder?) {
            if(name?.shortClassName?.contains(BackgroundCallService::class.simpleName.toString()) == true) {
                callService = service!! as BackgroundCallService.Binder
                callService?.addListener(this@FullviewFragment)
            }
        }

        override fun onServiceDisconnected(name: ComponentName?) {
            if(name?.shortClassName?.contains(BackgroundCallService::class.simpleName.toString()) == true) {
                callService = null
            }
        }
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        fragmentManager = (requireActivity()).supportFragmentManager

        arguments?.let {
            hostType = it.getParcelable(ARG_HOST) ?: HostType.NATIVE
        }

        // initializing the mediaProjectionManager that we are going to use later to ask for screen share
        mediaProjectionManager = getSystemService(requireContext(), MediaProjectionManager::class.java)!!

        if(viewModel.sessionState.replayCache.isEmpty()) {
            viewModel.setState(FullviewSessionState.ATTACHED)
        }
    }

    private val paint = Paint()
    val drawPaint = Paint()
    val ripplePaint = Paint().apply {
        strokeWidth = 4f
    }
    private var highlightPath = Path()
    private var cameraActive = false

    override fun onCreateView(
        inflater: LayoutInflater, container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        root = inflater.inflate(R.layout.fragment_fullview, container, false)

        val payloadStream = resources.openRawResource(R.raw.rrwebsnapshot_template)
        val payload = payloadStream.bufferedReader(Charsets.ISO_8859_1).use { it.readText() }

        viewModel.setTemplate(payload)

        val statusBarHeightId = resources.getIdentifier("status_bar_height", "dimen", "android")
        windowInsetsTop = resources.getDimensionPixelSize(statusBarHeightId)

        cursorAgentSurfaceView = root.findViewById(R.id.agent_cursor_surface)
        cursorAgentSurfaceView.setZOrderOnTop(true)
        cursorAgentSurfaceView.holder.setFormat(PixelFormat.TRANSPARENT)

        highlightAgentSurfaceView = root.findViewById(R.id.agent_highlight_surface)
        highlightAgentSurfaceView.setZOrderOnTop(true)
        highlightAgentSurfaceView.holder.setFormat(PixelFormat.TRANSPARENT)

        ripplePaint.setColor(resources.getColor(R.color.highlighter))
        cursorBitmap = resources.getDrawable(R.drawable.cursor).toBitmap()

        val displayMetrics = DisplayMetrics()
        requireActivity().windowManager.defaultDisplay.getMetrics(displayMetrics)
        viewModel.displayInfo = DisplayInfo(displayMetrics.heightPixels, displayMetrics.widthPixels, displayMetrics.density)
        bitmapBuffer = Bitmap.createBitmap(displayMetrics.widthPixels, displayMetrics.heightPixels, Bitmap.Config.ARGB_8888)

        drawPaint.apply {
            isAntiAlias = true
            isDither = true
            setColor(resources.getColor(R.color.highlighter))
            alpha = 64
            style = Paint.Style.STROKE
            strokeJoin = Paint.Join.ROUND
            strokeCap = Paint.Cap.ROUND
            strokeWidth = 20f
        }

        /**
         * Draw annotation in app (see DrawView.kt for annotation outside app)
         */
        viewLifecycleOwner.lifecycleScope.launch(Dispatchers.Default) {
            repeatOnLifecycle(Lifecycle.State.RESUMED) {
                viewModel.highlighterPointer
                    .filterNotNull()
                    .collectLatest { pointer ->
                        if (this@FullviewFragment.isDetached.not() &&
                            viewModel.currentLayout.replayCache.lastOrNull() == HubLayout.DEFAULT &&
                            viewModel.sessionState.replayCache.lastOrNull().isActive() &&
                            viewModel.dailyState.replayCache.lastOrNull()?.screenShareActive == false) {

                            val offsetY = if (cameraActive) {
                                0f
                            } else {
                                (windowInsetsTop/viewModel.displayInfo.density)
                            }

                            val (x, y) = SurfaceViewHelpers.getDensityCoordinates((pointer.x-(drawPaint.strokeWidth/2)).toInt(), (pointer.y-(drawPaint.strokeWidth/2)-offsetY).toInt(), viewModel.displayInfo.density)

                            if(lastDrawn == null) {
                                highlightPath.moveTo(x, y)
                            }

                            // If last line was 300ms ago then move to the new spot, to prevent a line being drawn between the points
                            if(System.currentTimeMillis() - (lastDrawn?:System.currentTimeMillis()) > 300) {
                                highlightPath.moveTo(x, y)
                            }
                            val shouldClear = System.currentTimeMillis() - (lastDrawn?:System.currentTimeMillis()) > 3000
                            highlightPath.lineTo(x, y)
                            lastDrawn = System.currentTimeMillis()

                            lifecycleScope.launch {
                                viewModel.updateHighlightTimestamp(System.currentTimeMillis())
                            }

                            val canvas = highlightAgentSurfaceView.holder.lockCanvas()
                            canvas?.apply {
                                if (shouldClear) {
                                    drawColor(Color.TRANSPARENT, PorterDuff.Mode.CLEAR)
                                }
                                drawPath(highlightPath, drawPaint)
                                highlightAgentSurfaceView.holder.unlockCanvasAndPost(this)
                            }
                        }
                    }
            }
        }


        /**
         * Draw cursor
         */
        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.RESUMED) {
                viewModel.cursorFlow
                    .throttleLatest(10)
                    .filterNotNull()
                    .collectLatest { cursor ->
                        if (this@FullviewFragment.isDetached.not() &&
                            viewModel.currentLayout.replayCache.lastOrNull() == HubLayout.DEFAULT &&
                            viewModel.sessionState.replayCache.lastOrNull().isActive()) {
                            cursorBitmap.apply {
                                try {

                                    val offsetY = if (cameraActive) {
                                        0f
                                    } else {
                                        (windowInsetsTop/viewModel.displayInfo.density)
                                    }

                                    val canvas = cursorAgentSurfaceView.holder.lockCanvas()
                                    canvas?.let {
                                        canvas.drawColor(Color.TRANSPARENT, PorterDuff.Mode.CLEAR)
                                        val (x, y) = SurfaceViewHelpers.getDensityCoordinates(cursor.x, cursor.y-offsetY.toInt(), viewModel.displayInfo.density)
                                        it.drawBitmap(
                                            this,
                                            x,
                                            y,
                                            paint
                                        )
                                        cursorAgentSurfaceView.holder.unlockCanvasAndPost(canvas)
                                    }
                                } catch (e: Exception) {
                                    Log.e(this::class.java.simpleName, "Error when drawing cursor", e)
                                }
                            }
                        }
                    }
            }
        }

        /**
         * Clear annotation drawing after X time
         */
        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.RESUMED) {
                viewModel.highlighterFlow
                    .debounce(3.seconds)
                    .collectLatest {
                        if (this@FullviewFragment.isDetached.not() &&
                            viewModel.currentLayout.replayCache.lastOrNull() == HubLayout.DEFAULT &&
                            viewModel.sessionState.replayCache.lastOrNull().isActive()) {
                            highlightPath.reset()
                            try {
                                if (highlightAgentSurfaceView.isAttachedToWindow && highlightAgentSurfaceView.isVisible) {
                                    SurfaceViewHelpers.clear(highlightAgentSurfaceView)
                                }
                            } catch (e: Exception) {
                                Log.e(this::class.java.simpleName,  "Error when clearing highlight surface", e)
                            }
                        }
                    }
            }
        }

        /**
         * Click effect
         */
        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.RESUMED) {
                viewModel
                    .clickFlow
                    .collectLatest {
                        if(it.isEmpty().not() &&
                            this@FullviewFragment.isDetached.not() &&
                            viewModel.currentLayout.replayCache.lastOrNull() == HubLayout.DEFAULT &&
                            viewModel.sessionState.replayCache.lastOrNull().isActive()) {
                                val canvas = highlightAgentSurfaceView.holder.lockCanvas()
                                canvas?.apply {
                                    canvas.drawColor(Color.TRANSPARENT, PorterDuff.Mode.CLEAR)
                                    val newList = mutableListOf<ClickRipple>()
                                    it.forEach {
                                        if (it.sendClickToUI) {
                                            ViewHelpers.sendClick(requireActivity().window.decorView.rootView, it.x, it.y)
                                        }
                                        val newRadius = it.radius + 10f
                                        val newAlpha = it.alpha - 0.05f
                                        if (newRadius <= ClickRipple.MAX_RADIUS) {
                                            canvas.drawCircle(it.x, it.y-windowInsetsTop, newRadius, ripplePaint.apply {
                                                this.alpha = (newAlpha*255).toInt()
                                            })
                                            newList.add(ClickRipple(it.x, it.y, newRadius, newAlpha, it.id))
                                        }
                                    }
                                    highlightAgentSurfaceView.holder.unlockCanvasAndPost(canvas)
                                    viewModel.updateClicks(newList)
                                }
                        }
                    }
            }
        }

        /**
         * Screenshot loop
         */
        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.RESUMED) {
                while (this@FullviewFragment.isResumed) {
                    if (cameraActive.not()) {
                        ScreenshotHelpers.takeScreenshot(hostType, requireActivity(), bitmapBuffer, flutterContentView) { bitmap ->
                            viewModel.consumeScreenshot(bitmap= bitmap)
                        }
                    }
                    delay(100.milliseconds)
                }
            }
        }


        /**
         * Handles pure state related changes
         */
        viewLifecycleOwner.lifecycleScope.launch(Dispatchers.Default) {
            repeatOnLifecycle(Lifecycle.State.RESUMED) {
                viewModel
                    .sessionState
                    .distinctUntilChanged()
                    .collectLatest { handleState(it) }
            }
        }

        /**
         * Handles layout changes + state bound
         */
        viewLifecycleOwner.lifecycleScope.launch(Dispatchers.Default) {
            repeatOnLifecycle(Lifecycle.State.RESUMED) {
                viewModel
                    .currentLayout
                    .distinctUntilChanged()
                    .combine(
                        viewModel
                            .sessionState
                            .distinctUntilChanged()
                    ) { layout, state -> Pair(layout, state) }
                        .distinctUntilChanged { old, new -> old.first != new.first && old.second is FullviewSessionState.CO_BROWSE_ACTIVE && new.second is FullviewSessionState.CO_BROWSE_ACTIVE }
                        .collectLatest {
                            handleLayoutState(it.first, it.second)
                        }
            }
        }


        /**
         * Handle daily state changes
         */
        viewLifecycleOwner.lifecycleScope.launch(Dispatchers.Default) {
            repeatOnLifecycle(Lifecycle.State.RESUMED) {
                viewModel
                    .dailyState
                    .filterNotNull()
                    .collectLatest {
                    when (it.status) {
                        CallState.initialized, CallState.left -> {
                            triggeredForegroundService = false
                            callService?.toggleCamera(false)
                            callService?.toggleMicrophone(false)
                        }
                        CallState.joining -> {}
                        CallState.leaving -> {}
                        CallState.joined -> {
                            viewModel.updateParticipant(
                                CoBrowseUpdateParticipant(
                                    audioEnabled = it.inputs.micEnabled,
                                    videoEnabled = it.inputs.cameraEnabled,
                                    screenShareEnabled = it.screenShareActive,
                                )
                            )
                            if (!triggeredForegroundService) {

                                // Start the foreground service to keep the call alive
                                requireActivity().startForegroundService(Intent(requireContext(), ActiveCallService::class.java))
                                triggeredForegroundService = true
                            }
                        }
                    }
                }
            }
        }


        /**
         * Find the SurfaceView that renders content in flutter applications
         */
        if(hostType == HostType.FLUTTER) {
            val contentView = requireActivity().findViewById<ViewGroup>(android.R.id.content)
            val children = mutableListOf<View>()
            children.addAll(contentView.children)
            while (children.isNotEmpty() && flutterContentView == null) {
                val current = children.removeFirst()
                if (current is SurfaceView && current::class.java.name.contains("FlutterSurfaceView")) {
                    flutterContentView = current
                } else if (current is ViewGroup) {
                    children.addAll(current.children)
                }
            }
        }

        return root
    }

    override fun declineCall(invitationId: String) {
        viewModel.declineCobrowse(invitationId)
        incomingCallFragment.setListener(null)
        fragmentManager.beginTransaction().remove(incomingCallFragment).commitAllowingStateLoss()
    }

    override fun acceptCall(invitationId: String) {
        viewModel.acceptCobrowse(invitationId)
        incomingCallFragment.setListener(null)
        fragmentManager.beginTransaction().remove(incomingCallFragment).commitAllowingStateLoss()
    }

    override fun attachService(className: String) {
        when(className) {
            OngoingCallFragment.TAG -> callService?.addListener(ongoingCallFragment)
            CameraFragment.TAG -> cameraFragment?.apply {
                callService?.addListener(this)
            }
            FullscreenFragment.TAG -> callService?.addListener(fullscreenFragment)
        }
    }

    override fun toggleActiveCamera() {
        callService?.toggleActiveCamera()
    }

    override fun toggleTorch(enabled: Boolean?) {
        callService?.toggleTorch(enabled)
    }

    override fun onEndCall() {
        viewModel.endCall()
    }

    override fun setLayout(layout: HubLayout) {
        viewModel.setLayout(layout)
    }

    override fun startFullscreen() {
        cameraActive = true
        cameraFragment = CameraFragment.newInstance()
        cameraFragment?.setListener(this)
        fragmentManager.beginTransaction().replace(R.id.fullview_container, cameraFragment!!).commit()
        viewModel.clearScreenshot()
    }

    override fun toggleCamera() {
        callService?.toggleCamera()
    }

    override fun toggleMicrophone() {
        callService?.toggleMicrophone()
    }

    override fun toggleScreenShare() {
        if(viewModel.dailyState.replayCache.lastOrNull()?.screenShareActive == true) {
            callService?.stopScreenShare()
        } else {
            startScreenShare()
        }
    }

    override fun onClose() {
        fragmentManager.beginTransaction().remove(cameraFragment!!).commit()
        cameraActive = false
        cameraFragment = null
        fragmentManager.beginTransaction().replace(R.id.fullview_container, ongoingCallFragment).commit() // Change to DialogFragment?
        ScreenshotHelpers.takeScreenshot(hostType, requireActivity(), bitmapBuffer, flutterContentView) { bitmap ->
            viewModel.consumeScreenshot(true, bitmap)
        }
    }

    override fun logout() {
        viewModel.logout()
    }

    override fun requestCoBrowse() {
        viewModel.requestCoBrowse()
    }

    override fun cancelCoBrowseRequest() {
        viewModel.cancelCoBrowseRequest()
    }

    @OptIn(ExperimentalUuidApi::class)
    override fun register(
        organisationId: String,
        userId: String,
        deviceId: String,
        name: String,
        email: String,
        region: Region,
        config: Map<String, String>,
    ) {
        checkPermissions()

        val layout = config[FullviewInterface.LAYOUT]?.let { HubLayout.valueOf(it)  } ?: HubLayout.DEFAULT
        viewModel.setLayout(layout)

        viewModel.register(
            organisationId = organisationId,
            userId = userId,
            deviceId = deviceId,
            name = name,
            email= email,
            tabId = Uuid.random().toString(),
            region= region,
            config = config
        )
    }

    override fun endCoBrowse() {
        callService?.leave {

            if(this::incomingCallFragment.isInitialized && incomingCallFragment.isResumed) {
                fragmentManager.beginTransaction().remove(incomingCallFragment).commitAllowingStateLoss()
            }

            if(cameraFragment != null) {
                fragmentManager.beginTransaction().remove(cameraFragment!!).commitAllowingStateLoss()
                cameraFragment = null
            }

            if(this::fullscreenFragment.isInitialized && fullscreenFragment.isResumed) {
                fullscreenFragment.setListener(null)
                fragmentManager.beginTransaction().remove(fullscreenFragment).commitAllowingStateLoss()
            }


            if(this::ongoingCallFragment.isInitialized && ongoingCallFragment.isResumed) {
                ongoingCallFragment.setListener(null)
                fragmentManager.beginTransaction().remove(ongoingCallFragment).commitAllowingStateLoss()
            }

            if (highlightAgentSurfaceView.isAttachedToWindow && highlightAgentSurfaceView.isVisible) {
                SurfaceViewHelpers.clear(highlightAgentSurfaceView)
            }
            if (cursorAgentSurfaceView.isAttachedToWindow && cursorAgentSurfaceView.isVisible) {
                SurfaceViewHelpers.clear(cursorAgentSurfaceView)
            }

            cursorAgentSurfaceView.apply {
                val canvas = holder.lockCanvas()
                canvas.drawColor(Color.TRANSPARENT, PorterDuff.Mode.CLEAR)
                holder.unlockCanvasAndPost(canvas)
            }

            viewModel.clearScreenshot()
            viewModel.setState(FullviewSessionState.IDLE)
        }
    }

    private fun handleState(state: FullviewSessionState) {
        Timber.d("Handling state: $state")
        when(state) {
            FullviewSessionState.ATTACHED -> {}
            FullviewSessionState.IDLE -> {}
            is FullviewSessionState.CO_BROWSE_ACTIVE -> {
                ScreenshotHelpers.takeScreenshot(hostType, requireActivity(), bitmapBuffer, flutterContentView) { bitmap ->
                    viewModel.consumeScreenshot(true, bitmap)
                }

                callService?.join(state.webRTCURL) {
                    viewModel.updateLocalParticipant(it)
                }

                viewModel.setupCobrowse(state)
            }
            is FullviewSessionState.CO_BROWSE_INVITATION -> {
                incomingCallFragment = IncomingCallFragment.newInstance(state.id, state.agentName, state.privacyPolicyUrl, state.privacyPolicyText)
                incomingCallFragment.setListener(this@FullviewFragment)
                fragmentManager.beginTransaction().replace(R.id.fullview_container, incomingCallFragment).commit()
            }

            FullviewSessionState.CO_BROWSE_ENDED -> {
                endCoBrowse()
            }
            FullviewSessionState.CO_BROWSE_REQUESTED -> {}
        }
    }

    private fun handleLayoutState(layout: HubLayout, state: FullviewSessionState) {
        Timber.d("Handling layout + state: $layout $state")
        when(state) {
            FullviewSessionState.ATTACHED -> {}
            is FullviewSessionState.CO_BROWSE_ACTIVE -> {
                when(layout) {
                    HubLayout.LOCKED_FULLSCREEN, HubLayout.FULLSCREEN -> {
                        fullscreenFragment = FullscreenFragment.newInstance(state.id, state.agentName, layout)
                        fullscreenFragment.setListener(this@FullviewFragment)
                        fragmentManager.beginTransaction().replace(R.id.fullview_container, fullscreenFragment).commit()
                    }
                    HubLayout.DEFAULT -> {
                        ongoingCallFragment = OngoingCallFragment.newInstance(state.id, state.agentName, state.webRTCConfig, hostType)
                        ongoingCallFragment.setListener(this@FullviewFragment)
                        fragmentManager.beginTransaction().replace(R.id.fullview_container, ongoingCallFragment).commit()
                    }
                }
            }
            FullviewSessionState.CO_BROWSE_ENDED -> {}
            is FullviewSessionState.CO_BROWSE_INVITATION -> {}
            FullviewSessionState.CO_BROWSE_REQUESTED -> {}
            FullviewSessionState.IDLE -> {}
        }
    }


    /**
     * Daily
     */
    private fun initialize() {
        try {
            requireActivity().bindService(Intent(requireContext(), BackgroundCallService::class.java), serviceConnection, Context.BIND_AUTO_CREATE or Context.BIND_IMPORTANT)
        } catch (e: Exception) {
            Log.e(this::class.java.simpleName,  "Failed to bind to call service", e)
        }
    }

    private fun checkPermissions() {
        val permissionList = mutableListOf(
            Manifest.permission.CAMERA,
            Manifest.permission.RECORD_AUDIO,
        )

        if (Build.VERSION.SDK_INT >= 33) {
            permissionList.add(Manifest.permission.POST_NOTIFICATIONS)
        }

        val notGrantedPermissions = permissionList.map {
            Pair(it, ContextCompat.checkSelfPermission(requireActivity().applicationContext, it))
        }.filter {
            it.second != PackageManager.PERMISSION_GRANTED
        }.map {
            it.first
        }.toTypedArray()

        if (notGrantedPermissions.isNotEmpty()) {
            requestPermissionLauncher.launch(notGrantedPermissions)
        } else {
            // permission is granted, we can initialize
            initialize()
        }
    }

    override fun onStateChanged(newState: DailyState) {
        viewModel.setDailyState(newState)
    }

    private fun startScreenShare() {
        // Disable annotation for 31+ for now
        if (Build.VERSION.SDK_INT < 31) {
            if (!Settings.canDrawOverlays(requireActivity())) {
                val intent = Intent(
                    Settings.ACTION_MANAGE_OVERLAY_PERMISSION,
                    Uri.parse("package:" + requireActivity().applicationContext.packageName)
                )
                startActivityForResult(intent, 4713)
            }
        }
        val mediaProjectionIntent = mediaProjectionManager.createScreenCaptureIntent()
        requestMediaProjection.launch(mediaProjectionIntent)
    }
}