package io.fullview.fullview_sdk.camera

import android.graphics.Bitmap
import android.os.Bundle
import android.os.Handler
import android.os.HandlerThread
import android.util.DisplayMetrics
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.ImageButton
import androidx.fragment.app.Fragment
import androidx.fragment.app.viewModels
import androidx.lifecycle.lifecycleScope
import co.daily.capture.VideoTrackCapturerToBitmap
import co.daily.model.CallState
import co.daily.model.Participant
import co.daily.model.ParticipantId
import co.daily.view.VideoView
import io.fullview.fullview_sdk.DailyState
import io.fullview.fullview_sdk.DailyStateListener
import io.fullview.fullview_sdk.FullviewFragmentInterface
import io.fullview.fullview_sdk.R
import io.fullview.fullview_sdk.data.DisplayInfo
import io.fullview.fullview_sdk.ui.ToggleImageButton
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch

class CameraFragment : Fragment(), DailyStateListener {

    interface CameraFragmentInterface : FullviewFragmentInterface {
        fun onClose()
    }

    companion object {

        const val TAG = "CameraFragment"

        fun newInstance() = CameraFragment()
    }

    private val viewModel: CameraViewModel by viewModels<CameraViewModel>()

    private lateinit var root : View
    private lateinit var mainVideo : VideoView
    private val videoViews = mutableMapOf<ParticipantId, VideoView>()
    private var listener : CameraFragmentInterface? = null
    private lateinit var bitmapBuffer : Bitmap
    private lateinit var dailyCapture : VideoTrackCapturerToBitmap
    private var localParticipant : Participant? = null
    private val handlerThread = HandlerThread("VideoTrackCapturerToBitmapBackgroundThread")
    private var stopCapture = false

    fun setListener(callback: CameraFragmentInterface?) {
        listener = callback
    }

    override fun onCreateView(
        inflater: LayoutInflater, container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        root = inflater.inflate(R.layout.fragment_camera, container, false)

        if (handlerThread.isAlive.not())
            handlerThread.start()

        val payloadStream = resources.openRawResource(R.raw.rrwebsnapshot_template_fullscreen_camera)
        val payload = payloadStream.bufferedReader(Charsets.ISO_8859_1).use { it.readText() }
        viewModel.setTemplate(payload)

        dailyCapture = VideoTrackCapturerToBitmap(requireContext().applicationContext, Handler(handlerThread.looper))

        mainVideo = root.findViewById(R.id.daily_main)
        root.findViewById<ImageButton>(R.id.back_btn).setOnClickListener {
            stopCapture = true
            dailyCapture.track = null
            dailyCapture.release()
            mainVideo.track = null
            handlerThread.quitSafely()
            listener?.onClose()
        }

        root.findViewById<ImageButton>(R.id.switch_camera).setOnClickListener {
            listener?.toggleActiveCamera()
        }

        root.findViewById<ToggleImageButton>(R.id.flash).apply {
            onCheckedChangeListener = object : ToggleImageButton.OnCheckedChangeListener {
                override fun onCheckedChanged(buttonView: ToggleImageButton?, isChecked: Boolean) {
                    listener?.toggleTorch()
                    val drawable = if (isChecked) R.drawable.baseline_flashlight_on_24 else R.drawable.baseline_flashlight_off_24
                    (this@apply as ToggleImageButton).setImageResource(drawable)
                }
            }

            isChecked = false
        }

        val displayMetrics = DisplayMetrics()
        requireActivity().windowManager.defaultDisplay.getMetrics(displayMetrics)
        viewModel.displayInfo = DisplayInfo(displayMetrics.heightPixels, displayMetrics.widthPixels, displayMetrics.density)
        bitmapBuffer = Bitmap.createBitmap(displayMetrics.widthPixels, displayMetrics.heightPixels, Bitmap.Config.ARGB_8888)

        return root
    }

    override fun onResume() {
        super.onResume()

        listener?.attachService(TAG)

        viewLifecycleOwner.lifecycleScope.launch {
            while (this@CameraFragment.isResumed && stopCapture.not()) {
                dailyCapture.captureFrame {
                    viewModel.consumeScreenshot(it)
                }
                delay(100)
            }
        }
    }

    override fun onStateChanged(newState: DailyState) {
        super.onStateChanged(newState)
        if (newState.status == CallState.joined) {
            localParticipant = newState.allParticipants.values.first { it.info.isLocal }
            mainVideo.track = newState.localParticipantTrack
            videoViews[localParticipant!!.id] = mainVideo
            dailyCapture.track = mainVideo.track
        }
    }

    override fun onError(msg: String) {
        super.onError(msg)
    }
}