package io.fullview.fullview_sdk.fullscreen

import android.annotation.SuppressLint
import android.content.res.ColorStateList
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.ImageButton
import android.widget.TextView
import androidx.annotation.ColorRes
import androidx.annotation.DrawableRes
import androidx.appcompat.app.AlertDialog
import androidx.core.content.ContextCompat
import androidx.fragment.app.Fragment
import androidx.fragment.app.viewModels
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import co.daily.model.CallState
import co.daily.model.Participant
import co.daily.model.ParticipantId
import co.daily.view.VideoView
import io.fullview.fullview_sdk.DailyState
import io.fullview.fullview_sdk.DailyStateListener
import io.fullview.fullview_sdk.HubLayout
import io.fullview.fullview_sdk.R
import io.fullview.fullview_sdk.data.WebRTCConfig
import io.fullview.fullview_sdk.ongoing.OngoingCallInterface
import io.fullview.fullview_sdk.ui.ToggleImageButton
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.filterNotNull
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.launch
import kotlin.time.Duration.Companion.seconds

class FullscreenFragment : Fragment(), DailyStateListener {

    companion object {

        const val TAG = "FullscreenFragment"
        const val ARG_AGENT_NAME = "fullview_agent_name-fullscreen"
        const val ARG_COBROWSE_ID = "fullview_cobrowse_id-fullscreen"
        const val ARG_HUB_LAYOUT = "fullview_hub_layout"


        @JvmStatic
        fun newInstance(invitationId: String, agentName: String, layout: HubLayout) = FullscreenFragment().apply {
            arguments = Bundle().apply {
                putString(ARG_AGENT_NAME, agentName)
                putString(ARG_COBROWSE_ID, invitationId)
                putParcelable(ARG_HUB_LAYOUT, layout)
            }
        }
    }
    private val viewModel: FullscreenViewModel by viewModels<FullscreenViewModel>()

    private lateinit var root : View
    private lateinit var hostVideoView : VideoView
    private lateinit var agentVideoView : VideoView
    private lateinit var agentVideoDisabledLayout : View
    private lateinit var agentVideoEnabledLayout : View
    private lateinit var hostVideoDisabledLayout : View
    private lateinit var hostVideoEnabledLayout : View
    private lateinit var startScreenShareButtonLayout : View
    private lateinit var stopScreenShareButtonLayout : View
    private lateinit var minimizeButton : ImageButton

    private lateinit var toggleMic : ToggleImageButton
    private lateinit var toggleCamera : ToggleImageButton
    private var localParticipant: Participant? = null



    // Create map of video views
    private val videoViews = mutableMapOf<ParticipantId, VideoView>()

    private lateinit var agentName: String
    private lateinit var id: String
    private lateinit var layout: HubLayout

    private var listener: OngoingCallInterface? = null
    fun setListener(listener: OngoingCallInterface?) {
        this.listener = listener
    }

    override fun onResume() {
        super.onResume()
        listener?.attachService(TAG)
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        val currentOrientation = resources.configuration.orientation
        viewModel.updateOrientation(currentOrientation)

        arguments?.let {
            agentName = it.getString(ARG_AGENT_NAME) ?: "Unknown"
            id = it.getString(ARG_COBROWSE_ID) ?: "-"
            layout = it.getParcelable(ARG_HUB_LAYOUT) ?: HubLayout.LOCKED_FULLSCREEN
        }
    }

    override fun onCreateView(
        inflater: LayoutInflater, container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        root =  inflater.inflate(R.layout.fragment_fullscreen, container, false)
        hostVideoView = root.findViewById(R.id.host_video)
        agentVideoView = root.findViewById(R.id.agent_video)
        toggleMic = root.findViewById(R.id.toggle_mic)
        toggleCamera = root.findViewById(R.id.toggle_camera)
        agentVideoDisabledLayout = root.findViewById(R.id.no_agent_camera_layout)
        agentVideoEnabledLayout = root.findViewById(R.id.agent_camera_layout)
        hostVideoDisabledLayout = root.findViewById(R.id.no_host_camera_layout)
        hostVideoEnabledLayout = root.findViewById(R.id.host_camera_layout)
        startScreenShareButtonLayout = root.findViewById(R.id.start_screen_share)
        stopScreenShareButtonLayout = root.findViewById(R.id.stop_screen_share)
        minimizeButton = root.findViewById(R.id.minimize)

        if (layout == HubLayout.FULLSCREEN) {
            minimizeButton.visibility = View.VISIBLE
            minimizeButton.setOnClickListener {
                listener?.setLayout(HubLayout.DEFAULT)
            }
        } else {
            minimizeButton.visibility = View.GONE
        }

        if (viewModel.startOfCall.replayCache.isEmpty()) {
            viewModel.setStart()
        }

        root.findViewById<ImageButton>(R.id.end_call).setOnClickListener {
            AlertDialog.Builder(requireContext(), R.style.AlertDialog_AppCompat_Fullview)
                .setTitle(getString(R.string.end_cobrowse))
                .setMessage(getString(R.string.are_you_sure_you_want_to_end_the_current_cobrowsing_session))
                .setPositiveButton(android.R.string.ok
                ) { _, _ ->
                    viewModel.clearStart()
                    listener?.onEndCall()
                }
                .setNegativeButton(android.R.string.cancel) { _ , _ -> }
                .show()
        }


        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.RESUMED) {
                viewModel.tickerFlow(1.seconds)
                    .onEach {
                        viewModel.startOfCall.replayCache.lastOrNull()?.apply {
                            val secondsSinceStart = (System.currentTimeMillis() - this)/1000
                            requireActivity().runOnUiThread {
                                root.findViewById<TextView>(R.id.timer).text = getString(
                                    R.string.fullview_timer,
                                    (secondsSinceStart / 60).toString().padStart(2, '0'),
                                    (secondsSinceStart % 60).toString().padStart(2, '0')
                                )
                            }
                        }
                    }
                    .launchIn(this@launch)
            }
        }


        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.RESUMED) {
                viewModel
                    .dailyState
                    .filterNotNull()
                    .collectLatest {
                        when(it.status) {
                            CallState.initialized -> {}
                            CallState.joining -> {}
                            CallState.joined -> {
                                val local = it.allParticipants.values.first { it.info.isLocal }
                                localParticipant = local
                                videoViews[local.id] = hostVideoView

                                if (it.localParticipantTrack != null) {
                                    hostVideoView.track = it.localParticipantTrack
                                    hostVideoView.visibility = View.VISIBLE
                                    hostVideoEnabledLayout.visibility = View.VISIBLE
                                    hostVideoDisabledLayout.visibility = View.GONE
                                } else {
                                    hostVideoView.track = null
                                    hostVideoView.visibility = View.GONE
                                    hostVideoEnabledLayout.visibility = View.GONE
                                    hostVideoDisabledLayout.visibility = View.VISIBLE
                                }


                                val agent = it.allParticipants.values.firstOrNull { it.info.isLocal.not() }
                                if(agent?.media?.camera?.track !== null) {
                                    agentVideoView.track = agent.media?.camera?.track
                                    agentVideoView.visibility = View.VISIBLE
                                    videoViews[agent.id] = agentVideoView
                                    agentVideoDisabledLayout.visibility = View.GONE
                                    agentVideoEnabledLayout.visibility = View.VISIBLE
                                } else {
                                    agentVideoView.track = null
                                    agentVideoView.visibility = View.GONE
                                    agentVideoDisabledLayout.visibility = View.VISIBLE
                                    agentVideoEnabledLayout.visibility = View.GONE
                                }

                                toggleButtonUI(
                                    toggleCamera,
                                    it.inputs.cameraEnabled,
                                    enabledDrawable = R.drawable.baseline_videocam_24,
                                    disabledDrawable = R.drawable.baseline_videocam_off_24
                                )

                                toggleButtonUI(
                                    toggleMic,
                                    it.inputs.micEnabled,
                                    enabledDrawable = R.drawable.baseline_mic_24,
                                    disabledDrawable = R.drawable.baseline_mic_off_24
                                )

                                if (it.screenShareActive) {
                                    stopScreenShareButtonLayout.visibility = View.VISIBLE
                                    startScreenShareButtonLayout.visibility = View.GONE
                                } else {
                                    stopScreenShareButtonLayout.visibility = View.GONE
                                    startScreenShareButtonLayout.visibility = View.VISIBLE
                                }
                                /*
                                toggleButtonUI(
                                    screenShare,
                                    it.inputs.screenVideoEnabled,
                                    enabledDrawable = R.drawable.baseline_mobile_screen_share_24,
                                    disabledBackgroundColor = R.color.fullview_ios_dark_gray,
                                    disabledDrawable = R.drawable.baseline_mobile_screen_share_24,
                                    disabledIconTint = R.color.offwhite
                                )*/


                                setupToggles()
                            }
                            CallState.leaving -> {}
                            CallState.left -> {}
                        }
                    }
            }
        }

        return root
    }


    override fun onStateChanged(newState: DailyState) {
        if (isResumed) {
            viewModel.setDailyState(newState)
        }
    }

    override fun onError(msg: String) {

    }

    private fun toggleButtonUI(
        buttonView: ToggleImageButton?,
        isChecked: Boolean,
        @ColorRes enabledBackgroundColor: Int = R.color.cobrowse_experience_enable_button_background,
        @DrawableRes enabledDrawable: Int,
        @ColorRes enabledIconTint: Int = R.color.cobrowse_experience_enable_button_content,
        @ColorRes disabledBackgroundColor: Int = R.color.cobrowse_experience_disable_button_background,
        @DrawableRes disabledDrawable: Int,
        @ColorRes disabledIconTint: Int = R.color.cobrowse_experience_disable_button_content
    ) {
        if (isChecked) {
            buttonView?.backgroundTintList = ColorStateList.valueOf(
                ContextCompat.getColor(requireContext(),
                enabledBackgroundColor
            ))
            buttonView?.setImageResource(enabledDrawable)
            buttonView?.imageTintList = ColorStateList.valueOf(
                ContextCompat.getColor(requireContext(),
                enabledIconTint
            ))
        } else {
            buttonView?.backgroundTintList = ColorStateList.valueOf(
                ContextCompat.getColor(requireContext(),
                disabledBackgroundColor
            ))
            buttonView?.setImageResource(disabledDrawable)
            buttonView?.imageTintList = ColorStateList.valueOf(
                ContextCompat.getColor(requireContext(),
                disabledIconTint
            ))
        }
    }

    @SuppressLint("ClickableViewAccessibility")
    private fun setupToggles() {
        toggleCamera.setOnClickListener {
            listener?.toggleCamera()
        }

        toggleMic.setOnClickListener {
            listener?.toggleMicrophone()
        }

        startScreenShareButtonLayout.setOnClickListener {
            listener?.toggleScreenShare()
        }

        stopScreenShareButtonLayout.setOnClickListener {
            listener?.toggleScreenShare()
        }
    }
}