package io.fullview.fullview_sdk.services

import android.app.Notification
import android.app.NotificationChannel
import android.app.NotificationManager
import android.app.PendingIntent
import android.app.Service
import android.content.ComponentName
import android.content.Intent
import android.content.ServiceConnection
import android.content.pm.ServiceInfo
import android.os.Build
import android.os.IBinder
import androidx.core.app.NotificationCompat
import androidx.core.app.ServiceCompat
import io.fullview.fullview_sdk.DailyState
import io.fullview.fullview_sdk.DailyStateListener
import io.fullview.fullview_sdk.R
import timber.log.Timber

abstract class ForegroundService(
    private val notificationChannelId: String,
    private val notificationChannelName: String,
) : Service(), DailyStateListener {

    private val serviceConnection = object : ServiceConnection {
        override fun onServiceConnected(name: ComponentName?, service: IBinder?) {
            callService = service!! as BackgroundCallService.Binder
            callService!!.addListener(this@ForegroundService)

            callService!!.screenShareForegroundServiceStarted()
        }

        override fun onServiceDisconnected(name: ComponentName?) {
            callService = null
        }
    }

    private lateinit var notification: Notification

    private var callService: BackgroundCallService.Binder? = null

    override fun onBind(intent: Intent?): IBinder? = null

    override fun onCreate() {
        super.onCreate()

        val channel = NotificationChannel(
            notificationChannelId,
            notificationChannelName,
            NotificationManager.IMPORTANCE_LOW
        )

        val manager = getSystemService(NotificationManager::class.java)
        manager.createNotificationChannel(channel)

        val pendingIntent = PendingIntent.getActivity(
            this,
            0,
            packageManager.getLaunchIntentForPackage(applicationContext.packageName),
            PendingIntent.FLAG_IMMUTABLE
        )

        val leaveIntent = PendingIntent.getService(
            this,
            0,
            BackgroundCallService.leaveIntent(this),
            PendingIntent.FLAG_IMMUTABLE
        )

        notification = NotificationCompat.Builder(this, notificationChannelId)
            .setSmallIcon(R.drawable.logo_white)
            .setContentTitle(resources.getString(R.string.service_notification_title))
            .setContentText(resources.getString(R.string.service_notification_message))
            .setShowWhen(true)
            .setContentIntent(pendingIntent)
            .addAction(
                NotificationCompat.Action(
                    0,
                    resources.getString(R.string.hangup_button),
                    leaveIntent
                )
            )
            .build()

        if (!bindService(
                Intent(
                        this,
                        BackgroundCallService::class.java
                    ),
                serviceConnection, BIND_AUTO_CREATE or BIND_IMPORTANT
            )
        ) {
            throw RuntimeException("Failed to bind to call service")
        }
    }

    override fun onStartCommand(intent: Intent?, flags: Int, startId: Int): Int {
        if(intent?.component?.className?.contains("ActiveCallService") == true) {
            ServiceCompat.startForeground(
                this,
                4711,
                notification,
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
                    ServiceInfo.FOREGROUND_SERVICE_TYPE_MICROPHONE
                } else {
                    0
                }
            )
        } else if (intent?.component?.className?.contains("ScreenShareService") == true) {


            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                ServiceCompat.startForeground(
                    this,
                    4711,
                    notification,
                    ServiceInfo.FOREGROUND_SERVICE_TYPE_MEDIA_PROJECTION
                )
            } else {
                ServiceCompat.startForeground(
                    this,
                    4711,
                    notification,
                    0
                )
            }
        } else {
            // noop
        }
        return START_STICKY
    }

    abstract override fun onStateChanged(newState: DailyState)

    override fun onDestroy() {
        super.onDestroy()
        callService?.removeListener(this)
        unbindService(serviceConnection)
    }
}
