package repository

import access.models.AcceptCoBrowse
import access.models.AcceptCoBrowseResponse
import access.models.CoBrowseUpdateParticipant
import access.models.DeclineCoBrowse
import access.models.OrganisationSDKConfig
import access.models.RegisterCustomer
import access.models.RegisterCustomerCustomerExternalData
import access.models.RegisterCustomerResponse
import access.models.SupportRequest
import android.util.Log
import io.fullview.fullview_sdk.BuildConfig
import io.fullview.fullview_sdk.KoinApp
import io.fullview.fullview_sdk.Region
import io.fullview.fullview_sdk.data.User
import io.fullview.fullview_sdk.data.toUser
import io.fullview.fullview_sdk.helpers.toBaseUrl
import io.fullview.fullview_sdk.helpers.toBearer
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.channels.BufferOverflow
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.SharedFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.launch
import timber.log.Timber
import java.util.Calendar
import java.util.TimeZone
import kotlin.uuid.ExperimentalUuidApi
import kotlin.uuid.Uuid

class AccessRepository {

    private val api = KoinApp.getApiService()

    private val _currentUser: MutableSharedFlow<User?> = MutableSharedFlow(replay = 1, onBufferOverflow = BufferOverflow.DROP_OLDEST)
    val currentUser: SharedFlow<User?> = _currentUser.asSharedFlow()

    private val _organisationConfig: MutableSharedFlow<OrganisationSDKConfig> = MutableSharedFlow(replay = 1, onBufferOverflow = BufferOverflow.DROP_OLDEST)
    val organisationConfig = _organisationConfig.asSharedFlow()


    fun clearUser() {
        _currentUser.tryEmit(null)
    }

    fun register(
        organisationId: String,
        userId: String,
        deviceId: String,
        name: String,
        email: String,
        tabId: String,
        region: Region,
        invitationId: String?
    ) {
        CoroutineScope(Dispatchers.Default).launch {
            val registerFlow = flow {
                registerCall(
                    organisationId = organisationId,
                    deviceId = deviceId,
                    userId = userId,
                    name = name,
                    email = email,
                    tabId = tabId,
                    region = region,
                    invitationId = invitationId
                )
                    .onSuccess { emit(it) }
                    .onFailure { Log.e(this@AccessRepository::class.java.simpleName, "Error when registering user", it) }
            }

            val orgConfigFlow = flow {
                fetchOrganisationConfig(organisationId, region)
                    .onSuccess { emit(it) }
                    .onFailure { Log.e(this@AccessRepository::class.java.simpleName, "Error when fetching organisation config", it) }
            }

            combine(flow = orgConfigFlow, flow2 = registerFlow) { orgConfig, user ->
                _organisationConfig.tryEmit(orgConfig)
                if (orgConfig.isBlocked?.not() == true) {
                    _currentUser.tryEmit(user.toUser(region))
                }
            }.collectLatest {}
        }
    }

    private suspend fun registerCall(
        organisationId: String,
        deviceId: String,
        userId: String,
        name: String,
        email: String,
        tabId: String,
        region: Region,
        invitationId: String? = null,
    ) : Result<RegisterCustomerResponse> {
        return try {

            val data = RegisterCustomer(
                organisationId = organisationId,
                deviceId= deviceId,
                tabId= tabId,
                externalData = RegisterCustomerCustomerExternalData(
                    externalId = userId,
                    name = name,
                    email = email,
                    env = null,
                    customData = emptyMap()
                ),
                domain = "app.fullview.io",
                sdkPlatform = "android",
                sdkVersion = BuildConfig.FULLVIEW_SDK_VERSION,
                userAgent = "android-fullview-io-sdk:${BuildConfig.FULLVIEW_SDK_VERSION}",
                tabOpenAt = Calendar.getInstance(TimeZone.getTimeZone("UTC")).timeInMillis,
                tabCurrentUrl = "https://app.fullview.io",
                deviceInfo = null,
                isAnonymous = false,
                organisationInvitationCode = invitationId
            )


            Result.success(api.registerCustomer(
                url= region.toBaseUrl("/access/api/customers/register"),
                data
            ))
        } catch (e: Exception) {
            Result.failure(e)
        }
    }

    private suspend fun fetchOrganisationConfig(
        organisationId: String,
        region: Region
    ) : Result<OrganisationSDKConfig> {
        return try {
            Result.success(api.getOrganisationConfig(url = region.toBaseUrl("/access/api/organisations/$organisationId/config/sdk")))
        } catch (e: Exception) {
            Result.failure(e)
        }
    }

    suspend fun endCoBrowse(user: User) {
        try {
            api.endCoBrowse(url= user.region.toBaseUrl("/access/api/customers/end-co-browse"), user.accessToken.toBearer())
        } catch (e: Exception) {
            Log.e(this::class.java.simpleName, "Error when ending cobrowse", e)
        }
    }

    suspend fun acceptCoBrowse(
        id: String,
        user: User,
    ) : Result<AcceptCoBrowseResponse> {
        return try {
            Result.success(api.acceptCoBrowse(url = user.region.toBaseUrl("/access/api/customers/accept-co-browse"), AcceptCoBrowse(id), user.accessToken.toBearer()))
        } catch (e: Exception) {
            Result.failure(e)
        }
    }

    suspend fun declineCobrowse(
        id: String,
        user: User,
    ) {
        try {
            api.declineCobrowse(url = user.region.toBaseUrl("/access/api/customers/decline-co-browse"), DeclineCoBrowse(id), user.accessToken.toBearer())
        } catch (e: Exception) {
            Log.e(this::class.java.simpleName, "Error when declining cobrowse", e)
        }
    }

    @OptIn(ExperimentalUuidApi::class)
    suspend fun requestCoBrowse(
        user: User,
    ) : Result<Unit> {
        return try {
            Result.success(
                api.requestCoBrowse(
                    user.region.toBaseUrl("/access/api/customers/request-support"),
                    user.accessToken.toBearer(),
                    SupportRequest(Uuid.random().toString())
                )
            )
        } catch (e: Exception) {
            Result.failure(e)
        }
    }

    suspend fun cancelRequestCoBrowse(
        user: User,
    ) : Result<Unit> {
        return try {
            Result.success(api.cancelCoBrowseRequest(url = user.region.toBaseUrl("/access/api/customers/cancel-support-request"), user.accessToken.toBearer()))
        }catch (e: Exception) {
            Result.failure(e)
        }
    }

    suspend fun updateParticipant(
        participant: CoBrowseUpdateParticipant,
        user: User,
    ) : Result<Unit> {
        return try {
            Result.success(api.updateParticipant(url = user.region.toBaseUrl("/access/api/customers/update-participant"), participant, user.accessToken.toBearer()))
        } catch (e: Exception) {
            Result.failure(e)
        }
    }
}