package repository

import io.fullview.fullview_sdk.FullviewSessionState
import io.fullview.fullview_sdk.HubLayout
import io.fullview.fullview_sdk.data.HubCoordinates
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.channels.BufferOverflow
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.asSharedFlow

class SessionRepository {

    private val _sessionState: MutableSharedFlow<FullviewSessionState> =
        MutableSharedFlow(replay = 1, onBufferOverflow = BufferOverflow.DROP_OLDEST)
    internal val sessionState = _sessionState.asSharedFlow()

    private val _hubCoordinates: MutableSharedFlow<HubCoordinates> =
        MutableSharedFlow(replay = 1, onBufferOverflow = BufferOverflow.DROP_OLDEST)
    internal val hubCoordinates = _hubCoordinates.asSharedFlow()

    private val _queue : MutableSharedFlow<Int> =
        MutableSharedFlow(replay = 1, onBufferOverflow = BufferOverflow.DROP_OLDEST)
    val queue = _queue.asSharedFlow()

    private val _orientation : MutableSharedFlow<Int> =
        MutableSharedFlow(replay = 1, onBufferOverflow = BufferOverflow.DROP_OLDEST)

    private val _startOfCall : MutableSharedFlow<Long> =
        MutableSharedFlow(replay = 1, onBufferOverflow = BufferOverflow.DROP_OLDEST)
    val startOfCall = _startOfCall.asSharedFlow()

    private val _layout : MutableSharedFlow<HubLayout> =
        MutableSharedFlow(replay = 1, onBufferOverflow = BufferOverflow.DROP_OLDEST)
    val currentLayout = _layout.asSharedFlow()

    fun updateSessionState(state: FullviewSessionState) {
        _sessionState.tryEmit(state)
    }

    fun updateHubPosition(position: HubCoordinates) {
        _hubCoordinates.tryEmit(position)
    }

    fun updateQueue(position: Int) {
        _queue.tryEmit(position)
    }

    fun setStart() {
        _startOfCall.tryEmit(System.currentTimeMillis())
    }

    fun setLayout(layout: HubLayout) {
        _layout.tryEmit(layout)
    }

    @OptIn(ExperimentalCoroutinesApi::class)
    fun clearStart() {
        _startOfCall.resetReplayCache()
    }

    @OptIn(ExperimentalCoroutinesApi::class)
    fun updateOrientation(orientation: Int) {
        if(_orientation.replayCache.lastOrNull() != orientation) {
            _orientation.tryEmit(orientation)
            _hubCoordinates.resetReplayCache()
        }
    }
}