/*
 * Copyright 2011-2022 GatlingCorp (https://gatling.io)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.gatling.core.util;

import java.nio.ByteBuffer;

public class Integers {

  static final byte[] DIGIT_TENS = {
    '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '1', '1', '1', '1', '1', '1', '1', '1', '1',
    '1', '2', '2', '2', '2', '2', '2', '2', '2', '2', '2', '3', '3', '3', '3', '3', '3', '3', '3',
    '3', '3', '4', '4', '4', '4', '4', '4', '4', '4', '4', '4', '5', '5', '5', '5', '5', '5', '5',
    '5', '5', '5', '6', '6', '6', '6', '6', '6', '6', '6', '6', '6', '7', '7', '7', '7', '7', '7',
    '7', '7', '7', '7', '8', '8', '8', '8', '8', '8', '8', '8', '8', '8', '9', '9', '9', '9', '9',
    '9', '9', '9', '9', '9',
  };

  static final byte[] DIGIT_ONES = {
    '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '0', '1', '2', '3', '4', '5', '6', '7', '8',
    '9', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '0', '1', '2', '3', '4', '5', '6', '7',
    '8', '9', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '0', '1', '2', '3', '4', '5', '6',
    '7', '8', '9', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '0', '1', '2', '3', '4', '5',
    '6', '7', '8', '9', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '0', '1', '2', '3', '4',
    '5', '6', '7', '8', '9',
  };

  static final byte[] DIGITS = {
    '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i',
    'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z'
  };

  private static final int[] SIZE_TABLE = {
    9, 99, 999, 9999, 99999, 999999, 9999999, 99999999, 999999999, 0xFFFFFFFF
  };

  public static int positiveIntStringSize(int num) {
    int l2 = 31 - Integer.numberOfLeadingZeros(num);
    int ans = ((77 * l2) >>> 8);
    if (num > SIZE_TABLE[ans]) {
      ans += 1;
    }
    return ans + 1;
  }

  public static void writePositiveIntString(int i, ByteBuffer bb) {
    writePositiveIntString(i, positiveIntStringSize(i), bb);
  }

  public static void writePositiveIntString(int i, int stringSize, ByteBuffer bb) {
    int q, r;
    int finalPosition = bb.position() + stringSize;
    int charPos = finalPosition;

    // Generate two digits per iteration
    while (i >= 65536) {
      q = i / 100;
      // really: r = i - (q * 100);
      r = i - ((q << 6) + (q << 5) + (q << 2));
      i = q;
      bb.put(--charPos, DIGIT_ONES[r]);
      bb.put(--charPos, DIGIT_TENS[r]);
    }

    // Fall thru to fast mode for smaller numbers
    // assert(i <= 65536, i);
    for (; ; ) {
      q = (i * 52429) >>> (16 + 3);
      r = i - ((q << 3) + (q << 1)); // r = i-(q*10) ...
      bb.put(--charPos, DIGITS[r]);
      i = q;
      if (i == 0) break;
    }

    bb.position(finalPosition);
  }
}
