/*
 * Copyright 2011-2022 GatlingCorp (https://gatling.io)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.gatling.plugin.util.checksum;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Base64;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

public final class PkgChecksum {

  private static final String MANIFEST_NAME = "META-INF/MANIFEST.MF";

  private PkgChecksum() {}

  public static String computeChecksum(File file) throws IOException {
    try {
      MessageDigest md5 = MessageDigest.getInstance("MD5");
      try (FileInputStream fis = new FileInputStream(file);
          BufferedInputStream bis = new BufferedInputStream(fis);
          ZipInputStream zis = new ZipInputStream(bis)) {
        ZipEntry entry;
        while ((entry = zis.getNextEntry()) != null) {
          if (!entry.getName().equals(MANIFEST_NAME)) {
            // ZipEntry#getCrc is only populated once the entry is closed
            zis.closeEntry();
            md5.update(BigInteger.valueOf(entry.getCrc()).toByteArray());
          }
        }
      }
      return Base64.getEncoder().encodeToString(md5.digest());
    } catch (NoSuchAlgorithmException e) {
      throw new IllegalStateException("Checksum algorithm was not available", e);
    }
  }
}
