/*
 * Copyright 2011-2023 GatlingCorp (https://gatling.io)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.gatling.plugin.util;

import java.io.*;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;

public final class ForkMain {

  static final String ARGS_RESOURCE = "META-INF/args.txt";

  public static void main(String[] args) {
    try {
      String mainClassName = args[0];
      ClassLoader cl = Thread.currentThread().getContextClassLoader();
      List<String> actualArgs = readActualArgs(cl);
      runMain(cl, mainClassName, actualArgs);
    } catch (Throwable t) {
      t.printStackTrace();
      System.exit(-1);
    }
  }

  private static List<String> readActualArgs(ClassLoader cl) throws IOException {
    List<String> args = new ArrayList<>();
    try (BufferedReader reader =
        new BufferedReader(
            new InputStreamReader(cl.getResourceAsStream(ARGS_RESOURCE), StandardCharsets.UTF_8))) {
      String line;
      while ((line = reader.readLine()) != null) {
        args.add(line);
      }
      return args;
    }
  }

  private static void runMain(ClassLoader cl, String mainClassName, List<String> args)
      throws Exception {
    Class<?> mainClass = cl.loadClass(mainClassName);
    Method mainMethod = mainClass.getMethod("main", String[].class);
    int mods = mainMethod.getModifiers();
    if (mainMethod.getReturnType() != void.class
        || !Modifier.isStatic(mods)
        || !Modifier.isPublic(mods)) {
      throw new NoSuchMethodException("main");
    }

    String[] argsArray = args.toArray(new String[0]);
    mainMethod.invoke(null, new Object[] {argsArray});
  }
}
