/*
 * Copyright 2011-2023 GatlingCorp (https://gatling.io)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.gatling.plugin;

import static io.gatling.plugin.util.ObjectsUtil.nonNullParam;

import io.gatling.plugin.client.EnterpriseClient;
import io.gatling.plugin.exceptions.*;
import io.gatling.plugin.io.PluginLogger;
import io.gatling.plugin.model.*;
import io.gatling.scanner.AsmSimulationScanner;
import io.gatling.scanner.HighestJavaVersionClass;
import io.gatling.scanner.SimulationScanResult;
import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Timer;
import java.util.UUID;
import java.util.concurrent.CountDownLatch;

abstract class PluginClient implements EnterprisePlugin {

  protected final EnterpriseClient enterpriseClient;
  protected final PluginLogger logger;

  public PluginClient(EnterpriseClient enterpriseClient, PluginLogger logger) {
    this.enterpriseClient = enterpriseClient;
    this.logger = logger;
  }

  protected long uploadPackageWithChecksum(UUID packageId, File file)
      throws EnterprisePluginException {
    nonNullParam(packageId, "packageId");
    nonNullParam(file, "file");
    if (enterpriseClient.uploadPackageWithChecksum(packageId, file) == -1) {
      logger.info("No code changes detected, skipping package upload");
    } else {
      logger.info("Package uploaded");
    }
    return file.length();
  }

  protected SimulationScanResult scanPackage(File file) throws EnterprisePluginException {
    SimulationScanResult scanResult;
    try {
      scanResult = AsmSimulationScanner.scan(file);
    } catch (IOException e) {
      throw new SimulationScannerIOException(file, e);
    }

    if (scanResult.getHighestJavaVersionClass() == null) {
      throw new PackageEmptyException();
    }
    if (scanResult.getSimulationClasses().isEmpty()) {
      throw new NoSimulationClassNameFoundException();
    }

    final int maximumJavaVersion =
        Integer.parseInt(enterpriseClient.getServerInformation().versions.java.max);
    HighestJavaVersionClass highestJavaVersionClass = scanResult.getHighestJavaVersionClass();
    if (highestJavaVersionClass.javaVersion > maximumJavaVersion) {
      throw new UnsupportedJavaVersionException(
          highestJavaVersionClass.clazz, highestJavaVersionClass.javaVersion, maximumJavaVersion);
    }

    return scanResult;
  }

  protected List<String> simulationClassesFromCompatibleByteCodeFile(File file)
      throws EnterprisePluginException {
    return scanPackage(file).getSimulationClasses();
  }

  @Override
  public SimulationEndResult waitForRunEnd(RunSummary startedRun) throws EnterprisePluginException {
    final CountDownLatch latch = new CountDownLatch(1);
    final Timer timer = new Timer();
    final RunStatusTask task = new RunStatusTask(enterpriseClient, logger, startedRun.runId, latch);

    final SimulationEndResult finishedRun;
    try {
      timer.schedule(task, 5000L, 5000L);
      latch.await();
    } catch (Exception e) {
      throw new SimulationOngoingRunException(startedRun.runId, e);
    } finally {
      timer.cancel();
    }
    if (task.exception != null) {
      throw new SimulationOngoingRunException(startedRun.runId, task.exception);
    }

    finishedRun = task.result;

    logAssertionResults(finishedRun.assertions);
    logResult(startedRun, finishedRun.status);

    return finishedRun;
  }

  @Override
  public boolean abortRun(UUID runId) throws EnterprisePluginException {
    return enterpriseClient.abortRun(runId);
  }

  private void logAssertionResults(List<RunAssertion> assertions) {
    if (!assertions.isEmpty()) {
      logger.info("Assertion results:");
      for (RunAssertion assertion : assertions) {
        if (assertion.result) {
          logger.info("> " + assertion.message + " succeeded with value " + assertion.actualValue);
        } else {
          logger.error("> " + assertion.message + " failed with value " + assertion.actualValue);
        }
      }
      logger.info("");
    }
  }

  private void logResult(RunSummary startedRun, RunStatus status) {
    if (status.successful) {
      logger.info("Simulation result: Run " + startedRun.runId + " finished with status " + status);
    } else {
      logger.error("Simulation result: Run " + startedRun.runId + " failed with status " + status);
    }
    logger.info("See the run reports at " + enterpriseClient.getBaseUrl() + startedRun.reportsPath);
  }
}
