/*
 * Copyright 2011-2023 GatlingCorp (https://gatling.io)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.gatling.plugin;

import static io.gatling.plugin.util.DurationFormatter.formatDuration;

import io.gatling.plugin.client.EnterpriseClient;
import io.gatling.plugin.exceptions.EnterprisePluginException;
import io.gatling.plugin.io.PluginLogger;
import io.gatling.plugin.model.*;
import java.time.Instant;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.List;
import java.util.TimerTask;
import java.util.UUID;
import java.util.concurrent.CountDownLatch;
import java.util.stream.Collectors;

final class RunStatusTask extends TimerTask {
  private final EnterpriseClient enterpriseClient;
  private final PluginLogger logger;
  private final UUID runId;
  private final CountDownLatch latch;

  private int errorCount = 0;
  private RunStatus oldStatus;

  public SimulationEndResult result;
  public Exception exception;

  RunStatusTask(
      EnterpriseClient enterpriseClient, PluginLogger logger, UUID runId, CountDownLatch latch) {
    this.enterpriseClient = enterpriseClient;
    this.logger = logger;
    this.runId = runId;
    this.latch = latch;
  }

  @Override
  public void run() {
    try {
      final RunInformation runInfo = enterpriseClient.getRunInformation(runId);
      if (runInfo.status != oldStatus) {
        logger.info(
            String.format("Run status is now %s [%s]", runInfo.status, runInfo.status.value));
      }
      oldStatus = runInfo.status;
      if (runInfo.injectStart > 0) {
        logMetricsSummary(getMetricsSummary(runInfo));
      }
      errorCount = 0;
      if (!runInfo.status.running) {
        setResult(new SimulationEndResult(runInfo.status, runInfo.assertions));
      }
    } catch (Exception e) {
      errorCount++;
      if (errorCount < 5) {
        final String causeMessage = e.getCause() != null ? ": " + e.getCause().getMessage() : "";
        logger.info(
            String.format(
                "Failed to retrieve current run information (attempt %s/5): %s%s",
                errorCount, e.getMessage(), causeMessage));
      } else {
        setException(e);
      }
    }
  }

  private MetricsSummary getMetricsSummary(RunInformation runInfo)
      throws EnterprisePluginException {
    final List<Series> seriesResponse =
        enterpriseClient.getConcurrentUserMetric(runInfo.runId, runInfo.scenario);
    final RequestsSummary requestsSummary = enterpriseClient.getRequestsSummary(runInfo.runId);

    final Instant currentTimestamp = Instant.now();
    final String date =
        DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")
            .withZone(ZoneOffset.UTC)
            .format(currentTimestamp);

    final String duration = formatDuration(runInfo.injectStart, currentTimestamp.toEpochMilli());

    final double nbUsers =
        seriesResponse.stream()
            .map(s -> s.values.isEmpty() ? 0.0 : s.values.get(s.values.size() - 1))
            .reduce(0.0, Double::sum);
    final double nbRequest = requestsSummary.out.counts.total;
    final double requestsSeconds = requestsSummary.out.rps.total;
    final double failureRatio = requestsSummary.in.counts.koPercent;
    final List<MetricsSummary.ChildMetric> listMetric =
        recursivelyGetChildren(requestsSummary.children);

    return new MetricsSummary(
        date, duration, nbUsers, nbRequest, requestsSeconds, failureRatio, listMetric);
  }

  private List<MetricsSummary.ChildMetric> recursivelyGetChildren(
      List<RequestsSummary.Child> children) {
    return children.stream()
        .map(
            child ->
                child.children == null
                    ? new MetricsSummary.ChildMetric(
                        child.name,
                        child.out.counts.total,
                        child.in.counts.koPercent,
                        child.out.rps.total)
                    : new MetricsSummary.ChildMetric(
                        child.name, recursivelyGetChildren(child.children)))
        .collect(Collectors.toList());
  }

  private void logMetricsSummary(MetricsSummary summary) {
    final StringBuilder builder = new StringBuilder();
    builder
        .append("Time: ")
        .append(summary.date)
        .append(", ")
        .append(summary.duration)
        .append(" elapsed\n");
    builder.append("Number of concurrent users: ").append(summary.nbUsers).append("\n");
    builder.append("Number of requests: ").append(summary.nbRequest).append("\n");
    builder.append("Number of requests per seconds: ").append(summary.requestsSeconds).append("\n");
    formatListMetrics(builder, summary.listMetric, 0);
    logger.info(builder.toString());
  }

  private void formatListMetrics(
      StringBuilder builder, List<MetricsSummary.ChildMetric> listMetric, int level) {
    String padding = pad(2 * level);
    for (MetricsSummary.ChildMetric metric : listMetric) {
      if (metric.children == null) {
        builder.append(padding).append("> Request ").append(metric.name).append("\n");
        builder.append(padding).append("   Counts: ").append(metric.nbRequest).append("\n");
        builder
            .append(padding)
            .append("   Requests per seconds: ")
            .append(metric.requestsSeconds)
            .append("\n");
        builder
            .append(padding)
            .append("   Failure ratio: ")
            .append(metric.failureRatio)
            .append("\n");
      } else {
        builder.append(padding).append("> Group ").append(metric.name).append("\n");
        formatListMetrics(builder, metric.children, level + 1);
      }
    }
  }

  private String pad(final int repeat) {
    if (repeat <= 0) {
      return "";
    }
    final char[] buf = new char[repeat];
    Arrays.fill(buf, ' ');
    return new String(buf);
  }

  private void setResult(SimulationEndResult result) {
    this.result = result;
    latch.countDown();
    this.cancel();
  }

  private void setException(Exception exception) {
    this.exception = exception;
    latch.countDown();
    this.cancel();
  }
}
