/*
 * Copyright 2011-2023 GatlingCorp (https://gatling.io)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.gatling.plugin.model;

import static io.gatling.plugin.util.ObjectsUtil.nonNullParam;

import java.util.List;
import java.util.Objects;

public final class MetricsSummary {
  public final String date;
  public final String duration;
  public final double nbUsers;
  public final double nbRequest;
  public final double requestsSeconds;
  public final double failureRatio;
  public final List<ChildMetric> listMetric;

  public MetricsSummary(
      String date,
      String duration,
      double nbUsers,
      double nbRequest,
      double requestsSeconds,
      double failureRatio,
      List<ChildMetric> listMetric) {
    nonNullParam(date, "date");
    nonNullParam(duration, "duration");
    nonNullParam(listMetric, "listMetric");
    this.date = date;
    this.duration = duration;
    this.nbUsers = nbUsers;
    this.nbRequest = nbRequest;
    this.requestsSeconds = requestsSeconds;
    this.failureRatio = failureRatio;
    this.listMetric = listMetric;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    MetricsSummary that = (MetricsSummary) o;
    return Objects.equals(date, that.date)
        && Objects.equals(duration, that.duration)
        && nbUsers == that.nbUsers
        && nbRequest == that.nbRequest
        && requestsSeconds == that.requestsSeconds
        && failureRatio == that.failureRatio
        && Objects.equals(listMetric, that.listMetric);
  }

  @Override
  public int hashCode() {
    return Objects.hash(
        date, duration, nbUsers, nbRequest, requestsSeconds, failureRatio, listMetric);
  }

  @Override
  public String toString() {
    return String.format(
        "MetricsSummary{date='%s',duration='%s',nbUsers='%s',nbRequest='%s',requestsSeconds='%s',failureRatio='%s',listMetric='%s'}",
        date, duration, nbUsers, nbRequest, requestsSeconds, failureRatio, listMetric);
  }

  public static final class ChildMetric {
    public final String name;
    public final Double nbRequest;
    public final Double failureRatio;
    public final Double requestsSeconds;
    public final List<ChildMetric> children;

    public ChildMetric(String name, List<ChildMetric> children) {
      nonNullParam(name, "name");
      nonNullParam(children, "children");
      this.name = name;
      this.nbRequest = null;
      this.failureRatio = null;
      this.requestsSeconds = null;
      this.children = children;
    }

    public ChildMetric(String name, Double nbRequest, Double failureRatio, Double requestsSeconds) {
      nonNullParam(name, "name");
      nonNullParam(nbRequest, "nbRequest");
      nonNullParam(failureRatio, "failureRatio");
      nonNullParam(requestsSeconds, "requestsSeconds");
      this.name = name;
      this.nbRequest = nbRequest;
      this.failureRatio = failureRatio;
      this.requestsSeconds = requestsSeconds;
      this.children = null;
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;
      ChildMetric that = (ChildMetric) o;
      return Objects.equals(name, that.name)
          && Objects.equals(nbRequest, that.nbRequest)
          && Objects.equals(failureRatio, that.failureRatio)
          && Objects.equals(requestsSeconds, that.requestsSeconds)
          && Objects.equals(children, that.children);
    }

    @Override
    public int hashCode() {
      return Objects.hash(name, nbRequest, failureRatio, requestsSeconds, children);
    }

    @Override
    public String toString() {
      return String.format(
          "ChildMetric{name='%s',nbRequest='%s',failureRatio='%s',requestsSeconds='%s',children='%s'}",
          name, nbRequest, failureRatio, requestsSeconds, children);
    }
  }
}
