/*
 * Copyright 2011-2023 GatlingCorp (https://gatling.io)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.gatling.plugin.model;

import static io.gatling.plugin.util.ObjectsUtil.nonNullParam;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

public final class RequestsSummary {
  public final In in;
  public final Out out;
  public final List<Child> children;

  @JsonCreator
  public RequestsSummary(
      @JsonProperty(value = "in", required = true) In in,
      @JsonProperty(value = "out", required = true) Out out,
      @JsonProperty(value = "children") List<Child> children) {
    nonNullParam(in, "in");
    nonNullParam(out, "out");
    this.in = in;
    this.out = out;
    this.children = children != null ? children : Collections.emptyList();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    RequestsSummary that = (RequestsSummary) o;
    return Objects.equals(in, that.in)
        && Objects.equals(out, that.out)
        && Objects.equals(children, that.children);
  }

  @Override
  public int hashCode() {
    return Objects.hash(in, out, children);
  }

  @Override
  public String toString() {
    return String.format("RequestsSummary{in='%s',out='%s',children='%s'}", in, out, children);
  }

  public static final class Child {
    public final String name;
    public final In in;
    public final Out out;
    public final Integer index;
    public final List<Child> children;

    @JsonCreator
    public Child(
        @JsonProperty(value = "name", required = true) String name,
        @JsonProperty(value = "in", required = true) In in,
        @JsonProperty(value = "out", required = true) Out out,
        @JsonProperty(value = "index") Integer index,
        @JsonProperty(value = "children") List<Child> children) {
      nonNullParam(name, "name");
      nonNullParam(in, "in");
      nonNullParam(out, "out");
      this.name = name;
      this.in = in;
      this.out = out;
      this.index = index;
      this.children = children;
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;
      Child that = (Child) o;
      return Objects.equals(name, that.name)
          && Objects.equals(in, that.in)
          && Objects.equals(out, that.out)
          && Objects.equals(index, that.index)
          && Objects.equals(children, that.children);
    }

    @Override
    public int hashCode() {
      return Objects.hash(name, in, out, index, children);
    }

    @Override
    public String toString() {
      return String.format(
          "Child{name='%s',in='%s',out='%s',index='%s',children='%s'}",
          name, in, out, index, children);
    }
  }

  public static final class In {
    public final KoPercent counts;
    public final Total rps;

    @JsonCreator
    public In(
        @JsonProperty(value = "counts", required = true) KoPercent counts,
        @JsonProperty(value = "rps", required = true) Total rps) {
      nonNullParam(counts, "counts");
      nonNullParam(rps, "rps");
      this.counts = counts;
      this.rps = rps;
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;
      In that = (In) o;
      return Objects.equals(counts, that.counts) && Objects.equals(rps, that.rps);
    }

    @Override
    public int hashCode() {
      return Objects.hash(counts, rps);
    }

    @Override
    public String toString() {
      return String.format("In{counts='%s',rps='%s'}", counts, rps);
    }
  }

  public static final class Out {
    public final Total counts;
    public final Total rps;

    @JsonCreator
    public Out(
        @JsonProperty(value = "counts", required = true) Total counts,
        @JsonProperty(value = "rps", required = true) Total rps) {
      nonNullParam(counts, "counts");
      nonNullParam(rps, "rps");
      this.counts = counts;
      this.rps = rps;
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;
      Out that = (Out) o;
      return Objects.equals(counts, that.counts) && Objects.equals(rps, that.rps);
    }

    @Override
    public int hashCode() {
      return Objects.hash(counts, rps);
    }

    @Override
    public String toString() {
      return String.format("Out{counts='%s',rps='%s'}", counts, rps);
    }
  }

  public static final class KoPercent {
    public final double koPercent;

    @JsonCreator
    public KoPercent(@JsonProperty(value = "koPercent", required = true) double koPercent) {
      this.koPercent = koPercent;
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;
      KoPercent that = (KoPercent) o;
      return Objects.equals(koPercent, that.koPercent);
    }

    @Override
    public int hashCode() {
      return Objects.hash(koPercent);
    }

    @Override
    public String toString() {
      return String.format("KoPercent{koPercent='%s'}", koPercent);
    }
  }

  public static final class Total {
    public final double total;

    @JsonCreator
    public Total(@JsonProperty(value = "total", required = true) double total) {
      this.total = total;
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;
      Total that = (Total) o;
      return Objects.equals(total, that.total);
    }

    @Override
    public int hashCode() {
      return Objects.hash(total);
    }

    @Override
    public String toString() {
      return String.format("Total{total='%s'}", total);
    }
  }
}
