/*
 * Copyright 2011-2023 GatlingCorp (https://gatling.io)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.gatling.plugin.util;

public final class DurationFormatter {

  private DurationFormatter() {}

  /**
   * Formats duration with a format like "2h 25m 12s".
   *
   * @param start timestamp in milliseconds
   * @param end timestamp in milliseconds
   */
  public static String formatDuration(long start, long end) {
    long totalSeconds = (end - start) / 1000;
    if (totalSeconds < 0) {
      throw new IllegalArgumentException("Cannot format negative duration");
    }

    final StringBuilder formattedStr = new StringBuilder();
    long remainder = totalSeconds;

    remainder = addFormattedValue(formattedStr, totalSeconds, remainder, 86400, "d ");
    remainder = addFormattedValue(formattedStr, totalSeconds, remainder, 3600, "h ");
    remainder = addFormattedValue(formattedStr, totalSeconds, remainder, 60, "m ");
    formattedStr.append(remainder).append("s");
    return formattedStr.toString();
  }

  /** @return The new remainder value */
  private static long addFormattedValue(
      StringBuilder formattedStr,
      long totalSeconds,
      long remainder,
      long lengthSeconds,
      String symbol) {
    if (totalSeconds >= lengthSeconds) {
      long value = remainder / lengthSeconds;
      formattedStr.append(value).append(symbol);
      return remainder % lengthSeconds;
    } else {
      return remainder;
    }
  }
}
