/*
 * Copyright 2011-2023 GatlingCorp (https://gatling.io)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.gatling.plugin.client.http;

import com.fasterxml.jackson.core.type.TypeReference;
import io.gatling.plugin.exceptions.EnterprisePluginException;
import io.gatling.plugin.exceptions.RunNotFoundException;
import io.gatling.plugin.model.RequestsSummary;
import io.gatling.plugin.model.RunInformation;
import io.gatling.plugin.model.Series;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.List;
import java.util.UUID;

class RunsApiRequests extends AbstractApiRequests {

  private static final ApiPath RUN_PATH = ApiPath.of("run");
  private static final ApiPath SERIES_PATH = ApiPath.of("series");
  private static final ApiPath SUMMARY_PATH = ApiPath.of("summaries", "requests");

  RunsApiRequests(URL baseUrl, String token) {
    super(baseUrl, token);
  }

  RunInformation getRunInformation(UUID runId) throws EnterprisePluginException {
    return getJson(
        RUN_PATH.addQueryParam("run", runId.toString()),
        RunInformation.class,
        response -> {
          if (response.code == HttpURLConnection.HTTP_NOT_FOUND) {
            throw new RunNotFoundException(runId);
          }
        });
  }

  public List<Series> getConcurrentUserMetric(UUID runId, String scenario)
      throws EnterprisePluginException {
    return getJson(
        SERIES_PATH
            .addQueryParam("run", runId.toString())
            .addQueryParam("scenario", scenario)
            .addQueryParam("group", "")
            .addQueryParam("request", "")
            .addQueryParam("node", "")
            .addQueryParam("remote", "")
            .addQueryParam("metric", "usrActive"),
        new TypeReference<List<Series>>() {});
  }

  public RequestsSummary getRequestsSummary(UUID runId) throws EnterprisePluginException {
    return getJson(SUMMARY_PATH.addQueryParam("run", runId.toString()), RequestsSummary.class);
  }
}
