/*
 * Copyright 2011-2023 GatlingCorp (https://gatling.io)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.gatling.plugin.client.http;

import io.gatling.plugin.client.EnterpriseClient;
import io.gatling.plugin.exceptions.ApiCallIOException;
import io.gatling.plugin.exceptions.EnterprisePluginException;
import io.gatling.plugin.exceptions.InvalidBaseUrlException;
import io.gatling.plugin.exceptions.PackageNotFoundException;
import io.gatling.plugin.model.*;
import io.gatling.plugin.util.checksum.PkgChecksum;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.*;

public final class HttpEnterpriseClient implements EnterpriseClient {

  private static final Map<String, String> DEFAULT_SYSTEM_PROPERTIES = Collections.emptyMap();
  private static final Map<String, String> DEFAULT_ENVIRONMENT_VARIABLES = Collections.emptyMap();
  private static final MeaningfulTimeWindow DEFAULT_TIME_WINDOW = new MeaningfulTimeWindow(0, 0);

  private final URL baseUrl;
  private final InfoApiRequests infoApiRequests;
  private final PackagesApiRequests packagesApiRequests;
  private final PrivatePackagesApiRequests privatePackagesApiRequests;
  private final LocationsApiRequests locationsApiRequests;
  private final SimulationsApiRequests simulationsApiRequests;
  private final TeamsApiRequests teamsApiRequests;
  private final RunsApiRequests runsApiRequests;

  /**
   * @param baseUrl Base URL for the Gatling Enterprise server, e.g. {@code
   *     https://cloud.gatling.io}
   * @param token Authentication token used to access the public API
   * @param client Name of the calling client, used to verify if it is supported by the API
   * @param version Version of the calling client, used to verify if it is supported by the API
   * @param privateControlPlaneUrl (optional) Base URL for a Gatling Enterprise private control
   *     plane providing a private repository. If this parameter is provided, packages will be
   *     registered as private packages and uploaded through this private control plane.
   */
  public HttpEnterpriseClient(
      URL baseUrl, String token, String client, String version, URL privateControlPlaneUrl)
      throws EnterprisePluginException {
    if (!"http".equals(baseUrl.getProtocol()) && !"https".equals(baseUrl.getProtocol())) {
      throw new InvalidBaseUrlException(baseUrl);
    }
    this.baseUrl = baseUrl;

    final URL publicApiBaseUrl = ApiPath.of("api", "public").buildUrl(baseUrl);
    infoApiRequests = new InfoApiRequests(publicApiBaseUrl, token);
    packagesApiRequests = new PackagesApiRequests(publicApiBaseUrl, token);
    privatePackagesApiRequests =
        Optional.ofNullable(privateControlPlaneUrl)
            .map(url -> new PrivatePackagesApiRequests(url, token))
            .orElse(null);
    locationsApiRequests = new LocationsApiRequests(publicApiBaseUrl, token);
    simulationsApiRequests = new SimulationsApiRequests(publicApiBaseUrl, token);
    teamsApiRequests = new TeamsApiRequests(publicApiBaseUrl, token);
    runsApiRequests = new RunsApiRequests(publicApiBaseUrl, token);

    new PrivateApiRequests(publicApiBaseUrl, token).checkVersionSupport(client, version);
  }

  @Override
  public URL getBaseUrl() {
    return baseUrl;
  }

  @Override
  public ServerInformation getServerInformation() throws EnterprisePluginException {
    return infoApiRequests.getServerInformation();
  }

  @Override
  public List<Simulation> getSimulations() throws EnterprisePluginException {
    return simulationsApiRequests.listSimulations().data;
  }

  @Override
  public Simulation getSimulation(UUID simulationId) throws EnterprisePluginException {
    return simulationsApiRequests.getSimulation(simulationId);
  }

  @Override
  public List<Team> getTeams() throws EnterprisePluginException {
    return teamsApiRequests.listTeams().data;
  }

  @Override
  public Locations getLocations() throws EnterprisePluginException {
    return new Locations(
        locationsApiRequests.listPublicLocations().data,
        locationsApiRequests.listPrivateLocations().data);
  }

  @Override
  public List<PkgIndex> getPackages() throws EnterprisePluginException {
    return packagesApiRequests.listPackages().data;
  }

  @Override
  public Pkg getPackage(UUID pkgId) throws EnterprisePluginException {
    return packagesApiRequests.readPackage(pkgId);
  }

  @Override
  public long uploadPackage(UUID packageId, File file) throws EnterprisePluginException {
    return this.privatePackagesApiRequests != null
        ? this.privatePackagesApiRequests.uploadPackage(packageId, file)
        : packagesApiRequests.uploadPackage(packageId, file);
  }

  @Override
  public RunSummary startSimulation(
      UUID simulationId,
      Map<String, String> systemProperties,
      Map<String, String> environmentVariables)
      throws EnterprisePluginException {

    final StartOptions options = new StartOptions(systemProperties, environmentVariables);

    return simulationsApiRequests.startSimulation(simulationId, options);
  }

  private boolean checksumComparison(UUID packageId, File file) throws EnterprisePluginException {
    try {
      Pkg pkg = getPackage(packageId);
      return pkg.file != null && PkgChecksum.computeChecksum(file).equals(pkg.file.checksum);
    } catch (PackageNotFoundException e) {
      return false;
    } catch (IOException e) {
      throw new ApiCallIOException(e);
    }
  }

  @Override
  public long uploadPackageWithChecksum(UUID packageId, File file)
      throws EnterprisePluginException {
    return checksumComparison(packageId, file) ? -1 : uploadPackage(packageId, file);
  }

  @Override
  public SimulationClassName updateSimulationClassName(UUID simulationId, String className)
      throws EnterprisePluginException {
    return simulationsApiRequests.updateSimulationClassName(simulationId, className);
  }

  @Override
  public Simulation createSimulation(
      String simulationName,
      UUID teamId,
      String className,
      UUID pkgId,
      Map<String, HostByLocation> hostsByLocation)
      throws EnterprisePluginException {
    return simulationsApiRequests.createSimulation(
        new SimulationCreationPayload(
            simulationName,
            teamId,
            className,
            pkgId,
            DEFAULT_SYSTEM_PROPERTIES,
            DEFAULT_ENVIRONMENT_VARIABLES,
            /* ignoreGlobalProperties */ false,
            DEFAULT_TIME_WINDOW,
            hostsByLocation,
            /* usePoolWeights */ false,
            /* usePoolDedicatedIps */ false));
  }

  @Override
  public Pkg createPackage(String packageName, UUID teamId) throws EnterprisePluginException {
    final PackageStorageType storageType =
        this.privatePackagesApiRequests != null
            ? PackageStorageType.PRIVATE
            : PackageStorageType.PUBLIC;

    return packagesApiRequests.createPackage(
        new PackageCreationPayload(packageName, teamId, storageType));
  }

  @Override
  public RunInformation getRunInformation(UUID runId) throws EnterprisePluginException {
    return runsApiRequests.getRunInformation(runId);
  }

  @Override
  public List<Series> getConcurrentUserMetric(UUID runId, String scenario)
      throws EnterprisePluginException {
    return runsApiRequests.getConcurrentUserMetric(runId, scenario);
  }

  @Override
  public RequestsSummary getRequestsSummary(UUID runId) throws EnterprisePluginException {
    return runsApiRequests.getRequestsSummary(runId);
  }

  @Override
  public boolean abortRun(UUID runId) throws EnterprisePluginException {
    return simulationsApiRequests.abortRun(runId);
  }
}
