package io.gatling.recorder.internal.bouncycastle.pqc.jcajce.provider.dilithium;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

import io.gatling.recorder.internal.bouncycastle.asn1.ASN1Set;
import io.gatling.recorder.internal.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import io.gatling.recorder.internal.bouncycastle.pqc.crypto.crystals.dilithium.DilithiumPrivateKeyParameters;
import io.gatling.recorder.internal.bouncycastle.pqc.crypto.crystals.dilithium.DilithiumPublicKeyParameters;
import io.gatling.recorder.internal.bouncycastle.pqc.crypto.util.PrivateKeyFactory;
import io.gatling.recorder.internal.bouncycastle.pqc.crypto.util.PrivateKeyInfoFactory;
import io.gatling.recorder.internal.bouncycastle.pqc.jcajce.interfaces.DilithiumPrivateKey;
import io.gatling.recorder.internal.bouncycastle.pqc.jcajce.interfaces.DilithiumPublicKey;
import io.gatling.recorder.internal.bouncycastle.pqc.jcajce.spec.DilithiumParameterSpec;
import io.gatling.recorder.internal.bouncycastle.util.Arrays;

public class BCDilithiumPrivateKey
    implements DilithiumPrivateKey
{
    private static final long serialVersionUID = 1L;

    private transient DilithiumPrivateKeyParameters params;
    private transient ASN1Set attributes;

    public BCDilithiumPrivateKey(
            DilithiumPrivateKeyParameters params)
    {
        this.params = params;
    }

    public BCDilithiumPrivateKey(PrivateKeyInfo keyInfo)
            throws IOException
    {
        init(keyInfo);
    }

    private void init(PrivateKeyInfo keyInfo)
            throws IOException
    {
        this.attributes = keyInfo.getAttributes();
        this.params = (DilithiumPrivateKeyParameters) PrivateKeyFactory.createKey(keyInfo);
    }

    /**
     * Compare this SPHINCS-256 private key with another object.
     *
     * @param o the other object
     * @return the result of the comparison
     */
    public boolean equals(Object o)
    {
        if (o == this)
        {
            return true;
        }

        if (o instanceof BCDilithiumPrivateKey)
        {
            BCDilithiumPrivateKey otherKey = (BCDilithiumPrivateKey)o;

            return Arrays.areEqual(params.getEncoded(), otherKey.params.getEncoded());
        }

        return false;
    }

    public int hashCode()
    {
        return Arrays.hashCode(params.getEncoded());
    }

    /**
     * @return name of the algorithm - "Dilithium"
     */
    public final String getAlgorithm()
    {
        return "Dilithium";
    }

    public DilithiumPublicKey getPublicKey()
    {
        return new BCDilithiumPublicKey(new DilithiumPublicKeyParameters(params.getParameters(), params.getRho(), params.getT1()));
    }

    public byte[] getEncoded()
    {
        try
        {
            PrivateKeyInfo pki = PrivateKeyInfoFactory.createPrivateKeyInfo(params, attributes);

            return pki.getEncoded();
        }
        catch (IOException e)
        {
            return null;
        }
    }

    public DilithiumParameterSpec getParameterSpec()
    {
        return DilithiumParameterSpec.fromName(params.getParameters().getName());
    }

    public String getFormat()
    {
        return "PKCS#8";
    }

    DilithiumPrivateKeyParameters getKeyParams()
    {
        return params;
    }

    private void readObject(
            ObjectInputStream in)
            throws IOException, ClassNotFoundException
    {
        in.defaultReadObject();

        byte[] enc = (byte[])in.readObject();

        init(PrivateKeyInfo.getInstance(enc));
    }

    private void writeObject(
            ObjectOutputStream out)
            throws IOException
    {
        out.defaultWriteObject();

        out.writeObject(this.getEncoded());
    }
}
