package io.gatling.recorder.internal.bouncycastle.pqc.jcajce.provider.sike;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.security.PublicKey;

import io.gatling.recorder.internal.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import io.gatling.recorder.internal.bouncycastle.pqc.crypto.sike.SIKEPublicKeyParameters;
import io.gatling.recorder.internal.bouncycastle.pqc.crypto.util.PublicKeyFactory;
import io.gatling.recorder.internal.bouncycastle.pqc.crypto.util.SubjectPublicKeyInfoFactory;
import io.gatling.recorder.internal.bouncycastle.pqc.jcajce.interfaces.SIKEKey;
import io.gatling.recorder.internal.bouncycastle.pqc.jcajce.spec.SIKEParameterSpec;
import io.gatling.recorder.internal.bouncycastle.util.Arrays;

public class BCSIKEPublicKey
    implements PublicKey, SIKEKey
{
    private static final long serialVersionUID = 1L;

    private transient SIKEPublicKeyParameters params;

    public BCSIKEPublicKey(
        SIKEPublicKeyParameters params)
    {
        this.params = params;
    }

    public BCSIKEPublicKey(SubjectPublicKeyInfo keyInfo)
        throws IOException
    {
        init(keyInfo);
    }

    private void init(SubjectPublicKeyInfo keyInfo)
        throws IOException
    {
        this.params = (SIKEPublicKeyParameters) PublicKeyFactory.createKey(keyInfo);
    }

    /**
     * Compare this SIKE public key with another object.
     *
     * @param o the other object
     * @return the result of the comparison
     */
    public boolean equals(Object o)
    {
        if (o == this)
        {
            return true;
        }

        if (o instanceof BCSIKEPublicKey)
        {
            BCSIKEPublicKey otherKey = (BCSIKEPublicKey)o;

            return Arrays.areEqual(params.getEncoded(), otherKey.params.getEncoded());
        }

        return false;
    }

    public int hashCode()
    {
        return Arrays.hashCode(params.getEncoded());
    }

    /**
     * @return name of the algorithm - "SIKE"
     */
    public final String getAlgorithm()
    {
        return "SIKE";
    }

    public byte[] getEncoded()
    {
        try
        {
            SubjectPublicKeyInfo pki = SubjectPublicKeyInfoFactory.createSubjectPublicKeyInfo(params);

            return pki.getEncoded();
        }
        catch (IOException e)
        {
            return null;
        }
    }

    public String getFormat()
    {
        return "X.509";
    }

    public SIKEParameterSpec getParameterSpec()
    {
        return SIKEParameterSpec.fromName(params.getParameters().getName());
    }

    SIKEPublicKeyParameters getKeyParams()
    {
        return params;
    }

    private void readObject(
        ObjectInputStream in)
        throws IOException, ClassNotFoundException
    {
        in.defaultReadObject();

        byte[] enc = (byte[])in.readObject();

        init(SubjectPublicKeyInfo.getInstance(enc));
    }

    private void writeObject(
        ObjectOutputStream out)
        throws IOException
    {
        out.defaultWriteObject();

        out.writeObject(this.getEncoded());
    }
}
