package io.gatling.recorder.internal.bouncycastle.asn1.x9;

import java.math.BigInteger;

import io.gatling.recorder.internal.bouncycastle.asn1.ASN1BitString;
import io.gatling.recorder.internal.bouncycastle.asn1.ASN1Integer;
import io.gatling.recorder.internal.bouncycastle.asn1.ASN1Object;
import io.gatling.recorder.internal.bouncycastle.asn1.ASN1Primitive;
import io.gatling.recorder.internal.bouncycastle.asn1.ASN1Sequence;
import io.gatling.recorder.internal.bouncycastle.asn1.ASN1TaggedObject;
import io.gatling.recorder.internal.bouncycastle.asn1.DERBitString;
import io.gatling.recorder.internal.bouncycastle.asn1.DERSequence;

/**
 * Diffie-Hellman domain validation parameters.
 * <pre>
 * ValidationParams ::= SEQUENCE {
 *    seed         BIT STRING,
 *    pgenCounter  INTEGER
 * }
 * </pre>
 */
public class ValidationParams
    extends ASN1Object
{
    private ASN1BitString seed;
    private ASN1Integer pgenCounter;

    public static ValidationParams getInstance(ASN1TaggedObject obj, boolean explicit)
    {
        return getInstance(ASN1Sequence.getInstance(obj, explicit));
    }

    public static ValidationParams getInstance(Object obj)
    {
        if (obj instanceof ValidationParams)
        {
            return (ValidationParams)obj;
        }
        else if (obj != null)
        {
            return new ValidationParams(ASN1Sequence.getInstance(obj));
        }

        return null;
    }

    public ValidationParams(byte[] seed, int pgenCounter)
    {
        if (seed == null)
        {
            throw new IllegalArgumentException("'seed' cannot be null");
        }

        this.seed = new DERBitString(seed);
        this.pgenCounter = new ASN1Integer(pgenCounter);
    }

    public ValidationParams(DERBitString seed, ASN1Integer pgenCounter)
    {
        if (seed == null)
        {
            throw new IllegalArgumentException("'seed' cannot be null");
        }
        if (pgenCounter == null)
        {
            throw new IllegalArgumentException("'pgenCounter' cannot be null");
        }

        this.seed = seed;
        this.pgenCounter = pgenCounter;
    }

    private ValidationParams(ASN1Sequence seq)
    {
        if (seq.size() != 2)
        {
            throw new IllegalArgumentException("Bad sequence size: " + seq.size());
        }

        this.seed = ASN1BitString.getInstance(seq.getObjectAt(0));
        this.pgenCounter = ASN1Integer.getInstance(seq.getObjectAt(1));
    }

    public byte[] getSeed()
    {
        return this.seed.getBytes();
    }

    public BigInteger getPgenCounter()
    {
        return this.pgenCounter.getPositiveValue();
    }

    public ASN1Primitive toASN1Primitive()
    {
        return new DERSequence(seed, pgenCounter);
    }
}
