package io.gatling.recorder.internal.bouncycastle.crypto.generators;

import java.math.BigInteger;
import java.security.SecureRandom;

import io.gatling.recorder.internal.bouncycastle.crypto.AsymmetricCipherKeyPair;
import io.gatling.recorder.internal.bouncycastle.crypto.AsymmetricCipherKeyPairGenerator;
import io.gatling.recorder.internal.bouncycastle.crypto.CryptoServicePurpose;
import io.gatling.recorder.internal.bouncycastle.crypto.CryptoServicesRegistrar;
import io.gatling.recorder.internal.bouncycastle.crypto.KeyGenerationParameters;
import io.gatling.recorder.internal.bouncycastle.crypto.constraints.ConstraintUtils;
import io.gatling.recorder.internal.bouncycastle.crypto.constraints.DefaultServiceProperties;
import io.gatling.recorder.internal.bouncycastle.crypto.params.ECDomainParameters;
import io.gatling.recorder.internal.bouncycastle.crypto.params.ECKeyGenerationParameters;
import io.gatling.recorder.internal.bouncycastle.crypto.params.ECPrivateKeyParameters;
import io.gatling.recorder.internal.bouncycastle.crypto.params.ECPublicKeyParameters;
import io.gatling.recorder.internal.bouncycastle.math.ec.ECConstants;
import io.gatling.recorder.internal.bouncycastle.math.ec.ECMultiplier;
import io.gatling.recorder.internal.bouncycastle.math.ec.ECPoint;
import io.gatling.recorder.internal.bouncycastle.math.ec.FixedPointCombMultiplier;
import io.gatling.recorder.internal.bouncycastle.math.ec.WNafUtil;
import io.gatling.recorder.internal.bouncycastle.util.BigIntegers;

public class ECKeyPairGenerator
    implements AsymmetricCipherKeyPairGenerator, ECConstants
{
    private final String name;
    ECDomainParameters  params;
    SecureRandom        random;

    public ECKeyPairGenerator()
    {
        this("ECKeyGen");
    }

    protected ECKeyPairGenerator(String name)
    {
        this.name = name;
    }

    public void init(
        KeyGenerationParameters param)
    {
        ECKeyGenerationParameters  ecP = (ECKeyGenerationParameters)param;

        this.random = ecP.getRandom();
        this.params = ecP.getDomainParameters();

        CryptoServicesRegistrar.checkConstraints(new DefaultServiceProperties(name, ConstraintUtils.bitsOfSecurityFor(this.params.getCurve()), ecP.getDomainParameters(), CryptoServicePurpose.KEYGEN));
    }

    /**
     * Given the domain parameters this routine generates an EC key
     * pair in accordance with X9.62 section 5.2.1 pages 26, 27.
     */
    public AsymmetricCipherKeyPair generateKeyPair()
    {
        BigInteger n = params.getN();
        int nBitLength = n.bitLength();
        int minWeight = nBitLength >>> 2;

        BigInteger d;
        for (;;)
        {
            d = BigIntegers.createRandomBigInteger(nBitLength, random);

            if (isOutOfRangeD(d, n))
            {
                continue;
            }

            if (WNafUtil.getNafWeight(d) < minWeight)
            {
                continue;
            }

            break;
        }

        ECPoint Q = createBasePointMultiplier().multiply(params.getG(), d);

        return new AsymmetricCipherKeyPair(
            new ECPublicKeyParameters(Q, params),
            new ECPrivateKeyParameters(d, params));
    }

    protected boolean isOutOfRangeD(BigInteger d, BigInteger n)
    {
        return d.compareTo(ONE) < 0 || (d.compareTo(n) >= 0);
    }

    protected ECMultiplier createBasePointMultiplier()
    {
        return new FixedPointCombMultiplier();
    }
}
