package io.gatling.recorder.internal.bouncycastle.pqc.jcajce.provider.mayo;

import java.io.ByteArrayOutputStream;
import java.security.InvalidKeyException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.SignatureException;
import java.security.spec.AlgorithmParameterSpec;

import io.gatling.recorder.internal.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import io.gatling.recorder.internal.bouncycastle.crypto.CipherParameters;
import io.gatling.recorder.internal.bouncycastle.crypto.params.ParametersWithRandom;
import io.gatling.recorder.internal.bouncycastle.pqc.crypto.mayo.MayoParameters;
import io.gatling.recorder.internal.bouncycastle.pqc.crypto.mayo.MayoSigner;
import io.gatling.recorder.internal.bouncycastle.util.Strings;

public class SignatureSpi
    extends java.security.Signature
{
    private final ByteArrayOutputStream bOut;
    private final MayoSigner signer;
    private SecureRandom random;
    private final MayoParameters parameters;

    protected SignatureSpi(MayoSigner signer)
    {
        super("Mayo");

        this.bOut = new ByteArrayOutputStream();
        this.signer = signer;
        this.parameters = null;
    }

    protected SignatureSpi(MayoSigner signer, MayoParameters parameters)
    {
        super(Strings.toUpperCase(parameters.getName()));
        this.parameters = parameters;

        this.bOut = new ByteArrayOutputStream();
        this.signer = signer;
    }

    protected void engineInitVerify(PublicKey publicKey)
        throws InvalidKeyException
    {
        if (!(publicKey instanceof BCMayoPublicKey))
        {
            try
            {
                publicKey = new BCMayoPublicKey(SubjectPublicKeyInfo.getInstance(publicKey.getEncoded()));
            }
            catch (Exception e)
            {
                throw new InvalidKeyException("unknown public key passed to Mayo: " + e.getMessage());
            }
        }

        BCMayoPublicKey key = (BCMayoPublicKey)publicKey;

        if (parameters != null)
        {
            String canonicalAlg = Strings.toUpperCase(parameters.getName());
            if (!canonicalAlg.equals(key.getAlgorithm()))
            {
                throw new InvalidKeyException("signature configured for " + canonicalAlg);
            }
        }

        signer.init(false, key.getKeyParams());
    }

    protected void engineInitSign(PrivateKey privateKey, SecureRandom random)
        throws InvalidKeyException
    {
        this.random = random;
        engineInitSign(privateKey);
    }

    protected void engineInitSign(PrivateKey privateKey)
        throws InvalidKeyException
    {
        if (privateKey instanceof BCMayoPrivateKey)
        {
            BCMayoPrivateKey key = (BCMayoPrivateKey)privateKey;
            CipherParameters param = key.getKeyParams();

            if (parameters != null)
            {
                String canonicalAlg = Strings.toUpperCase(parameters.getName());
                if (!canonicalAlg.equals(key.getAlgorithm()))
                {
                    throw new InvalidKeyException("signature configured for " + canonicalAlg);
                }
            }

            if (random != null)
            {
                signer.init(true, new ParametersWithRandom(param, random));
            }
            else
            {
                signer.init(true, param);
            }
        }
        else
        {
            throw new InvalidKeyException("unknown private key passed to Mayo");
        }
    }

    protected void engineUpdate(byte b)
        throws SignatureException
    {
        bOut.write(b);
    }

    protected void engineUpdate(byte[] b, int off, int len)
        throws SignatureException
    {
        bOut.write(b, off, len);
    }

    protected byte[] engineSign()
        throws SignatureException
    {
        try
        {
            byte[] message = bOut.toByteArray();

            bOut.reset();

            return signer.generateSignature(message);
        }
        catch (Exception e)
        {
            throw new SignatureException(e.toString());
        }
    }

    protected boolean engineVerify(byte[] sigBytes)
        throws SignatureException
    {
        byte[] message = bOut.toByteArray();

        bOut.reset();

        return signer.verifySignature(message, sigBytes);
    }

    protected void engineSetParameter(AlgorithmParameterSpec params)
    {
        // TODO
        throw new UnsupportedOperationException("engineSetParameter unsupported");
    }

    /**
     * @deprecated replaced with #engineSetParameter(java.security.spec.AlgorithmParameterSpec)
     */
    protected void engineSetParameter(String param, Object value)
    {
        throw new UnsupportedOperationException("engineSetParameter unsupported");
    }

    /**
     * @deprecated
     */
    protected Object engineGetParameter(String param)
    {
        throw new UnsupportedOperationException("engineSetParameter unsupported");
    }

    public static class Base
        extends io.gatling.recorder.internal.bouncycastle.pqc.jcajce.provider.mayo.SignatureSpi
    {
        public Base()
        {
            super(new MayoSigner());
        }
    }

    public static class Mayo1
        extends io.gatling.recorder.internal.bouncycastle.pqc.jcajce.provider.mayo.SignatureSpi
    {
        public Mayo1()
        {
            super(new MayoSigner(), MayoParameters.mayo1);
        }
    }

    public static class Mayo2
        extends io.gatling.recorder.internal.bouncycastle.pqc.jcajce.provider.mayo.SignatureSpi
    {
        public Mayo2()
        {
            super(new MayoSigner(), MayoParameters.mayo2);
        }
    }

    public static class Mayo3
        extends io.gatling.recorder.internal.bouncycastle.pqc.jcajce.provider.mayo.SignatureSpi
    {
        public Mayo3()
        {
            super(new MayoSigner(), MayoParameters.mayo3);
        }
    }

    public static class Mayo5
        extends io.gatling.recorder.internal.bouncycastle.pqc.jcajce.provider.mayo.SignatureSpi
    {
        public Mayo5()
        {
            super(new MayoSigner(), MayoParameters.mayo5);
        }
    }
}

