package io.gatling.recorder.internal.bouncycastle.pqc.jcajce.provider.snova;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.security.PrivateKey;

import io.gatling.recorder.internal.bouncycastle.asn1.ASN1Set;
import io.gatling.recorder.internal.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import io.gatling.recorder.internal.bouncycastle.pqc.crypto.snova.SnovaPrivateKeyParameters;
import io.gatling.recorder.internal.bouncycastle.pqc.crypto.util.PrivateKeyFactory;
import io.gatling.recorder.internal.bouncycastle.pqc.crypto.util.PrivateKeyInfoFactory;
import io.gatling.recorder.internal.bouncycastle.pqc.jcajce.interfaces.SnovaKey;
import io.gatling.recorder.internal.bouncycastle.pqc.jcajce.spec.SnovaParameterSpec;
import io.gatling.recorder.internal.bouncycastle.util.Arrays;
import io.gatling.recorder.internal.bouncycastle.util.Strings;

public class BCSnovaPrivateKey
    implements PrivateKey, SnovaKey
{
    private static final long serialVersionUID = 1L;

    private transient SnovaPrivateKeyParameters params;
    private transient ASN1Set attributes;

    public BCSnovaPrivateKey(
        SnovaPrivateKeyParameters params)
    {
        this.params = params;
    }

    public BCSnovaPrivateKey(PrivateKeyInfo keyInfo)
        throws IOException
    {
        init(keyInfo);
    }

    private void init(PrivateKeyInfo keyInfo)
        throws IOException
    {
        this.attributes = keyInfo.getAttributes();
        this.params = (SnovaPrivateKeyParameters) PrivateKeyFactory.createKey(keyInfo);
    }

    /**
     * Compare this private key with another object.
     *
     * @param o the other object
     * @return the result of the comparison
     */
    public boolean equals(Object o)
    {
        if (o == this)
        {
            return true;
        }

        if (o instanceof BCSnovaPrivateKey)
        {
            BCSnovaPrivateKey otherKey = (BCSnovaPrivateKey)o;

            return Arrays.areEqual(params.getEncoded(), otherKey.params.getEncoded());
        }

        return false;
    }

    public int hashCode()
    {
        return Arrays.hashCode(params.getEncoded());
    }

    /**
     * @return name of the algorithm - "Snova_[v]_[o]_[l]"
     */
    public final String getAlgorithm()
    {
        return Strings.toUpperCase(params.getParameters().getName());
    }

    public byte[] getEncoded()
    {

        try
        {
            PrivateKeyInfo pki = PrivateKeyInfoFactory.createPrivateKeyInfo(params, attributes);

            return pki.getEncoded();
        }
        catch (IOException e)
        {
            return null;
        }
    }

    public SnovaParameterSpec getParameterSpec()
    {
        return SnovaParameterSpec.fromName(params.getParameters().getName());
    }

    public String getFormat()
    {
        return "PKCS#8";
    }

    SnovaPrivateKeyParameters getKeyParams()
    {
        return params;
    }

    private void readObject(
        ObjectInputStream in)
        throws IOException, ClassNotFoundException
    {
        in.defaultReadObject();

        byte[] enc = (byte[])in.readObject();

        init(PrivateKeyInfo.getInstance(enc));
    }

    private void writeObject(
        ObjectOutputStream out)
        throws IOException
    {
        out.defaultWriteObject();

        out.writeObject(this.getEncoded());
    }
}


