package io.georocket.util;

import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.Locale;

/**
 * <p>Convert data sizes to human-readable strings. Used by all commands that
 * output sizes, so the output always looks the same.</p>
 * <p>Code has been adapted from
 * <a href="https://stackoverflow.com/questions/3263892/format-file-size-as-mb-gb-etc">https://stackoverflow.com/questions/3263892/format-file-size-as-mb-gb-etc</a>
 * and corrected in a way that it displays correct SI units.</p>
 * @author Michel Kraemer
 */
public class SizeFormat {
  private static final String[] UNITS = new String[] {
    "B", "kB", "MB", "GB", "TB", "PB", "EB"
  };
  private static final DecimalFormat FORMATTER = new DecimalFormat("#,##0.#",
      DecimalFormatSymbols.getInstance(Locale.ENGLISH));

  /**
   * Convert the given data size to a human-readable string
   * @param size the data size
   * @return the human-readable string
   */
  public static String format(long size) {
    if (size <= 0) {
      return "0 B";
    }
    int d = (int)(Math.log10(size) / 3);
    return FORMATTER.format(size / Math.pow(1000, d)) + " " + UNITS[d];
  }
}
