/*
 * Copyright 2018 stfalcon.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.getstream.photoview.dialog.viewer.view

import android.transition.AutoTransition
import android.transition.Transition
import android.transition.TransitionManager
import android.view.View
import android.view.ViewGroup
import android.view.animation.DecelerateInterpolator
import android.widget.FrameLayout
import android.widget.ImageView
import io.getstream.photoview.dialog.common.extensions.addListener
import io.getstream.photoview.dialog.common.extensions.applyMargin
import io.getstream.photoview.dialog.common.extensions.globalVisibleRect
import io.getstream.photoview.dialog.common.extensions.isRectVisible
import io.getstream.photoview.dialog.common.extensions.localVisibleRect
import io.getstream.photoview.dialog.common.extensions.makeViewMatchParent
import io.getstream.photoview.dialog.common.extensions.postApply
import io.getstream.photoview.dialog.common.extensions.postDelayed
import io.getstream.photoview.dialog.common.extensions.requestNewSize

internal class TransitionImageAnimator(
  private val externalImage: ImageView?,
  private val internalImage: ImageView,
  private val internalImageContainer: FrameLayout
) {

  companion object {
    private const val TRANSITION_DURATION_OPEN = 200L
    private const val TRANSITION_DURATION_CLOSE = 250L
  }

  internal var isAnimating = false

  private var isClosing = false

  private val transitionDuration: Long
    get() = if (isClosing) TRANSITION_DURATION_CLOSE else TRANSITION_DURATION_OPEN

  private val internalRoot: ViewGroup
    get() = internalImageContainer.parent as ViewGroup

  internal fun animateOpen(
    containerPadding: IntArray,
    onTransitionStart: (Long) -> Unit,
    onTransitionEnd: () -> Unit
  ) {
    if (externalImage.isRectVisible) {
      onTransitionStart(TRANSITION_DURATION_OPEN)
      doOpenTransition(containerPadding, onTransitionEnd)
    } else {
      onTransitionEnd()
    }
  }

  internal fun animateClose(
    shouldDismissToBottom: Boolean,
    onTransitionStart: (Long) -> Unit,
    onTransitionEnd: () -> Unit
  ) {
    if (externalImage.isRectVisible && !shouldDismissToBottom) {
      onTransitionStart(TRANSITION_DURATION_CLOSE)
      doCloseTransition(onTransitionEnd)
    } else {
      externalImage?.visibility = View.VISIBLE
      onTransitionEnd()
    }
  }

  private fun doOpenTransition(containerPadding: IntArray, onTransitionEnd: () -> Unit) {
    isAnimating = true
    prepareTransitionLayout()

    internalRoot.postApply {
      //ain't nothing but a kludge to prevent blinking when transition is starting
      externalImage?.postDelayed(50) { visibility = View.INVISIBLE }

      TransitionManager.beginDelayedTransition(internalRoot, createTransition {
        if (!isClosing) {
          isAnimating = false
          onTransitionEnd()
        }
      })

      internalImageContainer.makeViewMatchParent()
      internalImage.makeViewMatchParent()

      internalRoot.applyMargin(
        containerPadding[0],
        containerPadding[1],
        containerPadding[2],
        containerPadding[3]
      )

      internalImageContainer.requestLayout()
    }
  }

  private fun doCloseTransition(onTransitionEnd: () -> Unit) {
    isAnimating = true
    isClosing = true

    TransitionManager.beginDelayedTransition(
      internalRoot, createTransition { handleCloseTransitionEnd(onTransitionEnd) })

    prepareTransitionLayout()
    internalImageContainer.requestLayout()
  }

  private fun prepareTransitionLayout() {
    externalImage?.let {
      if (externalImage.isRectVisible) {
        with(externalImage.localVisibleRect) {
          internalImage.requestNewSize(it.width, it.height)
          internalImage.applyMargin(top = -top, start = -left)
        }
        with(externalImage.globalVisibleRect) {
          internalImageContainer.requestNewSize(width(), height())
          internalImageContainer.applyMargin(left, top, right, bottom)
        }
      }

      resetRootTranslation()
    }
  }

  private fun handleCloseTransitionEnd(onTransitionEnd: () -> Unit) {
    externalImage?.visibility = View.VISIBLE
    internalImage.post { onTransitionEnd() }
    isAnimating = false
  }

  private fun resetRootTranslation() {
    internalRoot
      .animate()
      .translationY(0f)
      .setDuration(transitionDuration)
      .start()
  }

  private fun createTransition(onTransitionEnd: (() -> Unit)? = null): Transition =
    AutoTransition()
      .setDuration(transitionDuration)
      .setInterpolator(DecelerateInterpolator())
      .addListener(onTransitionEnd = { onTransitionEnd?.invoke() })
}