package io.github.alexzhirkevich.keight.js.interpreter

import io.github.alexzhirkevich.keight.Constructor
import io.github.alexzhirkevich.keight.Delegate
import io.github.alexzhirkevich.keight.Expression
import io.github.alexzhirkevich.keight.ScriptRuntime
import io.github.alexzhirkevich.keight.VariableType
import io.github.alexzhirkevich.keight.expressions.OpAssign
import io.github.alexzhirkevich.keight.expressions.OpAssignByIndex
import io.github.alexzhirkevich.keight.expressions.OpBlock
import io.github.alexzhirkevich.keight.expressions.OpBreak
import io.github.alexzhirkevich.keight.expressions.OpCall
import io.github.alexzhirkevich.keight.expressions.OpCase
import io.github.alexzhirkevich.keight.expressions.OpCompare
import io.github.alexzhirkevich.keight.expressions.OpConstant
import io.github.alexzhirkevich.keight.expressions.OpContinue
import io.github.alexzhirkevich.keight.expressions.OpDestructAssign
import io.github.alexzhirkevich.keight.expressions.OpDoWhileLoop
import io.github.alexzhirkevich.keight.expressions.OpEquals
import io.github.alexzhirkevich.keight.expressions.OpExp
import io.github.alexzhirkevich.keight.expressions.OpForInLoop
import io.github.alexzhirkevich.keight.expressions.OpForLoop
import io.github.alexzhirkevich.keight.expressions.OpGetProperty
import io.github.alexzhirkevich.keight.expressions.OpGetter
import io.github.alexzhirkevich.keight.expressions.OpIfCondition
import io.github.alexzhirkevich.keight.expressions.OpIn
import io.github.alexzhirkevich.keight.expressions.OpIncDecAssign
import io.github.alexzhirkevich.keight.expressions.OpIndex
import io.github.alexzhirkevich.keight.expressions.OpColonAssignment
import io.github.alexzhirkevich.keight.expressions.OpLongInt
import io.github.alexzhirkevich.keight.expressions.OpLongLong
import io.github.alexzhirkevich.keight.expressions.OpMake
import io.github.alexzhirkevich.keight.expressions.OpMakeArray
import io.github.alexzhirkevich.keight.expressions.OpMakeObject
import io.github.alexzhirkevich.keight.expressions.OpNot
import io.github.alexzhirkevich.keight.expressions.OpNotEquals
import io.github.alexzhirkevich.keight.expressions.OpReturn
import io.github.alexzhirkevich.keight.expressions.OpSetter
import io.github.alexzhirkevich.keight.expressions.OpSpread
import io.github.alexzhirkevich.keight.expressions.OpSwitch
import io.github.alexzhirkevich.keight.expressions.OpTouple
import io.github.alexzhirkevich.keight.expressions.OpTryCatch
import io.github.alexzhirkevich.keight.expressions.OpWhileLoop
import io.github.alexzhirkevich.keight.expressions.PropertyAccessorFactory
import io.github.alexzhirkevich.keight.expressions.ThrowableValue
import io.github.alexzhirkevich.keight.expressions.asDestruction
import io.github.alexzhirkevich.keight.fastAll
import io.github.alexzhirkevich.keight.fastMap
import io.github.alexzhirkevich.keight.findJsRoot
import io.github.alexzhirkevich.keight.js.JSError
import io.github.alexzhirkevich.keight.js.JSFunction
import io.github.alexzhirkevich.keight.js.JsAny
import io.github.alexzhirkevich.keight.js.OpClassInit
import io.github.alexzhirkevich.keight.js.OpFunctionInit
import io.github.alexzhirkevich.keight.js.ReferenceError
import io.github.alexzhirkevich.keight.js.StaticClassMember
import io.github.alexzhirkevich.keight.js.SyntaxError
import io.github.alexzhirkevich.keight.js.Undefined
import io.github.alexzhirkevich.keight.Uninitialized
import io.github.alexzhirkevich.keight.expressions.AggregatingExportEntry
import io.github.alexzhirkevich.keight.expressions.ImportEntry
import io.github.alexzhirkevich.keight.expressions.OpAggregatingExport
import io.github.alexzhirkevich.keight.expressions.OpExport
import io.github.alexzhirkevich.keight.expressions.OpImport
import io.github.alexzhirkevich.keight.js.joinSuccess
import io.github.alexzhirkevich.keight.js.js
import io.github.alexzhirkevich.keight.js.listOf
import io.github.alexzhirkevich.keight.js.toFunctionParam
import io.github.alexzhirkevich.keight.js.toJsRegex
import io.github.alexzhirkevich.keight.js.validateFunctionParams
import kotlinx.coroutines.Deferred
import kotlinx.coroutines.Job
import kotlin.collections.List
import kotlin.collections.ListIterator
import kotlin.collections.associateBy
import kotlin.collections.buildList
import kotlin.collections.drop
import kotlin.collections.emptyList
import kotlin.collections.firstOrNull
import kotlin.collections.fold
import kotlin.collections.joinToString
import kotlin.collections.last
import kotlin.collections.lastOrNull
import kotlin.collections.listOf
import kotlin.collections.map
import kotlin.collections.mutableListOf
import kotlin.collections.mutableMapOf
import kotlin.collections.plus
import kotlin.collections.set
import kotlin.collections.single
import kotlin.collections.singleOrNull
import kotlin.contracts.ExperimentalContracts
import kotlin.contracts.contract
import kotlin.math.pow

internal fun List<Token>.parse() : Expression {
    return sanitize()
        .listIterator()
        .parseBlock(
            scoped = false,
            isExpressible = true,
            blockContext = emptyList(),
            type = ExpectedBlockType.Block
        )
}

internal enum class ExpectedBlockType {
    None, Object, Block
}

internal enum class BlockContext {
    None, Loop, Switch, Function, Class, Object, Ternary
}

private fun unexpected(expr : String) : String = "Unexpected token '$expr'"

private fun List<Token>.sanitize() : List<Token> {

    return fold(mutableListOf<Token>()) { l, token ->
        // skip comments, double newlines and newlines after semicolon
        if (
            token !is Token.Comment &&
            (token !is Token.NewLine || (l.lastOrNull() !is Token.NewLine && l.lastOrNull() !is Token.Operator.SemiColon))
        ) {
            l.add(token)
        }
        l
    }
}

private inline fun ListIterator<Token>.eat(token: Token) : Boolean {
    val i = nextIndex()
    if (nextSignificant() == token){
        return true
    } else {
        returnToIndex(i)
        return false
    }
}

private inline fun <reified R : Token> ListIterator<Token>.nextIsInstance() : Boolean {
    if (!hasNext())
        return false

    val i = nextIndex()

    return (nextSignificant() is R).also { returnToIndex(i) }
}

private fun ListIterator<Token>.nextSignificant() : Token {
    var n = next()
    while (n is Token.NewLine){
        n = next()
    }
    return n
}

private fun ListIterator<Token>.returnToIndex(idx : Int){
    while (nextIndex() > idx){
        previous()
    }
}

private fun ListIterator<Token>.prevSignificant() : Token {
    var n = previous()
    while (n is Token.NewLine){
        n = previous()
    }
    return n
}

private fun ListIterator<Token>.parseStatement(
    blockContext: List<BlockContext> = emptyList(),
    maxPrecedence: Int = 14,
    blockType: ExpectedBlockType,
    isBlockAnchor : Boolean = false,
) : Expression {

    var precedence = -1

    var x = parseFactor(blockContext, blockType)

    while (++precedence <= maxPrecedence) {

        if (x is OpConstant && (x.value is JSFunction || x.value is OpClassInit) && isBlockAnchor) {
            return x
        }

        while (true) {

            val i = nextIndex()

            x = when (precedence) {
                0 -> when (nextSignificant()) {
                    is Token.Operator.Bracket.RoundOpen -> {
                        prevSignificant()
                        parseFunctionCall(x, blockContext = blockContext)
                    }

                    is Token.Operator.Period,
                    is Token.Operator.DoublePeriod,
                    is Token.Operator.Bracket.SquareOpen -> {
                        prevSignificant()
                        parseMemberOf(x)
                    }

                    is Token.Operator.OptionalChaining -> parseOptionalChaining(x)
                    else -> { returnToIndex(i); break; }
                }

                1 -> when (val it = nextSignificant()) {
                    is Token.Operator.Arithmetic.Inc,
                    is Token.Operator.Arithmetic.Dec -> {
                        syntaxCheck(x.isAssignable()) {
                            "Value is not assignable"
                        }
                        OpIncDecAssign(
                            variable = x,
                            isPrefix = false,
                            isInc = it is Token.Operator.Arithmetic.Inc
                        )
                    }

                    else -> { returnToIndex(i); break; }
                }

                2 -> when (nextSignificant()) {
                    Token.Operator.Arithmetic.Exp -> OpExp(
                        x = x,
                        degree = parseStatement(
                            blockContext,
                            precedence,
                            ExpectedBlockType.Object
                        )
                    )

                    else -> { returnToIndex(i); break; }
                }

                3 -> when (nextSignificant()) {

                    Token.Operator.Arithmetic.Mul -> Delegate(
                        a = x,
                        b = parseStatement(
                            blockContext,
                            precedence - 1,
                            ExpectedBlockType.Object
                        ),
                        op = ScriptRuntime::mul
                    )

                    Token.Operator.Arithmetic.Div -> Delegate(
                        a = x,
                        b = parseStatement(
                            blockContext,
                            precedence - 1,
                            ExpectedBlockType.Object
                        ),
                        op = ScriptRuntime::div
                    )

                    Token.Operator.Arithmetic.Mod -> Delegate(
                        a = x,
                        b = parseStatement(
                            blockContext,
                            precedence - 1,
                            ExpectedBlockType.Object
                        ),
                        op = ScriptRuntime::mod
                    )

                    else -> { returnToIndex(i); break; }
                }

                4 -> when (nextSignificant()) {
                    Token.Operator.Arithmetic.Plus -> Delegate(
                        a = x,
                        b = parseStatement(
                            blockContext,
                            precedence - 1,
                            ExpectedBlockType.Object
                        ),
                        op = ScriptRuntime::sum
                    )

                    Token.Operator.Arithmetic.Minus -> Delegate(
                        a = x,
                        b = parseStatement(
                            blockContext,
                            precedence - 1,
                            ExpectedBlockType.Object
                        ),
                        op = ScriptRuntime::sub
                    )

                    else -> { returnToIndex(i); break; }
                }

                5 -> when (nextSignificant()) {
                    Token.Operator.Bitwise.Shl -> OpLongInt(
                        a = x,
                        b = parseStatement(
                            blockContext,
                            precedence - 1,
                            ExpectedBlockType.Object
                        ),
                        op = Long::shl
                    )

                    Token.Operator.Bitwise.Shr -> OpLongInt(
                        a = x,
                        b = parseStatement(
                            blockContext,
                            precedence - 1,
                            ExpectedBlockType.Object
                        ),
                        op = Long::shr
                    )

                    Token.Operator.Bitwise.Ushr -> OpLongInt(
                        a = x,
                        b = parseStatement(
                            blockContext,
                            precedence - 1,
                            ExpectedBlockType.Object
                        ),
                        op = Long::ushr
                    )

                    else -> { returnToIndex(i); break; }
                }

                6 -> when (nextSignificant()) {
                    Token.Operator.In -> parseInOperator(x, precedence)
                    Token.Operator.Instanceof -> parseInstanceOfOperator(x, precedence)
                    else -> { returnToIndex(i); break; }
                }

                7 -> when (nextSignificant()) {
                    Token.Operator.Compare.Less -> OpCompare(
                        a = x,
                        b = parseStatement(blockContext, precedence, ExpectedBlockType.Object),
                        result = { it < 0 }
                    )

                    Token.Operator.Compare.LessOrEq -> OpCompare(
                        a = x,
                        b = parseStatement(blockContext, precedence, ExpectedBlockType.Object),
                        result = { it <= 0 }
                    )

                    Token.Operator.Compare.Greater -> OpCompare(
                        a = x,
                        b = parseStatement(blockContext, precedence, ExpectedBlockType.Object),
                        result = { it > 0 }
                    )

                    Token.Operator.Compare.GreaterOrEq -> OpCompare(
                        a = x,
                        b = parseStatement(blockContext, precedence, ExpectedBlockType.Object),
                        result = { it >= 0 }
                    )

                    else -> { returnToIndex(i); break; }
                }

                8 -> when (nextSignificant()) {
                    Token.Operator.Compare.Eq -> OpEquals(
                        a = x,
                        b = parseStatement(blockContext, precedence, ExpectedBlockType.Object),
                        isTyped = false
                    )

                    Token.Operator.Compare.StrictEq -> OpEquals(
                        a = x,
                        b = parseStatement(blockContext, precedence, ExpectedBlockType.Object),
                        isTyped = true
                    )

                    Token.Operator.Compare.Neq -> OpNotEquals(
                        a = x,
                        b = parseStatement(blockContext, precedence, ExpectedBlockType.Object),
                        isTyped = false
                    )

                    Token.Operator.Compare.StrictNeq -> OpNotEquals(
                        a = x,
                        b = parseStatement(blockContext, precedence, ExpectedBlockType.Object),
                        isTyped = true
                    )

                    else -> { returnToIndex(i); break; }
                }

                9 -> when (nextSignificant()) {
                    Token.Operator.Bitwise.And -> OpLongLong(
                        a = x,
                        b = parseStatement(blockContext, precedence, ExpectedBlockType.Object),
                        op = Long::and
                    )

                    else -> { returnToIndex(i); break; }
                }

                10 -> when (nextSignificant()) {
                    Token.Operator.Bitwise.Xor -> OpLongLong(
                        a = x,
                        b = parseStatement(blockContext, precedence, ExpectedBlockType.Object),
                        op = Long::xor
                    )

                    else -> { returnToIndex(i); break; }
                }

                11 -> when (nextSignificant()) {
                    Token.Operator.Bitwise.Or -> OpLongLong(
                        a = x,
                        b = parseStatement(blockContext, precedence, ExpectedBlockType.Object),
                        op = Long::or
                    )

                    else -> { returnToIndex(i); break; }
                }

                12 -> when (nextSignificant()) {
                    Token.Operator.Logical.And -> {
                        val a = x
                        val b =
                            parseStatement(blockContext, precedence, ExpectedBlockType.Object)
                        Expression {
                            val av = a(it)
                            if (it.isFalse(av)) {
                                return@Expression av
                            }
                            b(it)
                        }
                    }

                    else -> { returnToIndex(i); break; }
                }

                13 -> when (nextSignificant()) {
                    Token.Operator.Logical.Or -> {
                        val a = x
                        val b =
                            parseStatement(blockContext, precedence, ExpectedBlockType.Object)
                        Expression {
                            val av = a(it)
                            if (!it.isFalse(av)) {
                                return@Expression av
                            }
                            b(it)
                        }
                    }

                    Token.Operator.NullishCoalescing -> {
                        val replacement =
                            parseStatement(blockContext, precedence, ExpectedBlockType.Object)
                        val subject = x
                        Expression {
                            subject(it)?.takeUnless { it is Undefined } ?: replacement(it)
                        }
                    }

                    else -> { returnToIndex(i); break; }
                }

                14 -> when (val next = nextSignificant()) {
                    Token.Operator.QuestionMark -> parseTernary(
                        condition = x,
                        precedence = precedence,
                        blockContext = blockContext
                    )

                    Token.Operator.Arrow -> OpFunctionInit(parseArrowFunction(blockContext, x))

                    is Token.Operator.Assign -> parseAssignmentValue(
                        x = x,
                        blockContext = blockContext,
                        merge = getMergeForAssignment(next)
                    )

                    is Token.Operator.Colon -> if (blockContext.lastOrNull() != BlockContext.Ternary) {
                        OpColonAssignment(
                            key = when (x) {
                                is OpGetProperty -> x.name.js
                                is OpConstant -> x.value
                                else -> throw SyntaxError("Invalid ussage of : operator")
                            },
                            value = parseStatement(
                                blockContext,
                                precedence,
                                ExpectedBlockType.Object
                            )
                        )
                    } else {
                        returnToIndex(i); break;
                    }

                    else -> { returnToIndex(i); break; }
                }


                else -> error("Invalid operator priority - $precedence")
            }
        }
    }
    return x
}

private fun ListIterator<Token>.parseFactor(
    blockContext: List<BlockContext>,
    blockType: ExpectedBlockType = ExpectedBlockType.None
): Expression {
    val expr =  when (val next = nextSignificant()) {
        is Token.Str -> OpConstant(next.value.js)
        is Token.Regex -> OpConstant(next.value.toJsRegex())
        is Token.TemplateString -> {
            val expressions = next.tokens.fastMap {
                when (it) {
                    is TemplateStringToken.Str -> OpConstant(it.value.js)
                    is TemplateStringToken.Template -> it.value.sanitize().listIterator()
                        .parseBlock(
                            type = ExpectedBlockType.Block,
                            isExpressible = true,
                            blockContext = emptyList()
                    )
                }
            }
            Expression { r ->
                expressions.fastMap { it(r) }.joinToString("").js
            }
        }
        is Token.Operator.Period, is Token.Num -> {
            val num = if (next is Token.Num) {
                next.value
            } else {
                val number = next()
                syntaxCheck(
                    number is Token.Num
                            && !number.isFloat
                            && number.format == NumberFormat.Dec
                ) {
                    unexpected(".")
                }
                "0.${number.value}".toDouble()
            }
            OpConstant(num.js)
        }

        Token.Operator.Spread -> OpSpread(
            parseStatement(
                blockType = ExpectedBlockType.Object,
            )
        ).also {
            if (it.value is OpFunctionInit && it.value.function.isArrow){
                throw SyntaxError(unexpected("..."))
            }
        }
        Token.Operator.Arithmetic.Inc,
        Token.Operator.Arithmetic.Dec -> {
            val isInc = next is Token.Operator.Arithmetic.Inc
            val variable = parseStatement(maxPrecedence = 0, blockType = ExpectedBlockType.Object)

            require(variable.isAssignable()) {
                unexpected(if (isInc) "++" else "--")
            }
            OpIncDecAssign(
                variable = variable,
                isPrefix = true,
                isInc = isInc
            )
        }

        Token.Operator.Arithmetic.Plus,
        Token.Operator.Arithmetic.Minus -> Delegate(
            a = parseStatement(maxPrecedence = 0, blockType = ExpectedBlockType.Object),
            op = if (next is Token.Operator.Arithmetic.Plus)
                ScriptRuntime::pos else ScriptRuntime::neg
        )

        Token.Operator.Logical.Not -> OpNot(
            condition = parseStatement(maxPrecedence = 0, blockType = ExpectedBlockType.Object)
        )

        Token.Operator.Bitwise.Reverse -> {
            val expr = parseStatement(maxPrecedence = 0, blockType = ExpectedBlockType.Object)
            Expression {
                it.toNumber(expr(it)).toLong().inv().js
            }
        }

        Token.Operator.Bracket.CurlyOpen -> {
            prevSignificant()
            parseBlock(
                blockContext = blockContext,
                type = blockType
            )
        }

        Token.Operator.Bracket.RoundOpen -> {
            prevSignificant()
            parseExpressionGrouping()
        }
        Token.Operator.Bracket.SquareOpen -> {
            prevSignificant()
            parseArrayCreation()
        }
        is Token.Operator.New -> parseNew()
        is Token.Operator.Typeof -> parseTypeof()
        is Token.Operator.Void -> parseVoid()
        is Token.Operator.Delete -> parseDelete()
        is Token.Identifier.Keyword -> parseKeyword(next, blockContext)
        is Token.Identifier.Reserved -> throw SyntaxError("Unexpected reserved word (${next.identifier})")
        is Token.Identifier.Property -> {
            val isObject = blockContext.lastOrNull() == BlockContext.Object
            var i = nextIndex()
            val n = nextSignificant()
            when {
                isObject && next.identifier == "get" && n is Token.Identifier ->
                    OpGetter(parseFunction(name = n.identifier, blockContext = emptyList()))
                isObject && next.identifier == "set" && n is Token.Identifier ->
                    OpSetter(parseFunction(name = n.identifier, blockContext = emptyList()))
                else -> {
                    returnToIndex(i)
                    OpGetProperty(next.identifier, receiver = null)
                }
            }
        }
        is Token.Operator.SemiColon -> Expression { Undefined }
        else -> throw SyntaxError(unexpected(next::class.simpleName.orEmpty()))
    }

    return expr
}

private fun ListIterator<Token>.parseAssignmentValue(
    x: Expression,
    blockContext: List<BlockContext>,
    merge: (suspend ScriptRuntime.(JsAny?, JsAny?) -> JsAny?)? = null
): Expression {
    return when (x) {
        is OpIndex -> OpAssignByIndex(
            receiver = x.receiver,
            index = x.index,
            assignableValue = parseStatement(blockType = ExpectedBlockType.Object, blockContext = blockContext),
            merge = merge
        )

        is OpGetProperty -> OpAssign(
            variableName = x.name,
            receiver = x.receiver,
            assignableValue = parseStatement(blockType = ExpectedBlockType.Object,blockContext = blockContext),
            merge = merge
        )
        is OpBlock, is OpMake -> OpDestructAssign(
            destruction = x.asDestruction(),
            variableType = null,
            value = parseStatement(blockType = ExpectedBlockType.Object, blockContext = blockContext)
        )
        is OpSpread -> throw SyntaxError("Rest parameter may not have a default initializer")

        else -> throw SyntaxError("Invalid left-hand in assignment ($x)")
    }
}


private fun getMergeForAssignment(operator: Token.Operator.Assign): (suspend ScriptRuntime.(JsAny?, JsAny?) -> JsAny?)? {
    return when (operator) {
        Token.Operator.Assign.Assignment -> null
        Token.Operator.Assign.PlusAssign -> ScriptRuntime::sum
        Token.Operator.Assign.MinusAssign -> ScriptRuntime::sub
        Token.Operator.Assign.MulAssign -> ScriptRuntime::mul
        Token.Operator.Assign.DivAssign -> ScriptRuntime::div
        Token.Operator.Assign.ExpAssign -> { a, b ->
            toNumber(a).toDouble().pow(toNumber(b).toDouble()).js
        }

        Token.Operator.Assign.ModAssign -> { a, b ->
            (toNumber(a).toLong() and toNumber(b).toLong()).js
        }

        Token.Operator.Assign.BitAndAssign -> { a, b ->
            (toNumber(a).toLong() and toNumber(b).toLong()).js
        }

        Token.Operator.Assign.LogicAndAssign -> { a, b ->
            if (isFalse(a)) a else b
        }

        Token.Operator.Assign.BitOrAssign -> { a, b ->
            (toNumber(a).toLong() or toNumber(b).toLong()).js
        }

        Token.Operator.Assign.LogicOrAssign -> { a, b ->
            if (isFalse(a)) b else a
        }

        Token.Operator.Assign.BitXorAssign -> { a, b ->
            (toNumber(a).toLong() xor toNumber(b).toLong()).js
        }

        Token.Operator.Assign.UshrAssign -> { a, b ->
            (toNumber(a).toLong() ushr toNumber(b).toInt()).js
        }

        Token.Operator.Assign.ShrAssign -> { a, b ->
            (toNumber(a).toLong() shr toNumber(b).toInt()).js
        }

        Token.Operator.Assign.ShlAssign -> { a, b ->
            (toNumber(a).toLong() shl toNumber(b).toInt()).js
        }

        Token.Operator.Assign.NullCoalescingAssign -> { a, b ->
            if (a == null || a is Undefined) b else a
        }
    }
}

private fun ListIterator<Token>.parseKeyword(keyword: Token.Identifier.Keyword, blockContext: List<BlockContext>): Expression {
    return when(keyword){
        Token.Identifier.Keyword.Var,
        Token.Identifier.Keyword.Let,
        Token.Identifier.Keyword.Const, -> parseVariable(
            when(keyword){
                Token.Identifier.Keyword.Var -> VariableType.Global
                Token.Identifier.Keyword.Let -> VariableType.Local
                else -> VariableType.Const
            }
        )
        Token.Identifier.Keyword.True -> OpConstant(true.js)
        Token.Identifier.Keyword.False -> OpConstant(false.js)
        Token.Identifier.Keyword.Null -> OpConstant(null)

        Token.Identifier.Keyword.Switch -> parseSwitch(blockContext)
        Token.Identifier.Keyword.Case,
        Token.Identifier.Keyword.Default -> {
            syntaxCheck(blockContext.last() == BlockContext.Switch) {
                unexpected("case")
            }
            val case = if (keyword == Token.Identifier.Keyword.Case)
                parseFactor(emptyList())
            else OpCase.Default

            syntaxCheck(nextSignificant() is Token.Operator.Colon) {
                "Expected ':' after 'case'"
            }
            OpCase(case)
        }

        Token.Identifier.Keyword.For -> parseForLoop(blockContext)
        Token.Identifier.Keyword.While -> parseWhileLoop(blockContext)
        Token.Identifier.Keyword.Do -> parseDoWhileLoop(blockContext)
        Token.Identifier.Keyword.Continue -> {
            val label = (next() as? Token.Identifier)?.identifier ?: null.also { previous() }
            OpContinue(label).also {
                syntaxCheck(blockContext.lastOrNull() == BlockContext.Loop) {
                    unexpected("continue")
                }
            }
        }
        Token.Identifier.Keyword.Break -> {
            val label = (next() as? Token.Identifier)?.identifier ?: null.also { previous() }
            OpBreak(label).also {
                val context = blockContext.lastOrNull()
                syntaxCheck(context == BlockContext.Loop || context == BlockContext.Switch || label != null) {
                    unexpected("break")
                }
            }
        }

        Token.Identifier.Keyword.If ->  OpIfCondition(
            condition = parseExpressionGrouping(),
            onTrue = parseBlock(blockContext = blockContext),
            onFalse = if (eat(Token.Identifier.Keyword.Else)) {
                parseBlock(blockContext = blockContext)
            } else null
        )
        Token.Identifier.Keyword.Function -> OpFunctionInit(parseFunction(blockContext = blockContext))
        Token.Identifier.Keyword.Return -> {
            syntaxCheck(BlockContext.Function in blockContext) {
                unexpected("return")
            }
            val next = next()
            if (next == Token.NewLine || next == Token.Operator.SemiColon){
                OpReturn(OpConstant(Undefined))
            } else {
                previous()
                OpReturn(parseStatement(blockType = ExpectedBlockType.Object))
            }
        }

        Token.Identifier.Keyword.Class -> parseClass()
        Token.Identifier.Keyword.Throw -> {
            val throwable = parseStatement(blockType = ExpectedBlockType.Object)
            Expression {
                val t = throwable(it)
                throw t as? Throwable ?: ThrowableValue(t)
            }
        }
        Token.Identifier.Keyword.Async -> parseAsync()
        Token.Identifier.Keyword.Await -> parseAwait()
        Token.Identifier.Keyword.Try -> parseTryCatch(blockContext)
        Token.Identifier.Keyword.This -> Expression { it.thisRef }
        Token.Identifier.Keyword.With -> {
            TODO()
//            val arg = parseExpressionGrouping().expressions.single()
//            val block = parseBlock(type = ExpectedBlockType.Block, scoped = false, blockContext = emptyList())
//
//            return Expression { r ->
//
//                val a = arg(r)
//
//                if (a is JSObject) {
//                    r.withScope(
//                        thisRef = a,
//                    ) {
//                        block(it)
//                    }
//                } else {
//
//                }
//            }
        }
        Token.Identifier.Keyword.Debugger -> throw SyntaxError("Debugger is not supported")
        Token.Identifier.Keyword.Else,
        Token.Identifier.Keyword.Extends,
        Token.Identifier.Keyword.Finally,
        Token.Identifier.Keyword.Catch -> throw SyntaxError(unexpected(keyword.identifier))
        Token.Identifier.Keyword.Import -> parseImport()
        Token.Identifier.Keyword.Export -> parseExport()

    }
}

private fun ListIterator<Token>.parseNew() : Expression {
    val index = nextIndex()
    val next = nextSignificant()

    if (next is Token.Identifier.Property) {
        syntaxCheck(next is Token.Identifier.Property) {
            "Invalid syntax after 'new'"
        }

        val args = if (nextIsInstance<Token.Operator.Bracket.RoundOpen>()) {
            parseExpressionGrouping().expressions
        } else {
            emptyList()
        }

        return Expression { runtime ->
            val constructor = runtime.get(next.identifier.js)
            runtime.typeCheck(constructor is Constructor) {
                "'$constructor' (${next.identifier}) is not a constructor".js
            }
            constructor.construct(args.fastMap { it(runtime) }, runtime)
        }
    } else {
        returnToIndex(index)
        val constructor = parseStatement(
            blockContext = emptyList(),
            blockType = ExpectedBlockType.Object
        )
        return Expression { runtime ->
            val c = constructor.invoke(runtime)
            runtime.typeCheck(c is Constructor) {
                "'$c' is not a constructor".js
            }
            runtime.typeCheck(c !is JSFunction || !c.isArrow){
                "(intermediate value) is not a constructor".js
            }
            c.construct(emptyList(), runtime)
        }
    }
}

private fun ListIterator<Token>.parseVoid() : Expression {
    val isArg = nextIsInstance<Token.Operator.Bracket.RoundOpen>()
    val expr = if (isArg) {
        parseExpressionGrouping()
    }  else {
        parseStatement(maxPrecedence = 1,blockType = ExpectedBlockType.Object)
    }
    return Expression {
        expr(it)
        Undefined
    }
}

private fun ListIterator<Token>.parseTypeof() : Expression {
    val isArg = nextIsInstance<Token.Operator.Bracket.RoundOpen>()
    val expr = if (isArg) {
        parseExpressionGrouping()
    }  else {
        parseStatement(maxPrecedence = 1,blockType = ExpectedBlockType.Object)
    }
    return Expression {
        try {
            expr(it)?.type?.js ?: "object".js
        } catch (t : ReferenceError){
            "undefined".js
        }
    }
}

private fun ListIterator<Token>.parseDelete() : Expression {
    val x = parseStatement(maxPrecedence = 1, blockType = ExpectedBlockType.Object)

    val (subj, obj) = when (x) {
        is OpIndex -> x.receiver to x.index
        is OpGetProperty -> x.receiver to OpConstant(x.name.js)
        else -> return OpConstant(false.js)
    }

    return Expression {
        val s = subj?.invoke(it)
        val o = obj(it)
        s?.delete(o, it)?.js ?: it.delete(o).js
    }
}


private fun ListIterator<Token>.parseArrayCreation(): Expression {
    check(eat(Token.Operator.Bracket.SquareOpen))

    val expressions = buildList {
        while (!eat(Token.Operator.Bracket.SquareClose)) {
            if (eat(Token.Operator.Comma)) {
                add(OpConstant(Uninitialized))
            } else {
                add(parseStatement(blockType = ExpectedBlockType.Object))
                if (!eat(Token.Operator.Comma)) {
                    syntaxCheck(nextSignificant() is Token.Operator.Bracket.SquareClose) {
                        "Expected ')'"
                    }
                    break
                }
            }
        }
    }

    return OpMakeArray(expressions)
}

private fun ListIterator<Token>.parseExpressionGrouping(): OpTouple {
    check(eat(Token.Operator.Bracket.RoundOpen))

    val expressions = if (nextIsInstance<Token.Operator.Bracket.RoundClose>()) {
        emptyList()
    } else buildList {
        do {
            if (nextIsInstance<Token.Operator.Bracket.RoundClose>()) {
                return@buildList
            }
            add(parseStatement(emptyList(), blockType = ExpectedBlockType.Object))
        } while (nextSignificant() is Token.Operator.Comma)
        prevSignificant()
    }
    syntaxCheck(eat(Token.Operator.Bracket.RoundClose)) {
        "Expected ')'"
    }

    return OpTouple(expressions)
}

private fun ListIterator<Token>.parseMemberOf(receiver: Expression): Expression {
    return when (nextSignificant()){

        is Token.Operator.Period, is Token.Operator.DoublePeriod -> {
            val next = nextSignificant()
            syntaxCheck(next is Token.Identifier) {
                "Illegal symbol after '.'"
            }
            OpGetProperty(name = next.identifier, receiver = receiver)
        }
        is Token.Operator.Bracket.SquareOpen -> {
            OpIndex(
                receiver = receiver,
                index = parseStatement(blockType = ExpectedBlockType.Object)
            ).also {
                syntaxCheck(nextSignificant() is Token.Operator.Bracket.SquareClose) {
                    "Missing ']'"
                }
            }
        }
        else -> throw IllegalStateException("Illegal 'member of' syntax")
    }
}

private fun ListIterator<Token>.parseOptionalChaining(receiver: Expression): Expression {
    return when(val next = nextSignificant()){
        is Token.Operator.Bracket.SquareOpen -> {
            OpIndex(
                receiver = receiver,
                index = parseStatement(blockType = ExpectedBlockType.Object),
                isOptional = true
            ).also {
                syntaxCheck(nextSignificant() is Token.Operator.Bracket.SquareClose) {
                    "Missing ']'"
                }
            }
        }
        is Token.Operator.Bracket.RoundOpen -> {
            prevSignificant()
            parseFunctionCall(receiver, optional = true, blockContext = emptyList())
        }
        is Token.Identifier -> {
            OpGetProperty(
                name = next.identifier,
                receiver = receiver,
                isOptional = true
            )
        }
        else -> throw SyntaxError("Invalid usage of ?. operator")
    }
}

private fun ListIterator<Token>.parseFunctionCall(
    function : Expression,
    optional : Boolean = false,
    blockContext: List<BlockContext>
) : Expression {

    val argsIndex = nextIndex()
    val arguments = parseExpressionGrouping().expressions
    val afterIndex = nextIndex()

    return if (
        function is OpGetProperty
        && blockContext.lastOrNull() == BlockContext.Object
        && hasNext()
        && next() == Token.Operator.Bracket.CurlyOpen
    ) {
        returnToIndex(argsIndex)
        OpFunctionInit(parseFunction(name = function.name, blockContext = blockContext))
    } else {
        returnToIndex(afterIndex)
        OpCall(
            callable = function,
            arguments = arguments,
            isOptional = optional
        )
    }
}


private fun ListIterator<Token>.parseInOperator(subject : Expression, precedence: Int) : Expression {
    val obj = parseStatement(maxPrecedence = precedence, blockType = ExpectedBlockType.Object)
    return OpIn(
        property = subject,
        inObject = obj
    )
}

private fun ListIterator<Token>.parseInstanceOfOperator(subject : Expression, precedence: Int) : Expression {
    val obj = parseStatement(maxPrecedence = precedence,blockType = ExpectedBlockType.Object)
    return Expression {
        val o = obj(it)
        syntaxCheck(o is Constructor) {
            "Illegal usage of 'instanceof' operator"
        }
        o.isInstance(subject(it), it).js
    }
}

private fun ListIterator<Token>.parseTernary(
    condition : Expression,
    precedence: Int,
    blockContext: List<BlockContext>
) : Expression {

    val body = parseStatement(
        blockContext = blockContext + BlockContext.Ternary,
        blockType = ExpectedBlockType.Block
    )

    val n = nextSignificant()

    syntaxCheck(n is Token.Operator.Colon){
        "Unexpected end of input: $n (after $body)"
    }

    return OpIfCondition(
        condition = condition,
        onTrue = body,
        onFalse = parseStatement(
            blockContext = blockContext,
            blockType = ExpectedBlockType.Block
        ),
        expressible = true
    )
}

private fun ListIterator<Token>.parseClass() : OpClassInit {

    val i = nextIndex()
    val name = nextSignificant().let {
        if (it is Token.Identifier){
            it.identifier
        } else {
            returnToIndex(i)
            ""
        }
    }

    val extends = if (eat(Token.Identifier.Keyword.Extends)) {
        parseStatement(blockType = ExpectedBlockType.Object)
    } else null

    syntaxCheck(eat(Token.Operator.Bracket.CurlyOpen)) {
        "Invalid class declaration"
    }

    val staticMembers = mutableListOf<StaticClassMember>()
    val properties = mutableMapOf<JsAny?, Expression>()
    var construct : JSFunction? = null

    while (!eat(Token.Operator.Bracket.CurlyClose)) {
        val token = nextSignificant()

        when {
            token is Token.Identifier && nextIsInstance<Token.Operator.Bracket.RoundOpen>() -> {
                val f = parseFunction(
                    name = token.identifier,
                    blockContext = listOf(BlockContext.Class)
                )
                if (token.identifier == "constructor"){
                    syntaxCheck(construct == null){
                        "A class may only have one constructor"
                    }
                    construct = f
                }
                properties[token.identifier.js] = OpFunctionInit(f)
            }

            token is Token.Identifier.Property && token.identifier == "static" -> {
                staticMembers.add(parseStaticClassMember())
            }

            token is Token.Identifier -> {
                prevSignificant()
                when (val statement = parseStatement(blockType = ExpectedBlockType.None)) {
                    is OpAssign -> properties[statement.variableName.js] = statement.assignableValue
                    is OpGetProperty -> properties[statement.name.js] = OpConstant(Undefined)
                    else -> throw SyntaxError("Invalid class member")
                }
            }
            else -> throw  SyntaxError("Invalid class declaration")
        }
    }

    return OpClassInit(
        name = name,
        extends = extends,
        properties = properties,
        static = staticMembers/*.reversed()*/.associateBy { it.name.js },
        construct = construct
    )
}

private fun ListIterator<Token>.parseStaticClassMember() : StaticClassMember {

    val name = nextSignificant()


    syntaxCheck(name is Token.Identifier) {
        "Invalid static class member"
    }

    return when(val n = nextSignificant()) {
        is Token.Operator.Assign.Assignment -> {
            prevSignificant()
            prevSignificant()
            val assign = parseStatement(blockType = ExpectedBlockType.Object)
            syntaxCheck(assign is OpAssign) {
                "Invalid static class member"
            }

            StaticClassMember.Variable(assign.variableName, assign.assignableValue)
        }

        is Token.Operator.Bracket.RoundOpen -> {
            prevSignificant()
            val func = parseFunction(name = name.identifier, blockContext = emptyList())

            StaticClassMember.Method(func)
        }

        else -> throw SyntaxError("Invalid static class member $n")
    }
}


private fun ListIterator<Token>.parseSwitch(blockContext: List<BlockContext>) : Expression {
    val value = parseStatement(blockType = ExpectedBlockType.Object) as OpTouple
    val body = parseBlock(
        type = ExpectedBlockType.Block,
        blockContext = blockContext + BlockContext.Switch
    ) as OpBlock

    return OpSwitch(
        value = value.expressions.single(),
        cases = body.expressions
    )
}

private fun ListIterator<Token>.parseForLoop(parentBlockContext: List<BlockContext>): Expression {

    syntaxCheck(nextSignificant() is Token.Operator.Bracket.RoundOpen) {
        "For loop must be followed by '('"
    }

    val assign = if (eat(Token.Operator.SemiColon)) {
        null
    }
    else {
        parseBlock(scoped = false, blockContext = emptyList())
    }

    // for (x in y)
    if (assign is OpBlock) {
        val opIn = assign.expressions.singleOrNull() as? OpIn
        if (opIn != null) {
            return parseForInLoop(opIn, parentBlockContext)
        }
    }

    if (assign != null) {
        syntaxCheck(nextSignificant() is Token.Operator.SemiColon) {
            "Invalid for loop"
        }
    }

    val comparison = if (eat(Token.Operator.SemiColon))
        null else parseStatement(blockType = ExpectedBlockType.Block)

    if (comparison != null) {
        syntaxCheck(nextSignificant() is Token.Operator.SemiColon) {
            "Invalid for loop"
        }
    }

    val increment = if (eat(Token.Operator.Bracket.RoundClose)) {
        null
    } else {
        parseBlock(scoped = false, blockContext = emptyList())
    }

    if (increment != null) {
        syntaxCheck(nextSignificant() is Token.Operator.Bracket.RoundClose) {
            "Invalid for loop"
        }
    }

    val body = parseBlock(blockContext = parentBlockContext + BlockContext.Loop)


    return OpForLoop(
        assignment = assign,
        increment = increment,
        comparison = comparison,
        body = body
    )
}

private fun ListIterator<Token>.parseForInLoop(opIn: OpIn, parentBlockContext : List<BlockContext>) : Expression {
    syntaxCheck(nextSignificant() is Token.Operator.Bracket.RoundClose) {
        "Invalid for loop"
    }

    val prepare = OpAssign(
        type = opIn.variableType,
        variableName = when (opIn.property){
            is OpAssign -> opIn.property.variableName
            is OpGetProperty -> opIn.property.name
            else -> throw SyntaxError("Invalid for..of loop syntax")
        },
        assignableValue = OpConstant(Undefined),
        merge = null
    )

    return OpForInLoop(
        prepare = prepare,
        assign = { r, v -> r.set(prepare.variableName.js, v, null) },
        inObject = opIn.inObject,
        body = parseBlock(blockContext = parentBlockContext + BlockContext.Loop)
    )
}

private fun ListIterator<Token>.parseWhileLoop(parentBlockContext: List<BlockContext>): Expression {
    return OpWhileLoop(
        condition = parseExpressionGrouping(),
        body = parseBlock(blockContext = parentBlockContext + BlockContext.Loop),
    )
}

private fun ListIterator<Token>.parseDoWhileLoop(blockContext: List<BlockContext>) : Expression {
    val body = parseBlock(
        type = ExpectedBlockType.Block,
        blockContext = blockContext + BlockContext.Loop,
        scoped = false // while condition should have the same scope with body
    )

    syntaxCheck(eat(Token.Identifier.Keyword.While)) {
        "Missing while condition in do/while block"
    }
    val condition = parseExpressionGrouping()

    return OpDoWhileLoop(
        condition = condition,
        body = body as OpBlock,
    )
}

private fun ListIterator<Token>.parseAsync(): Expression {
    val subject = parseStatement(blockType = ExpectedBlockType.Object)

    syntaxCheck(subject is OpFunctionInit && !subject.function.isAsync) {
        "Illegal usage of 'async' keyword"
    }

    return OpFunctionInit(subject.function.copy(isAsync = true))
}

private fun ListIterator<Token>.parseAwait(): Expression {
    val subject = parseStatement(blockType = ExpectedBlockType.Object)

    return Expression {

        if (!it.isSuspendAllowed) {
            throw JSError("Await is not allowed in current context")
        }

        val job = subject(it)?.toKotlin(it)
        it.typeCheck(job is Job){
            "$job is not a Promise".js
        }

        if (job is Deferred<*>){
            job.await() as JsAny?
        } else {
            job.joinSuccess()
            Undefined
        }
    }
}

private fun ListIterator<Token>.parseTryCatch(blockContext: List<BlockContext>): Expression {
    val tryBlock = parseBlock(type = ExpectedBlockType.Block, blockContext = blockContext)
    val catchBlock = if (eat(Token.Identifier.Keyword.Catch)) {
        if (eat(Token.Operator.Bracket.RoundOpen)) {
            val next = nextSignificant()
            syntaxCheck(next is Token.Identifier && eat(Token.Operator.Bracket.RoundClose)) {
                "Invalid syntax after 'catch'"
            }
            next.identifier
        } else {
            null
        } to parseBlock(
            type = ExpectedBlockType.Block,
            blockContext = blockContext
        )
    } else null

    val finallyBlock = if (eat(Token.Identifier.Keyword.Finally)) {
        parseBlock(type = ExpectedBlockType.Block, blockContext = blockContext)
    } else null

    return OpTryCatch(
        tryBlock = tryBlock,
        catchVariableName = catchBlock?.first,
        catchBlock = catchBlock?.second,
        finallyBlock = finallyBlock
    )
}

private fun ListIterator<Token>.parseArrowFunction(blockContext: List<BlockContext>, args: Expression) : JSFunction {
    val fArgs = when(args){
        is OpTouple -> args.expressions
        else -> listOf(args)
    }.map(Expression::toFunctionParam)

    validateFunctionParams(fArgs,true)

    val lambda = parseBlock(
        type = ExpectedBlockType.Block,
        blockContext = blockContext + BlockContext.Function,
        allowCommaSeparator = false
    ) as OpBlock

    syntaxCheck (lambda.isSurroundedWithBraces || lambda.expressions.size <= 1){
        "Invalid arrow function"
    }

    return JSFunction(
        name = "",
        parameters = fArgs,
        body = lambda.copy(isExpressible = !lambda.isSurroundedWithBraces),
        isArrow = true
    )
}

private fun ListIterator<Token>.parseFunction(
    name: String? = null,
    blockContext: List<BlockContext>
) : JSFunction {

    val actualName = name ?: run {
        if (nextIsInstance<Token.Identifier.Property>()) {
            (nextSignificant() as Token.Identifier.Property).identifier
        } else {
            ""
        }
    }

    val touple = parseStatement(blockType = ExpectedBlockType.None)

    syntaxCheck(touple is OpTouple) {
        "Invalid function declaration"
    }

    val args = touple.expressions.map(Expression::toFunctionParam)

    validateFunctionParams(args, false)

    val block = parseBlock(
        scoped = false,
        blockContext = blockContext + BlockContext.Function,
        type = ExpectedBlockType.Block
    )

    return JSFunction(
        name = actualName,
        parameters = args,
        body = block
    )
}

private fun ListIterator<Token>.parseImport() : Expression {

    val entries = buildList {
        while (true) {
            val i = nextIndex()
            when (nextSignificant()) {
                is Token.Operator.Bracket.CurlyOpen -> {
                    if (!eat(Token.Operator.Bracket.CurlyClose)) {
                        do {
                            add(parseImportEntry(true))
                        } while (eat(Token.Operator.Comma))

                        val close = nextSignificant()
                        syntaxCheck(close is Token.Operator.Bracket.CurlyClose) {
                            "Invalid import: '}' was expected but got '$close'"
                        }
                    }
                }
                else -> {
                    returnToIndex(i)
                    add(parseImportEntry(false))
                }
            }

            if (!eat(Token.Operator.Comma)){
                break
            }
        }
    }

    val fromKeyword = nextSignificant()

    syntaxCheck(fromKeyword is Token.Identifier && fromKeyword.identifier == "from") {
        "Invalid import: 'from' was expected"
    }

    val moduleName = parseStatement(blockType = ExpectedBlockType.Block)

    return OpImport(
        entries = entries,
        fromModule = moduleName
    )
}

private fun ListIterator<Token>.parseImportEntry(
    isInBrackets : Boolean
) : ImportEntry {
    return if (isInBrackets) {
        val import = when(val i = nextSignificant()){
            is Token.Identifier -> i.identifier
            is Token.Str -> i.value
            else -> throw SyntaxError( "Invalid import: unexpected token $i")
        }
        val i = nextIndex()
        val n = nextSignificant()
        val alias = if (n is Token.Identifier && n.identifier == "as"){
            nextSignificant().identifier()
        } else {
            returnToIndex(i)
            null
        }
        if (import == "default") {
            syntaxCheck(alias != null){
                "Invalid import: alias for default import was expected"
            }
            ImportEntry.Default(alias)
        } else {
            ImportEntry.Named(import, alias)
        }
    } else {
        when (val n = nextSignificant()){
            is Token.Operator.Arithmetic.Mul -> {
                val n = nextSignificant()
                syntaxCheck(n is Token.Identifier && n.identifier == "as"){
                    "Invalid import: unexpected token $n"
                }
                ImportEntry.Star(nextSignificant().identifier())
            }
            is Token.Identifier -> ImportEntry.Default(n.identifier)
            else -> throw SyntaxError("Invalid import: unexpected token $n")
        }
    }
}

private fun Token.identifier() : String {
    syntaxCheck(this is Token.Identifier){
        "Invalid import: unexpected token $this"
    }
    return identifier
}

private fun ListIterator<Token>.parseExport(isDefault : Boolean = false) : Expression {

    if (!isDefault){
        val i = nextIndex()

        if (nextSignificant() == Token.Identifier.Keyword.Default){
            return parseExport(isDefault = true)
        } else {
            returnToIndex(i)
        }
    }

    return when {
        isDefault -> parseExportDeclaration(true)
        eat(Token.Operator.Arithmetic.Mul) -> {
            val index = nextIndex()
            val (alias, declareProperty) = parseExportAliasAndType()
            if (alias == null) {
                returnToIndex(index)
            }

            val from = nextSignificant()
            syntaxCheck(from is Token.Identifier && from.identifier == "from"){
                "Invalid export: 'from' was unexpected but got $from"
            }
            val fromModule = parseStatement(blockType = ExpectedBlockType.None)

            OpAggregatingExport(
                exports = listOf(AggregatingExportEntry.Star(alias, declareProperty)),
                fromModule = fromModule
            )
        }
        eat(Token.Operator.Bracket.CurlyOpen) -> {
            val exports = buildList {
                while (!eat(Token.Operator.Bracket.CurlyClose)) {
                    add(parseAggregatingExportEntry())
                }
            }
            val from = nextSignificant()

            syntaxCheck(from is Token.Identifier && from.identifier == "from"){
                "Invalid export: 'from' was unexpected but got $from"
            }
            val fromModule = parseStatement(blockType = ExpectedBlockType.None)

            OpAggregatingExport(exports, fromModule)
        }
        else -> parseExportDeclaration()
    }
}

private fun ListIterator<Token>.parseExportAliasAndType() : Pair<String?, Boolean> {
    val n = nextSignificant()
    syntaxCheck(n is Token.Identifier){
        "Invalid export"
    }

    return if (n.identifier == "as"){
        nextSignificant().asExportAliasAndType()
    } else {
        null to false
    }
}

private fun Token.asExportAliasAndType() : Pair<String?, Boolean> {
    return when (this){
        is Token.Identifier -> identifier to true
        is Token.Str -> value to false
        else -> throw SyntaxError( "Invalid export: unexpected token $this")
    }
}

private fun ListIterator<Token>.parseAggregatingExportEntry() : AggregatingExportEntry {
    val import = nextSignificant()

    syntaxCheck(import is Token.Identifier) {
        "Invalid export"
    }

    val (alias, declareProperty) = if (eat(Token.Operator.Comma) || nextIsInstance<Token.Operator.Bracket.CurlyClose>()) {
        null to false
    } else {
        parseExportAliasAndType()
    }

    return AggregatingExportEntry.Single(
        import = if (import.identifier == "default") null else import.identifier,
        alias = alias,
        assignPropertyForAlias = declareProperty
    )
}


private fun ListIterator<Token>.parseExportDeclaration(isDefault: Boolean = false) : Expression {
    val expr = parseStatement(blockType = ExpectedBlockType.Block)

    if (isDefault){
        return OpExport(null, expr)
    }

    val (name, property) = when (expr) {
        is OpFunctionInit -> expr.function.name to expr
        is OpClassInit -> expr.name to expr
        is OpAssign -> expr.variableName to expr
        is OpGetProperty -> expr.name to expr
        else -> error("Invalid export")
    }

    return OpExport(name, property)
}

private fun ListIterator<Token>.parseBlock(
    scoped: Boolean = true,
    type: ExpectedBlockType = ExpectedBlockType.None,
    isExpressible: Boolean = false,
    allowCommaSeparator : Boolean = true,
    blockContext: List<BlockContext>,
): Expression {
    var hoistedIndex = 0

    var isSurroundedWithBraces = false
    val list = buildList {
        if (eat(Token.Operator.Bracket.CurlyOpen)) {
            isSurroundedWithBraces = true
            val context = if (type == ExpectedBlockType.Object)
                blockContext + BlockContext.Object
            else blockContext
            while (!nextIsInstance<Token.Operator.Bracket.CurlyClose>()) {
                val expr = parseStatement(
                    blockContext = context,
                    blockType = ExpectedBlockType.None,
                    isBlockAnchor = true
                )

                // hoisted
                when {
                    expr is OpClassInit -> {
                        val assign = OpAssign(
                            type = VariableType.Local,
                            variableName = expr.name,
                            receiver = null,
                            assignableValue = expr,
                            merge = null
                        )
                        add(
                            index = hoistedIndex++,
                            element = Expression { assign(it); Undefined }
                        )
                    }

                    expr is OpFunctionInit && !expr.function.isArrow -> {
                        val name = expr.function.name

                        syntaxCheck(name.isNotBlank()) {
                            "Function statements require a function name"
                        }

                        val assign = OpAssign(
                            type = VariableType.Local,
                            variableName = name,
                            receiver = null,
                            assignableValue = expr,
                            merge = null
                        )
                        add(
                            index = hoistedIndex++,
                            element = Expression { assign(it); Undefined }
                        )
                    }

                    expr is OpExport && expr.property is OpClassInit -> {
                        val assign = OpAssign(
                            type = VariableType.Local,
                            variableName = expr.property.name,
                            receiver = null,
                            assignableValue = expr.property,
                            merge = null
                        )
                        add(
                            index = hoistedIndex++,
                            element = Expression {
                                expr(it)
                                assign(it);
                                Undefined
                            }
                        )
                    }

                    expr is OpExport && expr.property is OpFunctionInit && !expr.property.function.isArrow -> {

                        val name = expr.property.function.name

                        syntaxCheck(name.isNotBlank()) {
                            "Function statements require a function name"
                        }

                        val assign = OpAssign(
                            type = VariableType.Local,
                            variableName = name,
                            receiver = null,
                            assignableValue = expr.property,
                            merge = null
                        )
                        add(
                            index = hoistedIndex++,
                            element = Expression {
                                expr(it)
                                assign(it);
                                Undefined
                            }
                        )
                    }
                    expr is OpImport -> add(hoistedIndex++, expr)

                    else -> add(expr)
                }
                var hasSeparator = false
                while (hasNext()) {
                    val next = next()
                    if (next !is Token.NewLine && next !is Token.Operator.SemiColon && next !is Token.Operator.Comma) {
                        previous()
                        break
                    }
                    hasSeparator = true
                }
                syntaxCheck(hasSeparator || nextIsInstance<Token.Operator.Bracket.CurlyClose>()) {
                    unexpected(next().toString())
                }
            }

            check(nextSignificant() is Token.Operator.Bracket.CurlyClose) {
                "} was expected"
            }
        } else {

            while (eat(Token.Operator.New)) {
                //skip
            }
            do {
                add(parseStatement(blockContext, blockType = type))
            } while (allowCommaSeparator && eat(Token.Operator.Comma))
        }
    }

    return if (
        type != ExpectedBlockType.Block
        && isSurroundedWithBraces
        && list.fastAll {
            it is OpColonAssignment //  { a : 'b' }
                    || it is OpSpread //  { ...obj }
                    || it is PropertyAccessorFactory //  { get x(){} }
        }
    ) {
        OpMakeObject(list)
    } else {
        val (isStrict, exprs) = if ((list.firstOrNull() as? OpConstant)?.value?.toString() == "use strict") {
            true to list.drop(1)
        } else {
            false to list
        }

        OpBlock(
            expressions = exprs,
            isScoped = scoped,
            isStrict = isStrict,
            isExpressible = isExpressible,
            isSurroundedWithBraces = isSurroundedWithBraces
        )
    }
}

private fun ListIterator<Token>.parseVariable(type: VariableType) : Expression {
    val expressions = buildList {
        do {
            val variable = when (val expr = parseStatement(blockType = ExpectedBlockType.None)) {
                is OpAssign -> OpAssign(
                    type = type,
                    variableName = expr.variableName,
                    assignableValue = expr.assignableValue,
                    merge = null
                )
                is OpGetProperty -> OpAssign(
                    type = type,
                    variableName = expr.name,
                    assignableValue = OpConstant(Undefined),
                    merge = null
                )

                is OpDestructAssign -> OpDestructAssign(
                    destruction = expr.destruction,
                    variableType = type,
                    value = expr.value
                )

                // for (let x in y) ...
                is OpIn -> expr.also { it.variableType = type }

                else -> throw SyntaxError(unexpected(expr::class.simpleName.orEmpty()))
            }
            add(variable)
        } while (eat(Token.Operator.Comma))
    }

    return expressions.singleOrNull() ?: OpBlock(
        expressions = expressions,
        isScoped = false,
        isExpressible = false,
        isSurroundedWithBraces = false
    )
}

@OptIn(ExperimentalContracts::class)
internal fun checkArgs(args : List<*>?, count : Int, func : String) {
    contract {
        returns() implies (args != null)
    }
    checkNotNull(args){
        "$func call was missing"
    }
    require(args.size == count){
        "$func takes $count arguments, but ${args.size} got"
    }
}


@OptIn(ExperimentalContracts::class)
internal inline fun syntaxCheck(value: Boolean, lazyMessage: () -> Any) {
    contract {
        returns() implies value
    }

    if (!value) {
        val message = lazyMessage()
        throw SyntaxError(message.toString())
    }
}

@OptIn(ExperimentalContracts::class)
internal suspend inline fun ScriptRuntime.typeCheck(value: Boolean, lazyMessage: () -> JsAny) {
    contract { returns() implies value }

    if (!value) {
        typeError(lazyMessage)
    }
}

internal suspend inline fun ScriptRuntime.typeError(lazyMessage: () -> JsAny) : Nothing {
    throw makeTypeError(lazyMessage)
}

@OptIn(ExperimentalContracts::class)
internal suspend inline fun ScriptRuntime.referenceCheck(value: Boolean, lazyMessage: () -> JsAny) {
    contract { returns() implies value }

    if (!value) {
        referenceError(lazyMessage)
    }
}

internal suspend inline fun ScriptRuntime.referenceError(lazyMessage: () -> JsAny) : Nothing {
    throw makeReferenceError(lazyMessage)
}

internal suspend inline fun ScriptRuntime.makeReferenceError(lazyMessage: () -> JsAny) : ReferenceError {
    return findJsRoot().ReferenceError
        .construct(lazyMessage().listOf(), this) as ReferenceError
}

internal suspend inline fun ScriptRuntime.makeTypeError(lazyMessage: () -> JsAny) : Throwable {
    return findJsRoot().TypeError
        .construct(lazyMessage().listOf(), this) as Throwable
}


internal fun Expression.isAssignable() : Boolean {
    return this is OpGetProperty ||
            this is OpIndex && receiver is OpGetProperty
}


