/**
 * <!--
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 * -->
 * <p>
 * <h1>cdk-nag</h1>
 * <p>
 * <a href="https://pypi.org/project/cdk-nag/"><img alt="PyPI version" src="https://img.shields.io/pypi/v/cdk-nag"></a>
 * <a href="https://www.npmjs.com/package/cdk-nag"><img alt="npm version" src="https://img.shields.io/npm/v/cdk-nag"></a>
 * <a href="https://search.maven.org/search?q=a:cdknag"><img alt="Maven version" src="https://img.shields.io/maven-central/v/io.github.cdklabs/cdknag"></a>
 * <a href="https://www.nuget.org/packages/Cdklabs.CdkNag"><img alt="NuGet version" src="https://img.shields.io/nuget/v/Cdklabs.CdkNag"></a>
 * <a href="https://github.com/cdklabs/cdk-nag-go"><img alt="Go version" src="https://img.shields.io/github/go-mod/go-version/cdklabs/cdk-nag-go?color=blue&amp;filename=cdknag%2Fgo.mod"></a>
 * <p>
 * <a href="https://constructs.dev/packages/cdk-nag"><img alt="View on Construct Hub" src="https://constructs.dev/badge?package=cdk-nag"></a>
 * <p>
 * Check CDK applications or <a href="#using-on-cloudformation-templates">CloudFormation templates</a> for best practices using a combination of available rule packs. Inspired by <a href="https://github.com/stelligent/cfn_nag">cfn_nag</a>.
 * <p>
 * Check out <a href="https://aws.amazon.com/blogs/devops/manage-application-security-and-compliance-with-the-aws-cloud-development-kit-and-cdk-nag/">this blog post</a> for a guided overview!
 * <p>
 * <img alt="demo" src="cdk_nag.gif">
 * <p>
 * <h2>Available Rules and Packs</h2>
 * <p>
 * See <a href="./RULES.md">RULES</a> for more information on all the available packs.
 * <p>
 * <ol>
 * <li><a href="./RULES.md#awssolutions">AWS Solutions</a></li>
 * <li><a href="./RULES.md#hipaa-security">HIPAA Security</a></li>
 * <li><a href="./RULES.md#nist-800-53-rev-4">NIST 800-53 rev 4</a></li>
 * <li><a href="./RULES.md#nist-800-53-rev-5">NIST 800-53 rev 5</a></li>
 * <li><a href="./RULES.md#pci-dss-321">PCI DSS 3.2.1</a></li>
 * </ol>
 * <p>
 * <a href="./RULES.md">RULES</a> also includes a collection of <a href="./RULES.md#additional-rules">additional rules</a> that are not currently included in any of the pre-built NagPacks, but are still available for inclusion in custom NagPacks.
 * <p>
 * Read the <a href="./docs/NagPack.md">NagPack developer docs</a> if you are interested in creating your own pack.
 * <p>
 * <h2>Usage</h2>
 * <p>
 * For a full list of options See <code>NagPackProps</code> in the <a href="./API.md#struct-nagpackprops">API.md</a>
 * <p>
 * <details>
 * <summary>Including in an application</summary>
 * <p>
 * <blockquote><pre>
 * import { App, Aspects } from 'aws-cdk-lib';
 * import { CdkTestStack } from '../lib/cdk-test-stack';
 * import { AwsSolutionsChecks } from 'cdk-nag';
 * 
 * const app = new App();
 * new CdkTestStack(app, 'CdkNagDemo');
 * // Simple rule informational messages
 * Aspects.of(app).add(new AwsSolutionsChecks());
 * // Additional explanations on the purpose of triggered rules
 * // Aspects.of(stack).add(new AwsSolutionsChecks({ verbose: true }));
 * </pre></blockquote>
 * <p>
 * </details>
 * <p>
 * <h2>Suppressing a Rule</h2>
 * <p>
 * <details>
 *   <summary>Example 1) Default Construct</summary>
 * <p>
 * <blockquote><pre>
 * import { SecurityGroup, Vpc, Peer, Port } from 'aws-cdk-lib/aws-ec2';
 * import { Stack, StackProps } from 'aws-cdk-lib';
 * import { Construct } from 'constructs';
 * import { NagSuppressions } from 'cdk-nag';
 * 
 * export class CdkTestStack extends Stack {
 *   constructor(scope: Construct, id: string, props?: StackProps) {
 *     super(scope, id, props);
 *     const test = new SecurityGroup(this, 'test', {
 *       vpc: new Vpc(this, 'vpc'),
 *     });
 *     test.addIngressRule(Peer.anyIpv4(), Port.allTraffic());
 *     NagSuppressions.addResourceSuppressions(test, [
 *       { id: 'AwsSolutions-EC23', reason: 'lorem ipsum' },
 *     ]);
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * </details><details>
 *   <summary>Example 2) On Multiple Constructs</summary>
 * <p>
 * <blockquote><pre>
 * import { SecurityGroup, Vpc, Peer, Port } from 'aws-cdk-lib/aws-ec2';
 * import { Stack, StackProps } from 'aws-cdk-lib';
 * import { Construct } from 'constructs';
 * import { NagSuppressions } from 'cdk-nag';
 * 
 * export class CdkTestStack extends Stack {
 *   constructor(scope: Construct, id: string, props?: StackProps) {
 *     super(scope, id, props);
 *     const vpc = new Vpc(this, 'vpc');
 *     const test1 = new SecurityGroup(this, 'test', { vpc });
 *     test1.addIngressRule(Peer.anyIpv4(), Port.allTraffic());
 *     const test2 = new SecurityGroup(this, 'test', { vpc });
 *     test2.addIngressRule(Peer.anyIpv4(), Port.allTraffic());
 *     NagSuppressions.addResourceSuppressions(
 *       [test1, test2],
 *       [{ id: 'AwsSolutions-EC23', reason: 'lorem ipsum' }]
 *     );
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * </details><details>
 *   <summary>Example 3) Child Constructs</summary>
 * <p>
 * <blockquote><pre>
 * import { User, PolicyStatement } from 'aws-cdk-lib/aws-iam';
 * import { Stack, StackProps } from 'aws-cdk-lib';
 * import { Construct } from 'constructs';
 * import { NagSuppressions } from 'cdk-nag';
 * 
 * export class CdkTestStack extends Stack {
 *   constructor(scope: Construct, id: string, props?: StackProps) {
 *     super(scope, id, props);
 *     const user = new User(this, 'rUser');
 *     user.addToPolicy(
 *       new PolicyStatement({
 *         actions: ['s3:PutObject'],
 *         resources: ['arn:aws:s3:::bucket_name/*'],
 *       })
 *     );
 *     // Enable adding suppressions to child constructs
 *     NagSuppressions.addResourceSuppressions(
 *       user,
 *       [
 *         {
 *           id: 'AwsSolutions-IAM5',
 *           reason: 'lorem ipsum',
 *           appliesTo: ['Resource::arn:aws:s3:::bucket_name/*'], // optional
 *         },
 *       ],
 *       true
 *     );
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * </details><details>
 *   <summary>Example 4) Stack Level </summary>
 * <p>
 * <blockquote><pre>
 * import { App, Aspects } from 'aws-cdk-lib';
 * import { CdkTestStack } from '../lib/cdk-test-stack';
 * import { AwsSolutionsChecks, NagSuppressions } from 'cdk-nag';
 * 
 * const app = new App();
 * const stack = new CdkTestStack(app, 'CdkNagDemo');
 * Aspects.of(app).add(new AwsSolutionsChecks());
 * NagSuppressions.addStackSuppressions(stack, [
 *   { id: 'AwsSolutions-EC23', reason: 'lorem ipsum' },
 * ]);
 * </pre></blockquote>
 * <p>
 * </details><details>
 *   <summary>Example 5) Construct path</summary>
 * <p>
 * If you received the following error on synth/deploy
 * <p>
 * <blockquote><pre>
 * [Error at /StackName/Custom::CDKBucketDeployment8675309/ServiceRole/Resource] AwsSolutions-IAM4: The IAM user, role, or group uses AWS managed policies
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * import { Bucket } from 'aws-cdk-lib/aws-s3';
 * import { BucketDeployment } from 'aws-cdk-lib/aws-s3-deployment';
 * import { Stack, StackProps } from 'aws-cdk-lib';
 * import { Construct } from 'constructs';
 * import { NagSuppressions } from 'cdk-nag';
 * 
 * export class CdkTestStack extends Stack {
 *   constructor(scope: Construct, id: string, props?: StackProps) {
 *     super(scope, id, props);
 *     new BucketDeployment(this, 'rDeployment', {
 *       sources: [],
 *       destinationBucket: Bucket.fromBucketName(this, 'rBucket', 'foo'),
 *     });
 *     NagSuppressions.addResourceSuppressionsByPath(
 *       this,
 *       '/StackName/Custom::CDKBucketDeployment8675309/ServiceRole/Resource',
 *       [{ id: 'AwsSolutions-IAM4', reason: 'at least 10 characters' }]
 *     );
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * </details><details>
 *   <summary>Example 6) Granular Suppressions of findings</summary>
 * <p>
 * Certain rules support granular suppressions of <code>findings</code>. If you received the following errors on synth/deploy
 * <p>
 * <blockquote><pre>
 * [Error at /StackName/rFirstUser/DefaultPolicy/Resource] AwsSolutions-IAM5[Action::s3:*]: The IAM entity contains wildcard permissions and does not have a cdk-nag rule suppression with evidence for those permission.
 * [Error at /StackName/rFirstUser/DefaultPolicy/Resource] AwsSolutions-IAM5[Resource::*]: The IAM entity contains wildcard permissions and does not have a cdk-nag rule suppression with evidence for those permission.
 * [Error at /StackName/rSecondUser/DefaultPolicy/Resource] AwsSolutions-IAM5[Action::s3:*]: The IAM entity contains wildcard permissions and does not have a cdk-nag rule suppression with evidence for those permission.
 * [Error at /StackName/rSecondUser/DefaultPolicy/Resource] AwsSolutions-IAM5[Resource::*]: The IAM entity contains wildcard permissions and does not have a cdk-nag rule suppression with evidence for those permission.
 * </pre></blockquote>
 * <p>
 * By applying the following suppressions
 * <p>
 * <blockquote><pre>
 * import { User } from 'aws-cdk-lib/aws-iam';
 * import { Stack, StackProps } from 'aws-cdk-lib';
 * import { Construct } from 'constructs';
 * import { NagSuppressions } from 'cdk-nag';
 * 
 * export class CdkTestStack extends Stack {
 *   constructor(scope: Construct, id: string, props?: StackProps) {
 *     super(scope, id, props);
 *     const firstUser = new User(this, 'rFirstUser');
 *     firstUser.addToPolicy(
 *       new PolicyStatement({
 *         actions: ['s3:*'],
 *         resources: ['*'],
 *       })
 *     );
 *     const secondUser = new User(this, 'rSecondUser');
 *     secondUser.addToPolicy(
 *       new PolicyStatement({
 *         actions: ['s3:*'],
 *         resources: ['*'],
 *       })
 *     );
 *     const thirdUser = new User(this, 'rSecondUser');
 *     thirdUser.addToPolicy(
 *       new PolicyStatement({
 *         actions: ['sqs:CreateQueue'],
 *         resources: [`arn:aws:sqs:${this.region}:${this.account}:*`],
 *       })
 *     );
 *     NagSuppressions.addResourceSuppressions(
 *       firstUser,
 *       [
 *         {
 *           id: 'AwsSolutions-IAM5',
 *           reason:
 *             "Only suppress AwsSolutions-IAM5 's3:*' finding on First User.",
 *           appliesTo: ['Action::s3:*'],
 *         },
 *       ],
 *       true
 *     );
 *     NagSuppressions.addResourceSuppressions(
 *       secondUser,
 *       [
 *         {
 *           id: 'AwsSolutions-IAM5',
 *           reason: 'Suppress all AwsSolutions-IAM5 findings on Second User.',
 *         },
 *       ],
 *       true
 *     );
 *     NagSuppressions.addResourceSuppressions(
 *       thirdUser,
 *       [
 *         {
 *           id: 'AwsSolutions-IAM5',
 *           reason: 'Suppress AwsSolutions-IAM5 on the SQS resource.',
 *           appliesTo: [
 *             {
 *               regex: '/^Resource::arn:aws:sqs:(.*):\\*$/g',
 *             },
 *           ],
 *         },
 *       ],
 *       true
 *     );
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * You would see the following error on synth/deploy
 * <p>
 * <blockquote><pre>
 * [Error at /StackName/rFirstUser/DefaultPolicy/Resource] AwsSolutions-IAM5[Resource::*]: The IAM entity contains wildcard permissions and does not have a cdk-nag rule suppression with evidence for those permission.
 * </pre></blockquote>
 * <p>
 * </details>
 * <p>
 * <h2>Suppressing <code>aws-cdk-lib/pipelines</code> Violations</h2>
 * <p>
 * The <a href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.pipelines.CodePipeline.html">aws-cdk-lib/pipelines.CodePipeline</a> construct and its child constructs are not guaranteed to be "Visited" by <code>Aspects</code>, as they are not added during the "Construction" phase of the <a href="https://docs.aws.amazon.com/cdk/v2/guide/apps.html#lifecycle">cdk lifecycle</a>. Because of this behavior, you may experience problems such as rule violations not appearing or the inability to suppress violations on these constructs.
 * <p>
 * You can remediate these rule violation and suppression problems by forcing the pipeline construct creation forward by calling <code>.buildPipeline()</code> on your <code>CodePipeline</code> object. Otherwise you may see errors such as:
 * <p>
 * <blockquote><pre>
 * Error: Suppression path "/this/construct/path" did not match any resource. This can occur when a resource does not exist or if a suppression is applied before a resource is created.
 * </pre></blockquote>
 * <p>
 * See <a href="https://github.com/aws/aws-cdk/issues/18440">this issue</a> for more information.
 * <p>
 * <details>
 *   <summary>Example) Supressing Violations in Pipelines</summary>
 * <p>
 * <code>example-app.ts</code>
 * <p>
 * <blockquote><pre>
 * import { App, Aspects } from 'aws-cdk-lib';
 * import { AwsSolutionsChecks } from 'cdk-nag';
 * import { ExamplePipeline } from '../lib/example-pipeline';
 * 
 * const app = new App();
 * new ExamplePipeline(app, 'example-cdk-pipeline');
 * Aspects.of(app).add(new AwsSolutionsChecks({ verbose: true }));
 * app.synth();
 * </pre></blockquote>
 * <p>
 * <code>example-pipeline.ts</code>
 * <p>
 * <blockquote><pre>
 * import { Stack, StackProps } from 'aws-cdk-lib';
 * import { Repository } from 'aws-cdk-lib/aws-codecommit';
 * import {
 *   CodePipeline,
 *   CodePipelineSource,
 *   ShellStep,
 * } from 'aws-cdk-lib/pipelines';
 * import { NagSuppressions } from 'cdk-nag';
 * import { Construct } from 'constructs';
 * 
 * export class ExamplePipeline extends Stack {
 *   constructor(scope: Construct, id: string, props?: StackProps) {
 *     super(scope, id, props);
 * 
 *     const exampleSynth = new ShellStep('ExampleSynth', {
 *       commands: ['yarn build --frozen-lockfile'],
 *       input: CodePipelineSource.codeCommit(
 *         new Repository(this, 'ExampleRepo', { repositoryName: 'ExampleRepo' }),
 *         'main'
 *       ),
 *     });
 * 
 *     const ExamplePipeline = new CodePipeline(this, 'ExamplePipeline', {
 *       synth: exampleSynth,
 *     });
 * 
 *     // Force the pipeline construct creation forward before applying suppressions.
 *     // &#64;See https://github.com/aws/aws-cdk/issues/18440
 *     ExamplePipeline.buildPipeline();
 * 
 *     // The path suppression will error if you comment out "ExamplePipeline.buildPipeline();""
 *     NagSuppressions.addResourceSuppressionsByPath(
 *       this,
 *       '/example-cdk-pipeline/ExamplePipeline/Pipeline/ArtifactsBucket/Resource',
 *       [
 *         {
 *           id: 'AwsSolutions-S1',
 *           reason: 'Because I said so',
 *         },
 *       ]
 *     );
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * </details>
 * <p>
 * <h2>Rules and Property Overrides</h2>
 * <p>
 * In some cases L2 Constructs do not have a native option to remediate an issue and must be fixed via <a href="https://docs.aws.amazon.com/cdk/latest/guide/cfn_layer.html#cfn_layer_raw">Raw Overrides</a>. Since raw overrides take place after template synthesis these fixes are not caught by cdk-nag. In this case you should remediate the issue and suppress the issue like in the following example.
 * <p>
 * <details>
 *   <summary>Example) Property Overrides</summary>
 * <p>
 * <blockquote><pre>
 * import {
 *   Instance,
 *   InstanceType,
 *   InstanceClass,
 *   MachineImage,
 *   Vpc,
 *   CfnInstance,
 * } from 'aws-cdk-lib/aws-ec2';
 * import { Stack, StackProps } from 'aws-cdk-lib';
 * import { Construct } from 'constructs';
 * import { NagSuppressions } from 'cdk-nag';
 * 
 * export class CdkTestStack extends Stack {
 *   constructor(scope: Construct, id: string, props?: StackProps) {
 *     super(scope, id, props);
 *     const instance = new Instance(this, 'rInstance', {
 *       vpc: new Vpc(this, 'rVpc'),
 *       instanceType: new InstanceType(InstanceClass.T3),
 *       machineImage: MachineImage.latestAmazonLinux(),
 *     });
 *     const cfnIns = instance.node.defaultChild as CfnInstance;
 *     cfnIns.addPropertyOverride('DisableApiTermination', true);
 *     NagSuppressions.addResourceSuppressions(instance, [
 *       {
 *         id: 'AwsSolutions-EC29',
 *         reason: 'Remediated through property override.',
 *       },
 *     ]);
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * </details>
 * <p>
 * <h2>Conditionally Ignoring Suppressions</h2>
 * <p>
 * You can optionally create a condition that prevents certain rules from being suppressed. You can create conditions for any variety of reasons. Examples include a condition that always ignores a suppression, a condition that ignores a suppression based on the date, a condition that ignores a suppression based on the reason. You can read <a href="./docs/IgnoreSuppressionConditions.md">the developer docs</a> for more information on creating your own conditions.
 * <p>
 * <details>
 *   <summary>Example) Using the pre-built `SuppressionIgnoreErrors` class to ignore suppressions on any `Error` level rules.</summary>
 * <p>
 * <blockquote><pre>
 * import { App, Aspects } from 'aws-cdk-lib';
 * import { CdkTestStack } from '../lib/cdk-test-stack';
 * import { AwsSolutionsChecks, SuppressionIgnoreErrors } from 'cdk-nag';
 * 
 * const app = new App();
 * new CdkTestStack(app, 'CdkNagDemo');
 * // Ignore Suppressions on any errors
 * Aspects.of(app).add(
 *   new AwsSolutionsChecks({
 *     suppressionIgnoreCondition: new SuppressionIgnoreErrors(),
 *   })
 * );
 * </pre></blockquote>
 * <p>
 * </details>
 * <p>
 * <h2>Customizing Logging</h2>
 * <p>
 * <code>NagLogger</code>s give <code>NagPack</code> authors and users the ability to create their own custom reporting mechanisms. All pre-built <code>NagPacks</code>come with the <code>AnnotationsLogger</code>and the <code>NagReportLogger</code> (with CSV reports) enabled by default.
 * <p>
 * See the <a href="./docs/NagLogger.md">NagLogger</a> developer docs for more information.
 * <p>
 * <details>
 *   <summary>Example) Adding the `ExtremelyHelpfulConsoleLogger` example from the NagLogger docs</summary>
 * <p>
 * <blockquote><pre>
 * import { App, Aspects } from 'aws-cdk-lib';
 * import { CdkTestStack } from '../lib/cdk-test-stack';
 * import { ExtremelyHelpfulConsoleLogger } from './docs/NagLogger';
 * import { AwsSolutionsChecks } from 'cdk-nag';
 * 
 * const app = new App();
 * new CdkTestStack(app, 'CdkNagDemo');
 * Aspects.of(app).add(
 *   new AwsSolutionsChecks({
 *     additionalLoggers: [new ExtremelyHelpfulConsoleLogger()],
 *   })
 * );
 * </pre></blockquote>
 * <p>
 * </details>
 * <p>
 * <h2>Using on CloudFormation templates</h2>
 * <p>
 * You can use cdk-nag on existing CloudFormation templates by using the <a href="https://docs.aws.amazon.com/cdk/latest/guide/use_cfn_template.html#use_cfn_template_install">cloudformation-include</a> module.
 * <p>
 * <details>
 *   <summary>Example 1) CloudFormation template with suppression</summary>
 * <p>
 * Sample CloudFormation template with suppression
 * <p>
 * <blockquote><pre>
 * {
 *   "Resources": {
 *     "rBucket": {
 *       "Type": "AWS::S3::Bucket",
 *       "Properties": {
 *         "BucketName": "some-bucket-name"
 *       },
 *       "Metadata": {
 *         "cdk_nag": {
 *           "rules_to_suppress": [
 *             {
 *               "id": "AwsSolutions-S1",
 *               "reason": "at least 10 characters"
 *             }
 *           ]
 *         }
 *       }
 *     }
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * Sample App
 * <p>
 * <blockquote><pre>
 * import { App, Aspects } from 'aws-cdk-lib';
 * import { CdkTestStack } from '../lib/cdk-test-stack';
 * import { AwsSolutionsChecks } from 'cdk-nag';
 * 
 * const app = new App();
 * new CdkTestStack(app, 'CdkNagDemo');
 * Aspects.of(app).add(new AwsSolutionsChecks());
 * </pre></blockquote>
 * <p>
 * Sample Stack with imported template
 * <p>
 * <blockquote><pre>
 * import { CfnInclude } from 'aws-cdk-lib/cloudformation-include';
 * import { NagSuppressions } from 'cdk-nag';
 * import { Stack, StackProps } from 'aws-cdk-lib';
 * import { Construct } from 'constructs';
 * 
 * export class CdkTestStack extends Stack {
 *   constructor(scope: Construct, id: string, props?: StackProps) {
 *     super(scope, id, props);
 *     new CfnInclude(this, 'Template', {
 *       templateFile: 'my-template.json',
 *     });
 *     // Add any additional suppressions
 *     NagSuppressions.addResourceSuppressionsByPath(
 *       this,
 *       '/CdkNagDemo/Template/rBucket',
 *       [
 *         {
 *           id: 'AwsSolutions-S2',
 *           reason: 'at least 10 characters',
 *         },
 *       ]
 *     );
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * </details><details>
 *   <summary>Example 2) CloudFormation template with granular suppressions</summary>
 * <p>
 * Sample CloudFormation template with suppression
 * <p>
 * <blockquote><pre>
 * {
 *   "Resources": {
 *     "myPolicy": {
 *       "Type": "AWS::IAM::Policy",
 *       "Properties": {
 *         "PolicyDocument": {
 *           "Statement": [
 *             {
 *               "Action": [
 *                 "kms:Decrypt",
 *                 "kms:DescribeKey",
 *                 "kms:Encrypt",
 *                 "kms:ReEncrypt*",
 *                 "kms:GenerateDataKey*"
 *               ],
 *               "Effect": "Allow",
 *               "Resource": ["some-key-arn"]
 *             }
 *           ],
 *           "Version": "2012-10-17"
 *         }
 *       },
 *       "Metadata": {
 *         "cdk_nag": {
 *           "rules_to_suppress": [
 *             {
 *               "id": "AwsSolutions-IAM5",
 *               "reason": "Allow key data access",
 *               "applies_to": [
 *                 "Action::kms:ReEncrypt*",
 *                 "Action::kms:GenerateDataKey*"
 *               ]
 *             }
 *           ]
 *         }
 *       }
 *     }
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * Sample App
 * <p>
 * <blockquote><pre>
 * import { App, Aspects } from 'aws-cdk-lib';
 * import { CdkTestStack } from '../lib/cdk-test-stack';
 * import { AwsSolutionsChecks } from 'cdk-nag';
 * 
 * const app = new App();
 * new CdkTestStack(app, 'CdkNagDemo');
 * Aspects.of(app).add(new AwsSolutionsChecks());
 * </pre></blockquote>
 * <p>
 * Sample Stack with imported template
 * <p>
 * <blockquote><pre>
 * import { CfnInclude } from 'aws-cdk-lib/cloudformation-include';
 * import { NagSuppressions } from 'cdk-nag';
 * import { Stack, StackProps } from 'aws-cdk-lib';
 * import { Construct } from 'constructs';
 * 
 * export class CdkTestStack extends Stack {
 *   constructor(scope: Construct, id: string, props?: StackProps) {
 *     super(scope, id, props);
 *     new CfnInclude(this, 'Template', {
 *       templateFile: 'my-template.json',
 *     });
 *     // Add any additional suppressions
 *     NagSuppressions.addResourceSuppressionsByPath(
 *       this,
 *       '/CdkNagDemo/Template/myPolicy',
 *       [
 *         {
 *           id: 'AwsSolutions-IAM5',
 *           reason: 'Allow key data access',
 *           appliesTo: ['Action::kms:ReEncrypt*', 'Action::kms:GenerateDataKey*'],
 *         },
 *       ]
 *     );
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * </details>
 * <p>
 * <h2>Contributing</h2>
 * <p>
 * See <a href="./CONTRIBUTING.md">CONTRIBUTING</a> for more information.
 * <p>
 * <h2>License</h2>
 * <p>
 * This project is licensed under the Apache-2.0 License.
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package io.github.cdklabs.cdknag;
