package com.cerve.development.data.result

sealed interface CerveResult<out T> {

    data class Success<T>(val value: T): CerveResult<T>
    data class Error<T>(val value: T? = null): CerveResult<T>
    data class Message<T>(val value: String? = null): CerveResult<T>

    infix fun <R> ifError(
        onFalse: () -> R
    ) : R {
        return when(this) {
            is Success -> value as R
            is Error -> onFalse()
            is Message -> value as R
        }
    }

    infix fun <R> ifSuccess(
        onTrue: (T) -> R
    ) : CerveResult<R> {
        return when (this) {
            is Success -> Success(onTrue(value))
            is Error -> Error(value as? R)
            is Message -> Message(value)
        }
    }

    fun dataOrDefault(default: @UnsafeVariance T) : T {
        return if (this is Success) {
            this.value
        } else default
    }

}

inline fun <reified R, T> CerveResult<T>.map(
    transform: T.() -> R = { Unit as R }
): CerveResult<R> = when(this) {
    is CerveResult.Success -> {
        try {
            CerveResult.Success(transform(value))
        } catch (e: Exception) {
            CerveResult.Error(transform(value))
        }
    }
    is CerveResult.Error -> CerveResult.Error(value as R)
    is CerveResult.Message -> CerveResult.Message(value)
}