package com.cerve.development.data.result

import com.cerve.development.data.result.CerveResult

sealed class CerveResult<T>(val data: T?) {

    data class Success<T>(val value: T): CerveResult<T>(value)

    data class Error<T>(
        val value: T? = null,
        val errorMessage: String? = null
    ): CerveResult<T>(value)

    val message = if (this is Error) errorMessage else null

    companion object {

        infix fun <T> CerveResult<T>.dataOrDefault(default: T & Any) : T & Any = data ?: default

        inline infix fun <T, reified R> CerveResult<T>.mapResult(
            onResult: (T?) -> CerveResult<R>
        ) : CerveResult<R> = onResult(data)


        inline infix fun <T> CerveResult<T>.mapToEmpty(
            onResult: T?.() -> Unit = {  }
        ) : CerveResult<Unit> {
            onResult(this.data)

            return when (this) {
                is Success -> Success(value = Unit)
                is Error -> Error(
                    value = Unit,
                    errorMessage = errorMessage
                )
            }
        }

        inline infix fun <T, R> CerveResult<T>.mapIfSuccessResult(
            onResult: T.() -> R
        ) : CerveResult<R?> = when (this) {
            is Success -> Success(onResult(value))
            is Error -> Error(null, errorMessage = errorMessage)
        }

        inline infix fun <T, R> CerveResult<T>.mapIfErrorResult(
            onResult: T?.() -> R
        ) : CerveResult<R?> = when (this) {
            is Success -> Success(null)
            is Error -> Error(onResult(value), errorMessage = errorMessage)
        }

        inline infix fun <T, R> CerveResult<T>.mapIfSuccess(
            onResult: T.() -> R
        ) : R? = when (this) {
            is Success -> onResult(value)
            is Error -> null
        }

        inline infix fun <T, R> CerveResult<T>.mapIfError(
            onResult: T?.() -> R
        ) : R? = when (this) {
            is Success -> null
            is Error -> onResult(value)
        }

        inline infix fun <T, R> CerveResult<T>.ifError(
            onResult: T?.() -> R
        ) : R? = if (this is Error) onResult(value) else null

        infix fun <T, R> CerveResult<T>.ifSuccess(
            onResult: CerveResult<T>.() -> R
        ) = Ternary(
            condition = this is Success,
            onTernaryResult = onResult,
            result = this
        )

        inline infix fun <reified T, R, reified S, reified U> Ternary<CerveResult<T>.() -> R, T>.ifError(
            onResult: T?.(String?) -> S
        ) : U {
            return when {
                condition -> onTernaryResult(result) as U
                result is Error -> onResult(result.value, result.errorMessage) as U
                else -> onResult(result.data, null) as U
            }
        }

        inline fun <reified T> CerveResult<T>.onError(
			condition: (String?) -> Unit
		) : CerveResult<T> {
			return when(this) {
				is Success -> this
				is Error -> {
					condition(errorMessage); this
				}
			}
		}

        inline fun <reified T> CerveResult<T>.onSuccess(
            condition: (T) -> Unit
        ) : CerveResult<T> {
            return when(this) {
                is Success -> {
                    condition(value) ; this
                }
                is Error -> this
            }
        }

    }

}
