package com.cerve.development.ui.canvas.operators

import com.cerve.development.ui.canvas.model.CerveLine
import com.cerve.development.ui.canvas.model.CerveOffset
import kotlin.math.max
import kotlin.math.min

fun CerveLine.checkLineIntersectionsOffset(
    intersectingLine: CerveLine
): Boolean {
    val p1 = start.toStandardOffset()
    val q1 = end.toStandardOffset()
    val p2 = intersectingLine.start.toStandardOffset()
    val q2 = intersectingLine.end.toStandardOffset()

    val o1 = orientationOffset(p1, q1, p2)
    val o2 = orientationOffset(p1, q1, q2)
    val o3 = orientationOffset(p2, q2, p1)
    val o4 = orientationOffset(p2, q2, q1)

    if (o1 != o2 && o3 != o4) {
        return true
    }

    // Special Cases
    if (o1 == 0 && onSegmentOffset(p1, p2, q1)) {
        return true
    }
    if (o2 == 0 && onSegmentOffset(p1, q2, q1)) {
        return true
    }
    if (o3 == 0 && onSegmentOffset(p2, p1, q2)) {
        return true
    }
    if (o4 == 0 && onSegmentOffset(p2, q1, q2)) {
        return true
    }

    return false
}

fun CerveLine.findIntersectionOffset(
    intersectingLine: CerveLine
): CerveOffset? {
    val (x1, y1) = start
    val (x2, y2) = end
    val (x3, y3) = intersectingLine.start
    val (x4, y4) = intersectingLine.end

    // Bounding box check
    if (max(x1, x2) < min(x3, x4) ||
        min(x1, x2) > max(x3, x4) ||
        max(y1, y2) < min(y3, y4) ||
        min(y1, y2) > max(y3, y4)
    ) {
        return null // No intersection
    }

    val denominator = (x1 - x2) * (y3 - y4) - (y1 - y2) * (x3 - x4)
    if (denominator == 0.0) return null

    val numeratorX = (x1 * y2 - y1 * x2) * (x3 - x4) - (x1 - x2) * (x3 * y4 - y3 * x4)
    val numeratorY = (x1 * y2 - y1 * x2) * (y3 - y4) - (y1 - y2) * (x3 * y4 - y3 * x4)

    val intersectionX = numeratorX / denominator
    val intersectionY = numeratorY / denominator

    if (intersectionX < min(x1, x2) || intersectionX > max(x1, x2) ||
        intersectionX < min(x3, x4) || intersectionX > max(x3, x4) ||
        intersectionY < min(y1, y2) || intersectionY > max(y1, y2) ||
        intersectionY < min(y3, y4) || intersectionY > max(y3, y4)
    ) {
        return null
    }

    return CerveOffset(intersectionX, intersectionY)
}
