package com.cerve.development.ui.canvas.operators

import androidx.compose.foundation.gestures.detectDragGestures
import androidx.compose.foundation.gestures.detectTransformGestures
import androidx.compose.runtime.State
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.input.pointer.PointerInputScope
import com.cerve.development.ui.canvas.model.CerveLine
import com.cerve.development.ui.canvas.model.CerveLine.Companion.defaultCerveLine
import com.cerve.development.ui.canvas.model.CerveOffset
import com.cerve.development.ui.canvas.model.CerveOffset.Companion.toCerveOffset

suspend fun PointerInputScope.zoomGestures(
    transformationScale: State<Float>,
    transformationRotation: State<Float>,
    translationOffset: State<Offset>,
    onChangeTransformationScale: (Float) -> Unit,
    onChangeTransformationRotation: (Float) -> Unit,
    onChangeTransformationOffset: (Offset) -> Unit
) = detectTransformGestures { centroid, offsetChange, zoomChange, rotationChange ->

    onChangeTransformationScale(transformationScale.value * zoomChange)
    onChangeTransformationRotation(transformationRotation.value + rotationChange)
    onChangeTransformationOffset(translationOffset.value + offsetChange)

    // adjust bounds based on inverse scale
    val bounds = 1000f * (1f / transformationScale.value)

    onChangeTransformationOffset(
        Offset(
            x = translationOffset.value.x.coerceIn(-bounds, bounds),
            y = translationOffset.value.y.coerceIn(-bounds, bounds)
        )
    )
}

suspend fun PointerInputScope.eraserGestures(
    stepSize: Float,
    currentLines: MutableList<CerveLine>,
    eraserLines: MutableList<CerveLine>
) = detectDragGestures(
    onDrag = { change, dragAmount ->
        eraserLines.consumeLine(change, dragAmount)
    },
    onDragEnd = {
        val x = eraserLines.first().start.x
        val y = eraserLines.first().start.y

        val xx = eraserLines.last().start.x
        val yy = eraserLines.last().start.y

        val eraserLine = defaultCerveLine(
            start = CerveOffset(x, y),
            end = CerveOffset(xx, yy)
        )

        currentLines.removeAll { line ->
            line.checkLineIntersectionsOffset(eraserLine)
        }

        currentLines.removeAll { line ->
            !line.travelsAtLeast(stepSize.div(6))
        }

        eraserLines.clear()
    }

)

suspend fun PointerInputScope.assistedBrushGestures(
    stepSize: Float,
    currentLines: MutableList<CerveLine>,
    currentLineCandidates: MutableList<CerveLine>
) = detectDragGestures(
    onDragStart = { offset ->
        val cerveOffset = offset.toCerveOffset()
        val line = CerveLine(start = cerveOffset, end = cerveOffset)

        currentLineCandidates.add(line)
    },
    onDrag = { change, dragAmount ->
        val lastIndex = currentLineCandidates.lastIndex
        val candidateLine = currentLineCandidates.last()
        val newCandidateLine = candidateLine.copy(end = change.position.toCerveOffset())
        currentLineCandidates[lastIndex] = newCandidateLine
    },
    onDragEnd = {

        val redrawnLines = currentLineCandidates
            .map { line -> line.roundToNearest(stepSize) }
            .distinctUntilChanged()

        currentLineCandidates.clear()
        currentLines.addAll(redrawnLines)

    }
)

suspend fun PointerInputScope.segmentedBrushGestures(
    stepSize: Float,
    currentLines: MutableList<CerveLine>,
    currentLineCandidates: MutableList<CerveLine>
) = detectDragGestures(
    onDrag = { change, dragAmount ->
        currentLineCandidates.consumeLine(change, dragAmount)
    },
    onDragEnd = {

        val roundedLine = currentLineCandidates
            .map { line -> line.roundToNearest(stepSize) }
            .distinctUntilChanged()

        roundedLine.forEach { newLine ->
            updateLinesOnPlane(
                allLines = currentLines,
                newLine = newLine
            )
        }
        currentLineCandidates.clear()

    }
)

suspend fun PointerInputScope.brushGestures(
    currentLines: MutableList<CerveLine>
) = detectDragGestures(
    onDrag = { change, dragAmount ->
        currentLines.consumeLine(change, dragAmount)
    }
)

