package com.cerve.development.ui.canvas.operators

import androidx.compose.foundation.gestures.detectDragGestures
import androidx.compose.foundation.gestures.detectTransformGestures
import androidx.compose.runtime.MutableState
import androidx.compose.runtime.State
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.input.pointer.PointerInputScope
import com.cerve.development.ui.canvas.model.CerveLine
import com.cerve.development.ui.canvas.model.CerveLine.Companion.toCerveLine
import com.cerve.development.ui.canvas.model.CerveOffset.Companion.toCerveOffset

suspend fun PointerInputScope.zoomGestures(
    transformationScale: State<Float>,
    transformationRotation: State<Float>,
    translationOffset: State<Offset>,
    onChangeTransformationScale: (Float?) -> Unit,
    onChangeTransformationRotation: (Float?) -> Unit,
    onChangeTransformationOffset: (Offset?) -> Unit
) = detectTransformGestures { centroid, offsetChange, zoomChange, rotationChange ->

    onChangeTransformationScale(transformationScale.value * zoomChange)
    onChangeTransformationRotation(transformationRotation.value + rotationChange)
    onChangeTransformationOffset(translationOffset.value + offsetChange)

    // adjust bounds based on inverse scale
    val bounds = 1000f * (1f / transformationScale.value)

    onChangeTransformationOffset(
        Offset(
            x = translationOffset.value.x.coerceIn(-bounds, bounds),
            y = translationOffset.value.y.coerceIn(-bounds, bounds)
        )
    )
}


suspend fun PointerInputScope.eraserGestures(
    eraserRadius: Float,
    eraserCenter: MutableState<Offset?>,
    currentLines: MutableList<CerveLine>
) = detectDragGestures(
    onDragStart = { offset ->
        eraserCenter.value = offset
    },
    onDrag = { change, dragAmount ->

        eraserCenter.value?.let { center ->

            val position = center + dragAmount

            eraserCenter.value = position

            currentLines.removeAll { line ->
                checkCircleSegmentIntersection(
                    circleCenter = position,
                    circleRadius = eraserRadius,
                    segmentStart = line.start.toStandardOffset(),
                    segmentEnd = line.end.toStandardOffset()
                )
            }

        }

    },
    onDragEnd = { eraserCenter.value = null }
)

suspend fun PointerInputScope.assistedBrushGestures(
    gridSpacing: Float,
    currentLines: MutableList<CerveLine>,
    currentLineCandidates: MutableList<CerveLine>
) = detectDragGestures(
    onDragStart = { offset ->
        val line = offset
            .toCerveOffset()
            .snapToGrid(gridSpacing)
            .toCerveLine()

        currentLineCandidates.add(line)
    },
    onDrag = { change, dragAmount ->
        val lastIndex = currentLineCandidates.lastIndex
        val candidateLine = currentLineCandidates.last()
        val end = change.position.snapToGrid(gridSpacing)

        val newCandidateLine = candidateLine.copy(end = end)
        currentLineCandidates[lastIndex] = newCandidateLine
    },
    onDragEnd = {

        val redrawnLines = currentLineCandidates
            .map { line -> line }
            .distinctUntilChanged()

        currentLineCandidates.clear()
        currentLines.addAll(redrawnLines)

    }
)

suspend fun PointerInputScope.segmentedBrushGestures(
    gridSpacing: Float,
    currentLines: MutableList<CerveLine>,
    currentLineCandidates: MutableList<CerveLine>
) = detectDragGestures(
    onDrag = { change, dragAmount ->
        currentLineCandidates.consumeLine(change, dragAmount)
    },
    onDragEnd = {

        val roundedLine = currentLineCandidates
            .map { line -> line.snapToGrid(gridSpacing) }
            .distinctUntilChanged()

        roundedLine.forEach { newLine ->
            updateLinesOnPlane(
                allLines = currentLines,
                newLine = newLine
            )
        }
        currentLineCandidates.clear()

    }
)

suspend fun PointerInputScope.brushGestures(
    currentLines: MutableList<CerveLine>
) = detectDragGestures(
    onDrag = { change, dragAmount ->
        currentLines.consumeLine(change, dragAmount)
    }
)

