package com.cerve.development.ui.canvas.operators

import androidx.compose.foundation.gestures.detectDragGestures
import androidx.compose.ui.input.pointer.PointerInputScope
import com.cerve.development.ui.canvas.model.CerveLine
import com.cerve.development.ui.canvas.model.CerveOffset
import com.cerve.development.ui.canvas.model.CerveOffset.Companion.offset

suspend fun PointerInputScope.eraserGestures(
    eraserRadius: Float,
    translationAmount: CerveOffset,
    eraserCenter: () -> CerveOffset?,
    currentLines: MutableList<CerveLine>,
    onChangeEraserCenter: (CerveOffset?) -> Unit
) = detectDragGestures(
    onDragStart = { startPosition ->

        val position = startPosition - translationAmount.offset
        onChangeEraserCenter(position.offset)

    },
    onDrag = { change, dragAmount ->

        eraserCenter()?.offset?.let { center ->

            val position = (change.position - translationAmount.offset)
            onChangeEraserCenter(position.offset)

            currentLines.removeAll { line ->
                checkCircleSegmentIntersection(
                    circleCenter = position,
                    circleRadius = eraserRadius,
                    segmentStart = line.start.offset,
                    segmentEnd =  line.end.offset
                )
            }

        }

    },
    onDragEnd = { onChangeEraserCenter(null) }
)

suspend fun PointerInputScope.assistedBrushGestures(
    gridSpacing: Int,
    translationAmount: CerveOffset,
    currentLines: MutableList<CerveLine>,
    currentLineCandidates: MutableList<CerveLine>
) = detectDragGestures(
    onDragStart = { startPosition ->

        val position = startPosition - translationAmount.offset
        val line = CerveLine.Zero.plus(position)
            .snapToNear(spacing = gridSpacing)

        currentLineCandidates.add(line)
    },
    onDrag = { change, _ ->

        val lastIndex = currentLineCandidates.lastIndex
        val candidateLine = currentLineCandidates.last()

        val end = (change.position - translationAmount.offset)
            .snapToNear(spacing = gridSpacing)

        currentLineCandidates[lastIndex] = candidateLine.copy(end = end)
    },
    onDragEnd = {

        val redrawnLines = currentLineCandidates
            .snapDistinctCandidateLine(spacing = gridSpacing)

        currentLineCandidates.clear()
        currentLines.addAll(redrawnLines)

    }
)

suspend fun PointerInputScope.segmentedBrushGestures(
    gridSpacing: Int,
    translationAmount: CerveOffset,
    currentLines: MutableList<CerveLine>,
    currentLineCandidates: MutableList<CerveLine>
) = detectDragGestures(
    onDrag = { change, dragAmount ->
        currentLineCandidates.consumeLine(
            change = change,
            dragAmount = dragAmount,
            translationAmount = translationAmount.offset
        )
    },
    onDragEnd = {

        val filteredRoundedLine = currentLineCandidates
            .snapDistinctCandidateLine(spacing = gridSpacing)

        currentLines.addAll(filteredRoundedLine)
        currentLineCandidates.clear()

    }
)

suspend fun PointerInputScope.brushGestures(
    translationAmount: CerveOffset,
    currentLines: MutableList<CerveLine>
) = detectDragGestures(
    onDrag = { change, dragAmount ->
        currentLines.consumeLine(
            change = change,
            dragAmount = dragAmount,
            translationAmount = translationAmount.offset
        )
    }
)

