package com.cerve.development.ui.canvas.operators.draw

import androidx.compose.ui.Modifier
import androidx.compose.ui.input.pointer.pointerInput
import com.cerve.development.ui.canvas.model.CerveCanvasEraserProperties
import com.cerve.development.ui.canvas.model.CerveCanvasGridProperties
import com.cerve.development.ui.canvas.model.CerveCanvasInteractionType
import com.cerve.development.ui.canvas.model.CerveCanvasInteractionType.CellMultiSelector
import com.cerve.development.ui.canvas.model.CerveCanvasInteractionType.CellSingleSelector
import com.cerve.development.ui.canvas.model.CerveCanvasInteractionType.AssistedBrush
import com.cerve.development.ui.canvas.model.CerveCanvasInteractionType.SegmentedBrush
import com.cerve.development.ui.canvas.model.CerveCanvasInteractionType.Eraser
import com.cerve.development.ui.canvas.model.CerveCell
import com.cerve.development.ui.canvas.model.CerveLine
import com.cerve.development.ui.canvas.model.CerveOffset
import com.cerve.development.ui.canvas.model.CerveSize
import com.cerve.development.ui.canvas.operators.guestures.assistedBrushGestures
import com.cerve.development.ui.canvas.operators.guestures.brushGestures
import com.cerve.development.ui.canvas.operators.guestures.eraserGestures
import com.cerve.development.ui.canvas.operators.guestures.multiSelectorGestures
import com.cerve.development.ui.canvas.operators.guestures.segmentedBrushGestures
import com.cerve.development.ui.canvas.operators.guestures.singleSelectorGestures
import com.cerve.development.ui.canvas.operators.snapToFloor

fun Modifier.canvasInput(
    interactionType: CerveCanvasInteractionType?,
    drawnLines: MutableList<CerveLine>,
    drawnLineCandidates: MutableList<CerveLine>,
    selectedGridCells: MutableList<CerveCell>,
    gridProperties: CerveCanvasGridProperties,
    eraserProperties: () -> CerveCanvasEraserProperties,
    onUpdateEraserProperties: (CerveCanvasEraserProperties) -> Unit,
) = pointerInput(interactionType) {
    when(interactionType) {
        CerveCanvasInteractionType.Brush -> {
            brushGestures(
                translationAmount = gridProperties.center,
                currentLines = drawnLines
            )
        }
        AssistedBrush -> {
            assistedBrushGestures(
                gridSpacing = gridProperties.spacing,
                translationAmount = gridProperties.center,
                currentLines = drawnLines,
                currentLineCandidates = drawnLineCandidates,
            )
        }
        SegmentedBrush -> {
            segmentedBrushGestures(
                gridSpacing = gridProperties.spacing,
                translationAmount = gridProperties.center,
                currentLines = drawnLines,
                currentLineCandidates = drawnLineCandidates
            )
        }
        Eraser -> {
            eraserGestures(
                translationAmount = gridProperties.center,
                currentLines = drawnLines,
                eraserProperties = eraserProperties,
                onUpdateEraserProperties = onUpdateEraserProperties
            )
        }
        CellMultiSelector -> {
            multiSelectorGestures(
                translationAmount = gridProperties.center,
                drawnLines = drawnLines,
                selectedGridCells = selectedGridCells,
                selectableDimensions = gridProperties.dimensions
            )
        }
        CellSingleSelector -> {
            singleSelectorGestures(
                translationAmount = gridProperties.center,
                drawnLines = drawnLines,
                selectedGridCells = selectedGridCells,
                selectableDimensions = gridProperties.dimensions
            )
        }
        else -> Unit
    }
}

fun Modifier.gridCellSelectorInput(
    interactionType: CerveCanvasInteractionType?,
    gridSpacing: Int,
    gridCenter: CerveOffset,
    onUpdateSelectedCellCandidate: (CerveCell?) -> Unit
) : Modifier {
    return pointerInput(interactionType) {
        when (interactionType) {
            CellMultiSelector, CellSingleSelector -> {
                awaitPointerEventScope {
                    while (true) {
                        val event = awaitPointerEvent()
                        val eventPosition = event.changes.firstOrNull()?.position
                        if (eventPosition != null) {
                            val position = (eventPosition - gridCenter.offset)
                                .snapToFloor(gridSpacing)

                            val cell = if (!event.changes.first().pressed) {
                                CerveCell(
                                    topLeft = position,
                                    cellSize = CerveSize(size = gridSpacing)
                                )
                            } else {
                                null
                            }

                            onUpdateSelectedCellCandidate(cell)
                        }
                    }
                }
            }
            else -> Unit
        }
    }
}