package com.cerve.development.ui.canvas.operators

import androidx.compose.foundation.gestures.detectTapGestures
import androidx.compose.foundation.gestures.detectTransformGestures
import androidx.compose.ui.Modifier
import androidx.compose.ui.input.pointer.pointerInput
import com.cerve.development.ui.canvas.model.CerveCanvasInteractionType
import com.cerve.development.ui.canvas.model.CerveLine
import com.cerve.development.ui.canvas.model.CerveOffset
import com.cerve.development.ui.canvas.model.CerveOffset.Companion.offset

fun Modifier.canvasInput(
    interactionType: () -> CerveCanvasInteractionType?,
    gridSpacing: Int,
    eraserRadius: Float,
    eraserCenter: () -> CerveOffset?,
    currentLines: MutableList<CerveLine>,
    currentLineCandidates: MutableList<CerveLine>,
    translationAmount: CerveOffset,
    onChangeEraserCenter: (CerveOffset?) -> Unit
) = pointerInput(interactionType()) {
    when(interactionType()) {
        CerveCanvasInteractionType.Brush -> {
            brushGestures(
                translationAmount = translationAmount,
                currentLines = currentLines
            )
        }
        CerveCanvasInteractionType.AssistedBrush -> {
            assistedBrushGestures(
                gridSpacing = gridSpacing,
                translationAmount = translationAmount,
                currentLines = currentLines,
                currentLineCandidates = currentLineCandidates,
            )
        }
        CerveCanvasInteractionType.SegmentedBrush -> {
            segmentedBrushGestures(
                gridSpacing = gridSpacing,
                translationAmount = translationAmount,
                currentLines = currentLines,
                currentLineCandidates = currentLineCandidates
            )
        }
        CerveCanvasInteractionType.Eraser -> {
            eraserGestures(
                translationAmount = translationAmount,
                eraserCenter = eraserCenter,
                eraserRadius = eraserRadius,
                currentLines = currentLines,
                onChangeEraserCenter = onChangeEraserCenter
            )
        }
        else -> Unit
    }
}

fun Modifier.zoomInput(
    interactionType: () -> CerveCanvasInteractionType?,
    transformationScale: () -> Float,
    transformationRotation: () -> Float,
    translationAmount: () -> CerveOffset,
    onChangeTransformationScale: (Float?) -> Unit,
    onChangeTransformationRotation: (Float?) -> Unit,
    onChangeTransformationOffset: (CerveOffset?) -> Unit,
    onResetTransformations: () -> Unit
) = pointerInput(interactionType()) {
    if (interactionType() == CerveCanvasInteractionType.Zoom) {
        detectTransformGestures { centroid, pan, gestureZoom, rotation ->

            val zoom = transformationScale() * gestureZoom
            val compensatedPan = pan * zoom * 1f

            val position = translationAmount().offset + compensatedPan

            val scale = transformationScale() * gestureZoom
            val rotation = transformationRotation() + rotation

            onChangeTransformationScale(scale)
            onChangeTransformationRotation(rotation)
            onChangeTransformationOffset(position.offset)
        }
    }
}.pointerInput(interactionType()) {
    if (interactionType() == CerveCanvasInteractionType.Zoom) {
        detectTapGestures(
            onLongPress = { pressPosition ->

                val scale = transformationScale() * 1.5f
                val position = (pressPosition + translationAmount().offset) / scale

                onChangeTransformationScale(scale)
                onChangeTransformationOffset(position.offset)
            },
            onDoubleTap = { onResetTransformations() }
        )
    }
}

