package com.cerve.development.ui.canvas.model

import androidx.compose.runtime.mutableFloatStateOf
import androidx.compose.runtime.mutableStateListOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.ui.graphics.TransformOrigin
import com.cerve.development.ui.canvas.model.CerveTransformOrigin.Companion.toCerveTransformOrigin
import kotlinx.serialization.Serializable
import kotlinx.serialization.Transient

@Serializable
data class CerveDrawCanvasState(
    private val initialLines: List<CerveLine> = emptyList(),
    private val initialCandidateLines: List<CerveLine> = emptyList(),
    private val initialHistoryLines: List<CerveLine> = emptyList(),
    private val initialOffset: CerveOffset = CerveOffset.Zero,
    private val initialScale: Float = 1f,
    private val initialRotation: Float = 0f,
    private val initialEraserRadius: Float = 10f,
    private val initialEraserCenter: CerveOffset? = null,
    private val initialTransformOrigin: CerveTransformOrigin = CerveTransformOrigin.Zero
) {

    val eraserRadius: Float = initialEraserRadius

    @Transient
    var eraserCenter = mutableStateOf(initialEraserCenter)

    val historyLines: MutableList<CerveLine> = mutableStateListOf<CerveLine>()
    val currentLines: MutableList<CerveLine> = mutableStateListOf<CerveLine>()
    val currentLineCandidates: MutableList<CerveLine> = mutableStateListOf<CerveLine>()
    val selectedGridCells: MutableList<CerveCell> = mutableStateListOf<CerveCell>()

    @Transient
    var interactionType = mutableStateOf(CerveCanvasInteractionType.None)
        private set

    @Transient
    var transformationOffset = mutableStateOf(initialOffset)
        private set

    @Transient
    var transformationScale = mutableFloatStateOf(initialScale)
        private set

    @Transient
    var transformationRotation = mutableStateOf(initialRotation)
        private set

    @Transient
    var transformationOrigin = mutableStateOf<CerveTransformOrigin>(initialTransformOrigin)
        private set

    init {
        currentLines.addAll(initialLines)
        historyLines.addAll(initialLines.reversed())
    }

    fun onChangeInteractionType(
        value: CerveCanvasInteractionType
    ) { interactionType.value = value }

    fun onChangeEraserCenter(
        value: CerveOffset?
    ) { eraserCenter.value = value }

    fun onChangeTransformationScale(
        value: Float?
    ) { transformationScale.value = value ?: initialScale }

    fun onChangeTransformationRotation(
        value: Float?
    ) { transformationRotation.value = value ?: initialRotation }

    fun onChangeTransformationOffset(
        value: CerveOffset?
    ) { transformationOffset.value = value ?: initialOffset }

    fun onChangeTransformationOrigin(
        value: TransformOrigin?
    ) { transformationOrigin.value = value?.toCerveTransformOrigin ?: initialTransformOrigin }

    fun onUndo() {
        val removed = currentLines.removeLastOrNull()
        removed?.let { historyLines.add(removed) }
    }

    fun onRedo() {
        val history = historyLines.removeLastOrNull()
        history?.let { currentLines.add(history) }
    }

    fun onClear() {
        historyLines.clear()
        eraserCenter.value = null
        currentLines.clear()
        currentLineCandidates.clear()
    }

    fun onResetAllTransformation() {
        transformationRotation.value = initialRotation
        transformationScale.value = initialScale
        transformationOffset.value = initialOffset
    }

}
