package com.cerve.development.ui.canvas.operators

import androidx.compose.foundation.gestures.detectDragGestures
import androidx.compose.foundation.gestures.detectTapGestures
import androidx.compose.ui.input.pointer.PointerInputScope
import com.cerve.development.ui.canvas.model.CerveCanvasEraserProperties
import com.cerve.development.ui.canvas.model.CerveCell
import com.cerve.development.ui.canvas.model.CerveLine
import com.cerve.development.ui.canvas.model.CerveOffset
import com.cerve.development.ui.canvas.model.CerveOffset.Companion.offset
import com.cerve.development.ui.canvas.model.CerveSize

suspend fun PointerInputScope.eraserGestures(
    translationAmount: CerveOffset,
    eraserProperties: () -> CerveCanvasEraserProperties,
    currentLines: MutableList<CerveLine>,
    onUpdateEraserProperties: (CerveCanvasEraserProperties) -> Unit
) = detectDragGestures(
    onDragStart = { startPosition ->

        val position = startPosition - translationAmount.offset
        onUpdateEraserProperties(eraserProperties().copy(center = position.offset))

    },
    onDrag = { change, dragAmount ->

        eraserProperties().center?.offset?.let { center ->

            val position = (change.position - translationAmount.offset)
            onUpdateEraserProperties(eraserProperties().copy(center = position.offset))

            currentLines.removeAll { line ->
                checkCircleSegmentIntersection(
                    circleCenter = position,
                    circleRadius = eraserProperties().radius,
                    segmentStart = line.start.offset,
                    segmentEnd =  line.end.offset
                )
            }

        }

    },
    onDragEnd = { onUpdateEraserProperties(eraserProperties().copy(center = null)) }
)

suspend fun PointerInputScope.assistedBrushGestures(
    gridSpacing: Int,
    translationAmount: CerveOffset,
    currentLines: MutableList<CerveLine>,
    currentLineCandidates: MutableList<CerveLine>
) = detectDragGestures(
    onDragStart = { startPosition ->

        val position = startPosition - translationAmount.offset
        val line = CerveLine.Zero.plus(position)
            .snapToNear(spacing = gridSpacing)

        currentLineCandidates.add(line)
    },
    onDrag = { change, _ ->

        val lastIndex = currentLineCandidates.lastIndex
        val candidateLine = currentLineCandidates.last()

        val end = (change.position - translationAmount.offset)
            .snapToNear(spacing = gridSpacing)

        currentLineCandidates[lastIndex] = candidateLine.copy(end = end)
    },
    onDragEnd = {

        val redrawnLines = currentLineCandidates
            .snapDistinctCandidateLine(spacing = gridSpacing)

        currentLineCandidates.clear()
        currentLines.addAll(redrawnLines)

    }
)

suspend fun PointerInputScope.segmentedBrushGestures(
    gridSpacing: Int,
    translationAmount: CerveOffset,
    currentLines: MutableList<CerveLine>,
    currentLineCandidates: MutableList<CerveLine>
) = detectDragGestures(
    onDrag = { change, dragAmount ->
        currentLineCandidates.consumeLine(
            change = change,
            dragAmount = dragAmount,
            translationAmount = translationAmount.offset
        )
    },
    onDragEnd = {

        val filteredRoundedLine = currentLineCandidates
            .snapDistinctCandidateLine(spacing = gridSpacing)

        currentLines.addAll(filteredRoundedLine)
        currentLineCandidates.clear()

    }
)

suspend fun PointerInputScope.brushGestures(
    translationAmount: CerveOffset,
    currentLines: MutableList<CerveLine>
) = detectDragGestures(
    onDrag = { change, dragAmount ->
        currentLines.consumeLine(
            change = change,
            dragAmount = dragAmount,
            translationAmount = translationAmount.offset
        )
    }
)

suspend fun PointerInputScope.cellSelectorGestures(
    gridSpacing: Int,
    translationAmount: CerveOffset,
    selectedGridCells: MutableList<CerveCell>
) = detectTapGestures { tapPosition ->

    val position = (tapPosition - translationAmount.offset)
        .snapToFloor(gridSpacing)

    val exists = selectedGridCells.firstOrNull { cell ->
        cell.offset == position.offset
    }

    if (exists != null) {
        selectedGridCells.remove(exists)
    } else {
        val cell = CerveCell(
            topLeft = position,
            cellSize = CerveSize(
                width = gridSpacing.toFloat(),
                height = gridSpacing.toFloat()
            )
        )

        selectedGridCells.add(cell)
    }

}