package com.cerve.development.ui.canvas.model

import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.compose.runtime.toMutableStateList
import com.cerve.development.ui.canvas.operators.CerveCanvasDefaults
import kotlinx.serialization.Serializable
import kotlinx.serialization.Transient

@Serializable
data class CerveCanvasState(
    private val initialLineData: CerveCanvasLineData = CerveCanvasDefaults.canvasLines,
    private val initialEraserProperties: CerveCanvasEraserProperties = CerveCanvasDefaults.canvasEraserProperties,
    private val initialTransformations: CerveCanvasTransformations = CerveCanvasDefaults.canvasTransformations,
    private val initialInteractionType: CerveCanvasInteractionType = CerveCanvasInteractionType.None
) {

    constructor(
        gridLineCount: Int,
        initialInteractionType: CerveCanvasInteractionType
    ) : this(
        initialLineData = CerveCanvasDefaults.canvasLines.copy(gridLineCount = gridLineCount),
        initialInteractionType = initialInteractionType
    )

    var eraserProperties by mutableStateOf(initialEraserProperties)
        private set

    var gridLineCount by mutableStateOf(initialLineData.gridLineCount)
        private set

    var transformations by mutableStateOf(initialTransformations)
        private set

    var interactionType by mutableStateOf(initialInteractionType)
        private set

    @Transient
    val drawnLines = initialLineData.drawnLines.toMutableStateList()
    @Transient
    val drawnHistoryLines = initialLineData.drawnHistoryLines.toMutableStateList()
    @Transient
    val drawnLineCandidates = initialLineData.drawnLineCandidates.toMutableStateList()

    val selectedCellList: MutableList<CerveCell> = initialLineData.selectedCellList.toMutableStateList()
    var selectedCell by mutableStateOf<CerveCell?>(null)

    var selectedCellCandidate by mutableStateOf<CerveCell?>(null)

    fun onUpdateEraserProperties(
        value: CerveCanvasEraserProperties
    ) { eraserProperties = value }

    fun onUpdateGridLineCount(
        value: Int
    ) { gridLineCount = value }

    fun onUpdateTransformations(
        value: CerveCanvasTransformations?
    ) { transformations = value ?: initialTransformations }

    fun onChangeInteractionType(
        value: CerveCanvasInteractionType
    ) { interactionType = value }

    fun onUpdateSelectedCell(
        value: CerveCell?
    ) { selectedCell = value }

    fun onUpdateSelectedCellCandidate(
        value: CerveCell?
    ) { selectedCellCandidate = value }

    fun onUndo() {
        val removed = drawnLines.removeLastOrNull()
        removed?.let { drawnHistoryLines.add(removed) }
    }

    fun onRedo() {
        val history = drawnHistoryLines.removeLastOrNull()
        history?.let { drawnLines.add(history) }
    }

    fun onResetDrawnLines(
        value: List<CerveLine>
    ) {
        drawnLines.clear()
        drawnLines.addAll(value)
    }

    fun onResetDrawnLineCandidate(
        value: List<CerveLine>
    ) {
        drawnLineCandidates.clear()
        drawnLineCandidates.addAll(value)
    }

    fun onClear() {
        drawnHistoryLines.clear()
        drawnLines.clear()
        drawnLineCandidates.clear()
    }

}
