package io.pointberry.eventtracker;

import android.content.Context;
import android.os.AsyncTask;
import android.util.Log;

import com.google.android.gms.ads.identifier.AdvertisingIdClient;
import com.google.android.gms.common.GooglePlayServicesNotAvailableException;
import com.google.android.gms.common.GooglePlayServicesRepairableException;

import java.io.IOException;
import java.net.URL;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Random;

import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;

public class PointBerryImpressionTracker {

    private static final String TAG = PointBerryImpressionTracker.class.getSimpleName();
    private static final String BASE_URL = "https://impression.pointberry.io/beacon.png";
    private static Context appContext;


    public PointBerryImpressionTracker(Context context) {
        PointBerryImpressionTracker.appContext = context;
    }

    public void logImpression(String inventoryId) {
        logImpression(inventoryId, false);
    }

    public void logImpression(String inventoryId, boolean development) {
        ImpressionTask impTask = new ImpressionTask(inventoryId, development);
        impTask.execute();
    }

    private static class ImpressionTask extends AsyncTask<Void, Void, String> {

        private final String inventoryId;
        private final boolean development;

        ImpressionTask(String inventoryId, boolean development) {
            this.inventoryId = inventoryId;
            this.development = development;
        }

        @Override
        protected String doInBackground(Void... params) {
            AdvertisingIdClient.Info adIdInfo;
            String adId;
            try {
                adIdInfo = AdvertisingIdClient.getAdvertisingIdInfo(appContext);
                adId = adIdInfo.getId();
            } catch (IOException | GooglePlayServicesRepairableException | GooglePlayServicesNotAvailableException e) {
                Log.e(TAG, "Failed to get advertising ID");
                return null;
            }

            String url = BASE_URL
                    + "?ii=" + inventoryId
                    + "&adid=" + adId
                    + "&ti=" + System.currentTimeMillis() / 1000
                    + "&cb=" + new Random().nextInt();

            HttpsURLConnection conn = null;
            int responseCode;
            try {
                SSLContext sc = SSLContext.getInstance("TLS");
                sc.init(null, null, new SecureRandom());

                conn = (HttpsURLConnection) new URL(url).openConnection();
                conn.setSSLSocketFactory(sc.getSocketFactory());
                conn.setRequestMethod("GET");
                conn.connect();
                responseCode = conn.getResponseCode();
            } catch (IOException | NoSuchAlgorithmException | KeyManagementException e) {
                Log.e(TAG, "Failed to log ad impression with " + e.getClass().getName());
                return null;
            } finally {
                if (conn != null) conn.disconnect();
            }

            String msg;
            if (responseCode == 200) {
                msg = "Successfully logged ad impression for inventory ID " + inventoryId + " and advertising ID " + adId;
                if (development) {
                    Log.i(TAG, msg);
                } else {
                    Log.d(TAG, msg);
                }
            } else {
                msg = "Failed to log ad impression with HTTP status code " + responseCode;
                Log.e(TAG, msg);
            }

            return msg;
        }
    }
}