/*
 * This file is part of ELKI:
 * Environment for Developing KDD-Applications Supported by Index-Structures
 *
 * Copyright (C) 2022
 * ELKI Development Team
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package elki.clustering.trivial;

import elki.data.Cluster;
import elki.data.Clustering;
import elki.data.model.ClusterModel;
import elki.data.model.Model;
import elki.data.type.NoSupportedDataTypeException;
import elki.data.type.TypeUtil;
import elki.database.Database;
import elki.result.Metadata;
import elki.utilities.Priority;

/**
 * Trivial class that will try to cluster by label, and fall back to an
 * "all-in-one" clustering.
 * 
 * @author Erich Schubert
 * @since 0.5.0
 */
@Priority(Priority.SUPPLEMENTARY - 50)
public class ByLabelOrAllInOneClustering extends ByLabelClustering {
  /**
   * Constructor.
   */
  public ByLabelOrAllInOneClustering() {
    super();
  }

  @Override
  public Clustering<Model> autorun(Database database) {
    // Prefer a true class label
    try {
      return run(database.getRelation(TypeUtil.CLASSLABEL));
    }
    catch(NoSupportedDataTypeException e) {
      // Ignore.
    }
    try {
      return run(database.getRelation(TypeUtil.GUESSED_LABEL));
    }
    catch(NoSupportedDataTypeException e) {
      // Ignore.
    }
    Clustering<Model> result = new ReferenceClustering<>();
    Metadata.of(result).setLongName("All-in-one Trivial Clustering");
    result.addToplevelCluster(new Cluster<>(database.getRelation(TypeUtil.ANY).getDBIDs(), ClusterModel.CLUSTER));
    return result;
  }
}
