/*
 * Decompiled with CFR 0.152.
 */
package elki.clustering.correlation;

import elki.clustering.optics.ClusterOrder;
import elki.clustering.optics.CorrelationClusterOrder;
import elki.clustering.optics.GeneralizedOPTICS;
import elki.data.NumberVector;
import elki.data.type.TypeInformation;
import elki.data.type.TypeUtil;
import elki.database.datastore.DataStoreUtil;
import elki.database.datastore.WritableDataStore;
import elki.database.datastore.WritableDoubleDataStore;
import elki.database.datastore.WritableIntegerDataStore;
import elki.database.ids.ArrayModifiableDBIDs;
import elki.database.ids.DBIDArrayMIter;
import elki.database.ids.DBIDIter;
import elki.database.ids.DBIDRef;
import elki.database.ids.DBIDUtil;
import elki.database.ids.DBIDs;
import elki.database.query.QueryBuilder;
import elki.database.query.knn.KNNSearcher;
import elki.database.relation.Relation;
import elki.database.relation.RelationUtil;
import elki.distance.Distance;
import elki.distance.minkowski.EuclideanDistance;
import elki.logging.Logging;
import elki.logging.progress.AbstractProgress;
import elki.logging.progress.FiniteProgress;
import elki.logging.statistics.MillisTimeDuration;
import elki.logging.statistics.Statistic;
import elki.math.MathUtil;
import elki.math.linearalgebra.VMath;
import elki.math.linearalgebra.pca.PCAFilteredResult;
import elki.math.linearalgebra.pca.PCAResult;
import elki.math.linearalgebra.pca.PCARunner;
import elki.math.linearalgebra.pca.filter.EigenPairFilter;
import elki.math.linearalgebra.pca.filter.PercentageEigenPairFilter;
import elki.result.Metadata;
import elki.utilities.documentation.Description;
import elki.utilities.documentation.Reference;
import elki.utilities.documentation.Title;
import elki.utilities.exceptions.AbortException;
import elki.utilities.optionhandling.OptionID;
import elki.utilities.optionhandling.Parameterizer;
import elki.utilities.optionhandling.constraints.CommonConstraints;
import elki.utilities.optionhandling.constraints.ParameterConstraint;
import elki.utilities.optionhandling.parameterization.Parameterization;
import elki.utilities.optionhandling.parameters.DoubleParameter;
import elki.utilities.optionhandling.parameters.IntParameter;
import elki.utilities.optionhandling.parameters.ObjectParameter;
import java.util.Arrays;
import java.util.Comparator;

@Title(value="Mining Hierarchies of Correlation Clusters")
@Description(value="Algorithm for detecting hierarchies of correlation clusters.")
@Reference(authors="Elke Achtert, Christian B\u00f6hm, Peer Kr\u00f6ger, Arthur Zimek", title="Mining Hierarchies of Correlation Clusters", booktitle="Proc. Int. Conf. on Scientific and Statistical Database Management (SSDBM'06)", url="https://doi.org/10.1109/SSDBM.2006.35", bibkey="DBLP:conf/ssdbm/AchtertBKZ06")
public class HiCO
implements GeneralizedOPTICS {
    private static final Logging LOG = Logging.getLogger(HiCO.class);
    public static final double DEFAULT_DELTA = 0.25;
    public static final double DEFAULT_ALPHA = 0.85;
    private double deltasq;
    private int mu;
    private int k;
    private PCARunner pca;
    private EigenPairFilter filter;

    public HiCO(int k, PCARunner pca, double alpha, int mu, double delta) {
        this.k = k;
        this.pca = pca;
        this.filter = new PercentageEigenPairFilter(alpha);
        this.mu = mu;
        this.deltasq = delta * delta;
    }

    public TypeInformation[] getInputTypeRestriction() {
        return TypeUtil.array((TypeInformation[])new TypeInformation[]{TypeUtil.NUMBER_VECTOR_FIELD});
    }

    public ClusterOrder run(Relation<? extends NumberVector> relation) {
        if (this.mu >= relation.size()) {
            throw new AbortException("Parameter mu is chosen unreasonably large. This won't yield meaningful results.");
        }
        return new Instance(relation).run();
    }

    public int correlationDistance(PCAFilteredResult pca1, PCAFilteredResult pca2, int dimensionality) {
        double[][] v1t = VMath.copy((double[][])pca1.getEigenvectors());
        double[][] v1t_strong = pca1.getStrongEigenvectors();
        int lambda1 = pca1.getCorrelationDimension();
        double[][] v2t = VMath.copy((double[][])pca2.getEigenvectors());
        double[][] v2t_strong = pca2.getStrongEigenvectors();
        int lambda2 = pca2.getCorrelationDimension();
        double[][] m1_czech = pca1.dissimilarityMatrix();
        for (int i = 0; i < v2t_strong.length; ++i) {
            double[] v2_i = v2t_strong[i];
            double distsq = VMath.squareSum((double[])v2_i) - VMath.transposeTimesTimes((double[])v2_i, (double[][])m1_czech, (double[])v2_i);
            if (lambda1 >= dimensionality || !(distsq > this.deltasq)) continue;
            this.adjust(v1t, v2_i, lambda1++);
            double[] e1_czech_d = new double[v1t.length];
            Arrays.fill(e1_czech_d, 0, lambda1, 1.0);
            m1_czech = VMath.transposeDiagonalTimes((double[][])v1t, (double[])e1_czech_d, (double[][])v1t);
        }
        double[][] m2_czech = pca2.dissimilarityMatrix();
        for (int i = 0; i < v1t_strong.length; ++i) {
            double[] v1_i = v1t_strong[i];
            double distsq = VMath.squareSum((double[])v1_i) - VMath.transposeTimesTimes((double[])v1_i, (double[][])m2_czech, (double[])v1_i);
            if (lambda2 >= dimensionality || !(distsq > this.deltasq)) continue;
            this.adjust(v2t, v1_i, lambda2++);
            double[] e2_czech_d = new double[v1t.length];
            Arrays.fill(e2_czech_d, 0, lambda2, 1.0);
            m2_czech = VMath.transposeDiagonalTimes((double[][])v2t, (double[])e2_czech_d, (double[][])v2t);
        }
        return Math.max(lambda1, lambda2);
    }

    private void adjust(double[][] v, double[] vector, int corrDim) {
        double[] sum = new double[v.length];
        for (int i = 0; i < corrDim; ++i) {
            VMath.plusTimesEquals((double[])sum, (double[])v[i], (double)VMath.transposeTimes((double[])vector, (double[])v[i]));
        }
        v[corrDim] = VMath.normalizeEquals((double[])VMath.minus((double[])vector, (double[])sum));
    }

    @Override
    public int getMinPts() {
        return this.mu;
    }

    public static class Par
    implements Parameterizer {
        public static final OptionID MU_ID = new OptionID("hico.mu", "Specifies the smoothing factor. The mu-nearest neighbor is used to compute the correlation reachability of an object.");
        public static final OptionID K_ID = new OptionID("hico.k", "Optional parameter to specify the number of nearest neighbors considered in the PCA. If this parameter is not set, k is set to the value of parameter mu.");
        public static final OptionID DELTA_ID = new OptionID("hico.delta", "Threshold of a distance between a vector q and a given space that indicates that q adds a new dimension to the space.");
        public static final OptionID ALPHA_ID = new OptionID("hico.alpha", "The threshold for 'strong' eigenvectors: the 'strong' eigenvectors explain a portion of at least alpha of the total variance.");
        protected int k = 0;
        protected PCARunner pca;
        int mu = -1;
        double alpha = 0.85;
        double delta = 0.25;

        public void configure(Parameterization config) {
            ((IntParameter)new IntParameter(MU_ID).addConstraint((ParameterConstraint)CommonConstraints.GREATER_EQUAL_ONE_INT)).grab(config, x -> {
                this.mu = x;
            });
            ((IntParameter)((IntParameter)new IntParameter(K_ID).addConstraint((ParameterConstraint)CommonConstraints.GREATER_EQUAL_ONE_INT)).setOptional(true)).grab(config, x -> {
                this.k = x;
            });
            ((DoubleParameter)((DoubleParameter)new DoubleParameter(DELTA_ID).setDefaultValue((Object)0.25)).addConstraint((ParameterConstraint)CommonConstraints.GREATER_EQUAL_ZERO_DOUBLE)).grab(config, x -> {
                this.delta = x;
            });
            ((DoubleParameter)((DoubleParameter)((DoubleParameter)new DoubleParameter(ALPHA_ID).setDefaultValue((Object)0.85)).addConstraint((ParameterConstraint)CommonConstraints.GREATER_THAN_ZERO_DOUBLE)).addConstraint((ParameterConstraint)CommonConstraints.LESS_THAN_ONE_DOUBLE)).grab(config, x -> {
                this.alpha = x;
            });
            new ObjectParameter(PCARunner.Par.PCARUNNER_ID, PCARunner.class, PCARunner.class).grab(config, x -> {
                this.pca = x;
            });
        }

        public HiCO make() {
            return new HiCO(this.k > 0 ? this.k : this.mu, this.pca, this.alpha, this.mu, this.delta);
        }
    }

    private class Instance
    extends GeneralizedOPTICS.Instance<CorrelationClusterOrder> {
        private Relation<? extends NumberVector> relation;
        protected WritableDataStore<PCAFilteredResult> localPCAs;
        private ArrayModifiableDBIDs clusterOrder;
        private WritableIntegerDataStore correlationValue;
        private WritableIntegerDataStore tmpCorrelation;
        private WritableDoubleDataStore tmpDistance;
        private ArrayModifiableDBIDs tmpIds;
        Comparator<DBIDRef> tmpcomp;

        public Instance(Relation<? extends NumberVector> relation) {
            super(relation.getDBIDs());
            this.tmpcomp = (o1, o2) -> {
                int c = Integer.compare(this.tmpCorrelation.intValue(o2), this.tmpCorrelation.intValue(o1));
                return c != 0 ? c : Double.compare(this.tmpDistance.doubleValue(o1), this.tmpDistance.doubleValue(o2));
            };
            DBIDs ids = relation.getDBIDs();
            this.clusterOrder = DBIDUtil.newArray((int)ids.size());
            this.relation = relation;
            this.correlationValue = DataStoreUtil.makeIntegerStorage((DBIDs)ids, (int)30, (int)Integer.MAX_VALUE);
            this.tmpIds = DBIDUtil.newArray((DBIDs)ids);
            this.tmpCorrelation = DataStoreUtil.makeIntegerStorage((DBIDs)ids, (int)3);
            this.tmpDistance = DataStoreUtil.makeDoubleStorage((DBIDs)ids, (int)3);
        }

        @Override
        public CorrelationClusterOrder run() {
            int dim = RelationUtil.dimensionality(this.relation);
            if (dim > 0 && HiCO.this.k <= dim) {
                LOG.warning((CharSequence)"PCA results with k < dim are meaningless. Choose k much larger than the dimensionality.");
            }
            this.localPCAs = DataStoreUtil.makeStorage((DBIDs)this.relation.getDBIDs(), (int)3, PCAFilteredResult.class);
            KNNSearcher knnQuery = new QueryBuilder(this.relation, (Distance)EuclideanDistance.STATIC).kNNByDBID(HiCO.this.k);
            MillisTimeDuration dur = new MillisTimeDuration(this.getClass() + ".preprocessing-time").begin();
            FiniteProgress progress = LOG.isVerbose() ? new FiniteProgress("Performing local PCA", this.relation.size(), LOG) : null;
            DBIDIter iditer = this.relation.iterDBIDs();
            while (iditer.valid()) {
                PCAResult epairs = HiCO.this.pca.processIds((DBIDs)knnQuery.getKNN((Object)iditer, HiCO.this.k), this.relation);
                int numstrong = HiCO.this.filter.filter(epairs.getEigenvalues());
                this.localPCAs.put((DBIDRef)iditer, (Object)new PCAFilteredResult(epairs.getEigenPairs(), numstrong, 1.0, 0.0));
                LOG.incrementProcessed((AbstractProgress)progress);
                iditer.advance();
            }
            LOG.ensureCompleted(progress);
            LOG.statistics((Statistic)dur.end());
            return (CorrelationClusterOrder)super.run();
        }

        @Override
        protected CorrelationClusterOrder buildResult() {
            CorrelationClusterOrder result = new CorrelationClusterOrder(this.clusterOrder, this.reachability, this.predecessor, this.correlationValue);
            Metadata.of((Object)result).setLongName("HiCO Cluster Order");
            return result;
        }

        @Override
        protected void initialDBID(DBIDRef id) {
            this.correlationValue.put(id, Integer.MAX_VALUE);
        }

        @Override
        protected void expandDBID(DBIDRef id) {
            this.clusterOrder.add(id);
            PCAFilteredResult pca1 = (PCAFilteredResult)this.localPCAs.get(id);
            NumberVector dv1 = (NumberVector)this.relation.get(id);
            int dim = dv1.getDimensionality();
            DBIDArrayMIter iter = this.tmpIds.iter();
            while (iter.valid()) {
                if (this.processedIDs.contains((DBIDRef)iter)) {
                    this.tmpCorrelation.putInt((DBIDRef)iter, 0);
                } else {
                    PCAFilteredResult pca2 = (PCAFilteredResult)this.localPCAs.get((DBIDRef)iter);
                    NumberVector dv2 = (NumberVector)this.relation.get((DBIDRef)iter);
                    this.tmpCorrelation.putInt((DBIDRef)iter, HiCO.this.correlationDistance(pca1, pca2, dim));
                    this.tmpDistance.putDouble((DBIDRef)iter, EuclideanDistance.STATIC.distance(dv1, dv2));
                }
                iter.advance();
            }
            this.tmpIds.sort(this.tmpcomp);
            double coredist = this.tmpDistance.doubleValue((DBIDRef)iter.seek(HiCO.this.mu - 1));
            iter.seek(0);
            while (iter.valid()) {
                int curcorr;
                int prevcorr;
                if (!this.processedIDs.contains((DBIDRef)iter) && (prevcorr = this.correlationValue.intValue((DBIDRef)iter)) >= (curcorr = this.tmpCorrelation.intValue((DBIDRef)iter))) {
                    double prevdist;
                    double currdist = MathUtil.max((double)this.tmpDistance.doubleValue((DBIDRef)iter), (double)coredist);
                    if (prevcorr != curcorr || !((prevdist = this.reachability.doubleValue((DBIDRef)iter)) <= currdist)) {
                        this.correlationValue.putInt((DBIDRef)iter, curcorr);
                        this.reachability.putDouble((DBIDRef)iter, currdist);
                        this.predecessor.putDBID((DBIDRef)iter, id);
                        if (prevcorr == Integer.MAX_VALUE) {
                            this.candidates.add((DBIDRef)iter);
                        }
                    }
                }
                iter.advance();
            }
        }

        @Override
        public int compare(DBIDRef o1, DBIDRef o2) {
            int c = Integer.compare(this.correlationValue.intValue(o2), this.correlationValue.intValue(o1));
            return c != 0 ? c : super.compare(o1, o2);
        }

        @Override
        protected Logging getLogger() {
            return LOG;
        }
    }
}

