/*
 * Decompiled with CFR 0.152.
 */
package elki.clustering.dbscan;

import elki.Algorithm;
import elki.clustering.ClusteringAlgorithm;
import elki.data.Cluster;
import elki.data.Clustering;
import elki.data.model.ClusterModel;
import elki.data.model.Model;
import elki.data.type.TypeInformation;
import elki.data.type.TypeUtil;
import elki.database.ids.ArrayModifiableDBIDs;
import elki.database.ids.DBIDIter;
import elki.database.ids.DBIDRef;
import elki.database.ids.DBIDUtil;
import elki.database.ids.DBIDVar;
import elki.database.ids.DBIDs;
import elki.database.ids.DoubleDBIDList;
import elki.database.ids.DoubleDBIDListIter;
import elki.database.ids.ModifiableDBIDs;
import elki.database.ids.ModifiableDoubleDBIDList;
import elki.database.query.QueryBuilder;
import elki.database.query.range.RangeSearcher;
import elki.database.relation.Relation;
import elki.distance.Distance;
import elki.distance.minkowski.EuclideanDistance;
import elki.logging.Logging;
import elki.logging.progress.AbstractProgress;
import elki.logging.progress.FiniteProgress;
import elki.logging.progress.IndefiniteProgress;
import elki.logging.statistics.DoubleStatistic;
import elki.logging.statistics.Statistic;
import elki.result.Metadata;
import elki.utilities.Priority;
import elki.utilities.documentation.Description;
import elki.utilities.documentation.Reference;
import elki.utilities.documentation.References;
import elki.utilities.documentation.Title;
import elki.utilities.optionhandling.OptionID;
import elki.utilities.optionhandling.Parameterizer;
import elki.utilities.optionhandling.constraints.CommonConstraints;
import elki.utilities.optionhandling.constraints.ParameterConstraint;
import elki.utilities.optionhandling.parameterization.Parameterization;
import elki.utilities.optionhandling.parameters.DoubleParameter;
import elki.utilities.optionhandling.parameters.IntParameter;
import elki.utilities.optionhandling.parameters.ObjectParameter;
import java.util.ArrayList;
import java.util.List;

@Title(value="DBSCAN: Density-Based Clustering of Applications with Noise")
@Description(value="Algorithm to find density-connected sets in a database based on the parameters 'minpts' and 'epsilon' (specifying a volume). These two parameters determine a density threshold for clustering.")
@References(value={@Reference(authors="Martin Ester, Hans-Peter Kriegel, J\u00f6rg Sander, Xiaowei Xu", title="A Density-Based Algorithm for Discovering Clusters in Large Spatial Databases with Noise", booktitle="Proc. 2nd Int. Conf. on Knowledge Discovery and Data Mining (KDD '96)", url="http://www.aaai.org/Library/KDD/1996/kdd96-037.php", bibkey="DBLP:conf/kdd/EsterKSX96"), @Reference(authors="Erich Schubert, J\u00f6rg Sander, Martin Ester, Hans-Peter Kriegel, Xiaowei Xu", title="DBSCAN Revisited, Revisited: Why and How You Should (Still) Use DBSCAN", booktitle="ACM Trans. Database Systems (TODS)", url="https://doi.org/10.1145/3068335", bibkey="DBLP:journals/tods/SchubertSEKX17")})
@Priority(value=200)
public class DBSCAN<O>
implements ClusteringAlgorithm<Clustering<Model>> {
    private static final Logging LOG = Logging.getLogger(DBSCAN.class);
    protected Distance<? super O> distance;
    protected double epsilon;
    protected int minpts;

    public DBSCAN(Distance<? super O> distance, double epsilon, int minpts) {
        this.distance = distance;
        this.epsilon = epsilon;
        this.minpts = minpts;
    }

    public TypeInformation[] getInputTypeRestriction() {
        return TypeUtil.array((TypeInformation[])new TypeInformation[]{this.distance.getInputTypeRestriction()});
    }

    public Clustering<Model> run(Relation<O> relation) {
        int size = relation.size();
        if (size < this.minpts) {
            Clustering<Model> result = new Clustering<Model>();
            Metadata.of(result).setLongName("DBSCAN Clustering");
            result.addToplevelCluster(new Cluster<ClusterModel>(relation.getDBIDs(), true, ClusterModel.CLUSTER));
            return result;
        }
        Instance dbscan = new Instance();
        dbscan.run(relation, (RangeSearcher<DBIDRef>)new QueryBuilder(relation, this.distance).rangeByDBID(this.epsilon));
        double averagen = (double)dbscan.ncounter / (double)relation.size();
        LOG.statistics((Statistic)new DoubleStatistic(DBSCAN.class.getName() + ".average-neighbors", averagen));
        if (averagen < 1.0 + 0.1 * (double)(this.minpts - 1)) {
            LOG.warning((CharSequence)"There are very few neighbors found. Epsilon may be too small.");
        }
        if (averagen > (double)(100 * this.minpts)) {
            LOG.warning((CharSequence)"There are very many neighbors found. Epsilon may be too large.");
        }
        Clustering<Model> result = new Clustering<Model>();
        Metadata.of(result).setLongName("DBSCAN Clustering");
        for (ModifiableDBIDs res : dbscan.resultList) {
            result.addToplevelCluster(new Cluster<ClusterModel>((DBIDs)res, ClusterModel.CLUSTER));
        }
        result.addToplevelCluster(new Cluster<ClusterModel>((DBIDs)dbscan.noise, true, ClusterModel.CLUSTER));
        return result;
    }

    public static class Par<O>
    implements Parameterizer {
        public static final OptionID EPSILON_ID = new OptionID("dbscan.epsilon", "The maximum radius of the neighborhood to be considered.");
        public static final OptionID MINPTS_ID = new OptionID("dbscan.minpts", "Threshold for minimum number of points in the epsilon-neighborhood of a point. The suggested value is '2 * dim - 1'.");
        protected double epsilon;
        protected int minpts;
        protected Distance<? super O> distance;

        public void configure(Parameterization config) {
            new ObjectParameter(Algorithm.Utils.DISTANCE_FUNCTION_ID, Distance.class, EuclideanDistance.class).grab(config, x -> {
                this.distance = x;
            });
            ((DoubleParameter)new DoubleParameter(EPSILON_ID).addConstraint((ParameterConstraint)CommonConstraints.GREATER_THAN_ZERO_DOUBLE)).grab(config, x -> {
                this.epsilon = x;
            });
            if (((IntParameter)new IntParameter(MINPTS_ID).addConstraint((ParameterConstraint)CommonConstraints.GREATER_EQUAL_ONE_INT)).grab(config, x -> {
                this.minpts = x;
            }) && this.minpts <= 2) {
                LOG.warning((CharSequence)"DBSCAN with minPts <= 2 is equivalent to single-link clustering at a single height. Consider using larger values of minPts.");
            }
        }

        public DBSCAN<O> make() {
            return new DBSCAN<O>(this.distance, this.epsilon, this.minpts);
        }
    }

    private class Instance {
        protected List<ModifiableDBIDs> resultList;
        protected ModifiableDBIDs noise;
        protected ModifiableDBIDs processedIDs;
        protected long ncounter;
        protected FiniteProgress objprog;
        protected IndefiniteProgress clusprog;
        protected RangeSearcher<DBIDRef> rangeQuery;
        protected final ModifiableDoubleDBIDList neighbors = DBIDUtil.newDistanceDBIDList();

        private Instance() {
        }

        protected void run(Relation<O> relation, RangeSearcher<DBIDRef> rangeSearcher) {
            int size = relation.size();
            this.objprog = LOG.isVerbose() ? new FiniteProgress("Processing objects", size, LOG) : null;
            this.clusprog = LOG.isVerbose() ? new IndefiniteProgress("Number of clusters", LOG) : null;
            this.rangeQuery = rangeSearcher;
            this.resultList = new ArrayList<ModifiableDBIDs>();
            this.noise = DBIDUtil.newHashSet();
            this.processedIDs = DBIDUtil.newHashSet((int)size);
            ArrayModifiableDBIDs seeds = DBIDUtil.newArray();
            DBIDIter iditer = relation.iterDBIDs();
            while (iditer.valid()) {
                if (!this.processedIDs.contains((DBIDRef)iditer)) {
                    this.expandCluster((DBIDRef)iditer, seeds);
                }
                if (this.objprog != null && this.clusprog != null) {
                    this.objprog.setProcessed(this.processedIDs.size(), LOG);
                    this.clusprog.setProcessed(this.resultList.size(), LOG);
                }
                if (this.processedIDs.size() == size) break;
                iditer.advance();
            }
            LOG.ensureCompleted(this.objprog);
            LOG.setCompleted(this.clusprog);
        }

        protected void expandCluster(DBIDRef startObjectID, ArrayModifiableDBIDs seeds) {
            this.rangeQuery.getRange((Object)startObjectID, DBSCAN.this.epsilon, this.neighbors.clear());
            this.processedIDs.add(startObjectID);
            LOG.incrementProcessed((AbstractProgress)this.objprog);
            this.ncounter += (long)this.neighbors.size();
            if (this.neighbors.size() < DBSCAN.this.minpts) {
                this.noise.add(startObjectID);
                return;
            }
            ArrayModifiableDBIDs currentCluster = DBIDUtil.newArray((int)this.neighbors.size());
            currentCluster.add(startObjectID);
            this.processNeighbors((DoubleDBIDList)this.neighbors, (ModifiableDBIDs)currentCluster, seeds);
            DBIDVar o = DBIDUtil.newVar();
            while (!seeds.isEmpty()) {
                this.rangeQuery.getRange((Object)seeds.pop(o), DBSCAN.this.epsilon, this.neighbors.clear());
                this.ncounter += (long)this.neighbors.size();
                if (this.neighbors.size() >= DBSCAN.this.minpts) {
                    this.processNeighbors((DoubleDBIDList)this.neighbors, (ModifiableDBIDs)currentCluster, seeds);
                }
                LOG.incrementProcessed((AbstractProgress)this.objprog);
            }
            this.resultList.add((ModifiableDBIDs)currentCluster);
            LOG.incrementProcessed((AbstractProgress)this.clusprog);
        }

        private void processNeighbors(DoubleDBIDList neighbors, ModifiableDBIDs currentCluster, ArrayModifiableDBIDs seeds) {
            boolean ismetric = DBSCAN.this.distance.isMetric();
            DoubleDBIDListIter neighbor = neighbors.iter();
            while (neighbor.valid()) {
                block8: {
                    block7: {
                        block6: {
                            if (!this.processedIDs.add((DBIDRef)neighbor)) break block6;
                            if (!ismetric || neighbor.doubleValue() > 0.0) {
                                seeds.add((DBIDRef)neighbor);
                            }
                            break block7;
                        }
                        if (!this.noise.remove((DBIDRef)neighbor)) break block8;
                    }
                    currentCluster.add((DBIDRef)neighbor);
                }
                neighbor.advance();
            }
        }
    }
}

