/*
 * Decompiled with CFR 0.152.
 */
package elki.clustering.dbscan;

import elki.Algorithm;
import elki.clustering.ClusteringAlgorithm;
import elki.clustering.dbscan.DBSCAN;
import elki.clustering.dbscan.util.Assignment;
import elki.clustering.dbscan.util.Border;
import elki.clustering.dbscan.util.Core;
import elki.clustering.dbscan.util.MultiBorder;
import elki.data.Cluster;
import elki.data.Clustering;
import elki.data.NumberVector;
import elki.data.model.ClusterModel;
import elki.data.model.Model;
import elki.data.type.CombinedTypeInformation;
import elki.data.type.TypeInformation;
import elki.data.type.TypeUtil;
import elki.database.datastore.DataStoreUtil;
import elki.database.datastore.WritableDataStore;
import elki.database.datastore.WritableIntegerDataStore;
import elki.database.ids.ArrayModifiableDBIDs;
import elki.database.ids.DBIDIter;
import elki.database.ids.DBIDMIter;
import elki.database.ids.DBIDRef;
import elki.database.ids.DBIDUtil;
import elki.database.ids.DBIDVar;
import elki.database.ids.DBIDs;
import elki.database.ids.DoubleDBIDList;
import elki.database.ids.DoubleDBIDListIter;
import elki.database.ids.ModifiableDBIDs;
import elki.database.ids.ModifiableDoubleDBIDList;
import elki.database.query.QueryBuilder;
import elki.database.query.range.RangeSearcher;
import elki.database.relation.ProxyView;
import elki.database.relation.Relation;
import elki.database.relation.RelationUtil;
import elki.distance.Distance;
import elki.distance.minkowski.EuclideanDistance;
import elki.distance.minkowski.LPNormDistance;
import elki.logging.Logging;
import elki.logging.progress.AbstractProgress;
import elki.logging.progress.FiniteProgress;
import elki.logging.statistics.DoubleStatistic;
import elki.logging.statistics.LongStatistic;
import elki.logging.statistics.Statistic;
import elki.logging.statistics.StringStatistic;
import elki.result.Metadata;
import elki.utilities.documentation.Reference;
import elki.utilities.documentation.Title;
import elki.utilities.exceptions.IncompatibleDataException;
import elki.utilities.optionhandling.OptionID;
import elki.utilities.optionhandling.Parameterizer;
import elki.utilities.optionhandling.constraints.CommonConstraints;
import elki.utilities.optionhandling.constraints.GreaterEqualConstraint;
import elki.utilities.optionhandling.constraints.ParameterConstraint;
import elki.utilities.optionhandling.parameterization.Parameterization;
import elki.utilities.optionhandling.parameters.DoubleParameter;
import elki.utilities.optionhandling.parameters.IntParameter;
import elki.utilities.optionhandling.parameters.ObjectParameter;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import java.util.Arrays;
import net.jafama.FastMath;

@Title(value="GriDBSCAN: Using Grid for Accelerating Density-Based Clustering")
@Reference(authors="S. Mahran, K. Mahar", title="Using grid for accelerating density-based clustering", booktitle="8th IEEE Int. Conf. on Computer and Information Technology", url="https://doi.org/10.1109/CIT.2008.4594646", bibkey="DBLP:conf/IEEEcit/MahranM08")
public class GriDBSCAN<V extends NumberVector>
implements ClusteringAlgorithm<Clustering<Model>> {
    private static final Logging LOG = Logging.getLogger(GriDBSCAN.class);
    protected Distance<? super V> distance;
    protected double epsilon;
    protected int minpts;
    protected double gridwidth;

    public GriDBSCAN(Distance<? super V> distance, double epsilon, int minpts, double gridwidth) {
        this.distance = distance;
        this.epsilon = epsilon;
        this.minpts = minpts;
        this.gridwidth = gridwidth;
    }

    public TypeInformation[] getInputTypeRestriction() {
        return TypeUtil.array((TypeInformation[])new TypeInformation[]{new CombinedTypeInformation(new TypeInformation[]{TypeUtil.NUMBER_VECTOR_FIELD, this.distance.getInputTypeRestriction()})});
    }

    public Clustering<Model> run(Relation<V> relation) {
        DBIDs ids = relation.getDBIDs();
        if (ids.size() < this.minpts) {
            Clustering<Model> result = new Clustering<Model>();
            Metadata.of(result).setLongName("DBSCAN Clustering");
            result.addToplevelCluster(new Cluster<ClusterModel>(ids, true, ClusterModel.CLUSTER));
            return result;
        }
        double adjgridwidth = this.gridwidth;
        if (adjgridwidth < 2.0 * this.epsilon) {
            LOG.warning((CharSequence)"Invalid grid width (less than 2*epsilon, recommended 10*epsilon). Increasing grid width automatically.");
            adjgridwidth = 2.0 * this.epsilon;
        }
        return new Instance<V>(this.distance, this.epsilon, this.minpts, adjgridwidth).run(relation);
    }

    public static class Par<O extends NumberVector>
    implements Parameterizer {
        public static final OptionID GRID_ID = new OptionID("gridbscan.gridwidth", "Width of the grid used, must be at least two times epsilon.");
        protected double epsilon;
        protected int minpts;
        protected double gridwidth;
        protected LPNormDistance distance;

        public void configure(Parameterization config) {
            new ObjectParameter(Algorithm.Utils.DISTANCE_FUNCTION_ID, LPNormDistance.class, EuclideanDistance.class).grab(config, x -> {
                this.distance = x;
            });
            ((DoubleParameter)new DoubleParameter(DBSCAN.Par.EPSILON_ID).addConstraint((ParameterConstraint)CommonConstraints.GREATER_THAN_ZERO_DOUBLE)).grab(config, x -> {
                this.epsilon = x;
            });
            ((IntParameter)new IntParameter(DBSCAN.Par.MINPTS_ID).addConstraint((ParameterConstraint)CommonConstraints.GREATER_EQUAL_ONE_INT)).grab(config, x -> {
                this.minpts = x;
                if (this.minpts <= 2) {
                    LOG.warning((CharSequence)"DBSCAN with minPts <= 2 is equivalent to single-link clustering at a single height. Consider using larger values of minPts.");
                }
            });
            ((DoubleParameter)((DoubleParameter)((DoubleParameter)new DoubleParameter(GRID_ID).addConstraint((ParameterConstraint)CommonConstraints.GREATER_THAN_ZERO_DOUBLE)).setDefaultValue((Object)(this.epsilon > 0.0 ? 10.0 * this.epsilon : 1.0))).addConstraint((ParameterConstraint)new GreaterEqualConstraint(2.0 * this.epsilon))).grab(config, x -> {
                this.gridwidth = x;
            });
        }

        public GriDBSCAN<O> make() {
            return new GriDBSCAN(this.distance, this.epsilon, this.minpts, this.gridwidth);
        }
    }

    protected static class Instance<V extends NumberVector> {
        protected static final int UNPROCESSED = 0;
        protected static final int NOISE = 1;
        protected Distance<? super V> distance;
        protected double epsilon;
        protected int minpts;
        protected double gridwidth;
        protected double[][] domain;
        protected int dim;
        protected double[] offset;
        protected int[] cells;
        Long2ObjectOpenHashMap<ModifiableDBIDs> grid;
        private Core[] cores;
        private Border[] borders;
        private WritableDataStore<Assignment> clusterids;
        private WritableIntegerDataStore temporary;
        private boolean overflown;

        public Instance(Distance<? super V> distance, double epsilon, int minpts, double gridwidth) {
            this.distance = distance;
            this.epsilon = epsilon;
            this.minpts = minpts;
            this.gridwidth = gridwidth;
        }

        public Clustering<Model> run(Relation<V> relation) {
            DBIDs ids = relation.getDBIDs();
            int size = ids.size();
            this.domain = RelationUtil.computeMinMax(relation);
            this.dim = this.domain[0].length;
            this.offset = new double[this.dim];
            this.cells = new int[this.dim];
            long numcells = this.computeGridBaseOffsets(size);
            this.buildGrid(relation, (int)numcells, this.offset);
            if (this.grid.size() <= this.dim) {
                LOG.warning((CharSequence)("There are only " + this.grid.size() + " occupied cells. This will likely be slower than regular DBSCAN!"));
            }
            int mincells = this.checkGridCellSizes(size, numcells);
            this.clusterids = DataStoreUtil.makeStorage((DBIDs)ids, (int)1, Assignment.class);
            this.temporary = DataStoreUtil.makeIntegerStorage((DBIDs)ids, (int)1, (int)0);
            ArrayModifiableDBIDs activeSet = DBIDUtil.newArray();
            int clusterid = 2;
            this.cores = new Core[2];
            this.borders = new Border[2];
            ModifiableDoubleDBIDList neighbors = DBIDUtil.newDistanceDBIDList((int)(this.minpts << 1));
            FiniteProgress cprog = LOG.isVerbose() ? new FiniteProgress("Processing grid cells", mincells, LOG) : null;
            for (ModifiableDBIDs cellids : this.grid.values()) {
                if (cellids.size() < this.minpts) continue;
                clusterid += this.runDBSCANOnCell((DBIDs)cellids, relation, neighbors, activeSet, clusterid);
                this.updateCoreBorderObjects(clusterid);
                this.mergeClusterInformation(cellids, this.temporary, this.clusterids);
                LOG.incrementProcessed((AbstractProgress)cprog);
            }
            LOG.ensureCompleted(cprog);
            this.temporary.destroy();
            return this.buildResult(ids, clusterid);
        }

        private int runDBSCANOnCell(DBIDs cellids, Relation<V> relation, ModifiableDoubleDBIDList neighbors, ArrayModifiableDBIDs activeSet, int clusterid) {
            this.temporary.clear();
            ProxyView rel = new ProxyView(cellids, relation);
            RangeSearcher rq = new QueryBuilder((Relation)rel, this.distance).rangeByDBID(this.epsilon);
            FiniteProgress pprog = LOG.isVerbose() ? new FiniteProgress("Running DBSCAN", cellids.size(), LOG) : null;
            DBIDIter id = cellids.iter();
            while (id.valid()) {
                if (this.temporary.intValue((DBIDRef)id) == 0) {
                    rq.getRange((Object)id, this.epsilon, neighbors.clear());
                    if (neighbors.size() >= this.minpts) {
                        this.expandCluster((DBIDRef)id, clusterid, this.temporary, neighbors, activeSet, (RangeSearcher<DBIDRef>)rq, pprog);
                        ++clusterid;
                    } else {
                        this.temporary.putInt((DBIDRef)id, 1);
                        LOG.incrementProcessed((AbstractProgress)pprog);
                    }
                }
                id.advance();
            }
            LOG.ensureCompleted(pprog);
            return clusterid;
        }

        private void updateCoreBorderObjects(int clusterid) {
            this.cores = Arrays.copyOf(this.cores, clusterid);
            this.borders = Arrays.copyOf(this.borders, clusterid);
            for (int i = this.cores.length; i < clusterid; ++i) {
                this.cores[i] = new Core(i);
                this.borders[i] = new Border(this.cores[i]);
            }
        }

        private long computeGridBaseOffsets(int size) {
            StringBuilder buf = LOG.isDebuggingFinest() ? new StringBuilder() : null;
            double[] min = this.domain[0];
            double[] max = this.domain[1];
            long total = 1L;
            for (int d = 0; d < this.dim; ++d) {
                double mi = min[d];
                double ma = max[d];
                double wi = ma - mi;
                if (mi == Double.NEGATIVE_INFINITY || ma == Double.POSITIVE_INFINITY || mi != mi || ma != ma) {
                    throw new IncompatibleDataException("Dimension " + d + " contains non-finite values.");
                }
                int c = this.cells[d] = Math.max(1, (int)FastMath.ceil((double)(wi / this.gridwidth)));
                this.offset[d] = mi - ((double)c * this.gridwidth - wi) * 0.5;
                assert (this.offset[d] <= mi || this.offset[d] + (double)c * this.gridwidth >= ma) : "Grid inconsistent.";
                if ((total *= (long)c) < 0L) {
                    LOG.warning((CharSequence)"Excessive amount of grid cells (long overflow)! Use larger grid cells.");
                    this.overflown = true;
                    total &= Long.MAX_VALUE;
                }
                if (buf == null) continue;
                buf.append(d).append(": min=").append(mi).append(" max=").append(ma);
                double s = this.offset[d];
                for (int i = 0; i <= c; ++i) {
                    buf.append(' ').append(s);
                    s += this.gridwidth;
                }
                buf.append('\n');
            }
            if (buf != null) {
                LOG.debugFinest((CharSequence)buf);
            }
            if (total > (long)size) {
                LOG.warning((CharSequence)"The generated grid has more cells than data points. This may need excessive amounts of memory.");
            } else if (total == 1L) {
                LOG.warning((CharSequence)"All data is in a single cell. This has degenerated to a non-indexed DBSCAN!");
            } else if (total <= (long)(this.dim * this.dim)) {
                LOG.warning((CharSequence)("There are only " + total + " cells. This will likely be slower than regular DBSCAN!"));
            }
            return total;
        }

        protected void buildGrid(Relation<V> relation, int numcells, double[] offset) {
            this.grid = new Long2ObjectOpenHashMap(numcells >>> 2);
            DBIDIter it = relation.iterDBIDs();
            while (it.valid()) {
                NumberVector obj = (NumberVector)relation.get((DBIDRef)it);
                this.insertIntoGrid((DBIDRef)it, obj, 0, 0);
                it.advance();
            }
        }

        private void insertIntoGrid(DBIDRef id, V obj, int d, int v) {
            int cn = this.cells[d];
            int nd = d + 1;
            int mi = Math.max(0, (int)FastMath.floor((double)((obj.doubleValue(d) - this.offset[d] - this.epsilon) / this.gridwidth)));
            int ma = Math.min(cn - 1, (int)FastMath.floor((double)((obj.doubleValue(d) - this.offset[d] + this.epsilon) / this.gridwidth)));
            assert (mi <= ma) : "Grid inconsistent.";
            for (int i = mi; i <= ma; ++i) {
                int c = v * cn + i;
                if (nd == this.cells.length) {
                    ((ModifiableDBIDs)this.grid.computeIfAbsent((long)c, x -> DBIDUtil.newArray())).add(id);
                    continue;
                }
                this.insertIntoGrid(id, obj, nd, c);
            }
        }

        protected int checkGridCellSizes(int size, long numcell) {
            int tcount = 0;
            int hasmin = 0;
            double sqcount = 0.0;
            for (ModifiableDBIDs cell : this.grid.values()) {
                int s = cell.size();
                if (s >= size >> 1) {
                    LOG.warning((CharSequence)("A single cell contains half of the database (" + s + " objects). This will not scale very well."));
                }
                tcount += s;
                sqcount += (double)((long)s * (long)s);
                if (s < this.minpts) continue;
                ++hasmin;
            }
            double savings = sqcount / (double)size / (double)size;
            if (savings >= 1.0) {
                LOG.warning((CharSequence)"Pairwise distances within each cells are more expensive than a full DBSCAN run due to overlap!");
            }
            if (this.overflown) {
                LOG.statistics((Statistic)new StringStatistic(GriDBSCAN.class.getName() + ".all-cells", "overflow"));
            } else {
                LOG.statistics((Statistic)new LongStatistic(GriDBSCAN.class.getName() + ".all-cells", numcell));
            }
            LOG.statistics((Statistic)new LongStatistic(GriDBSCAN.class.getName() + ".used-cells", (long)this.grid.size()));
            LOG.statistics((Statistic)new LongStatistic(GriDBSCAN.class.getName() + ".minpts-cells", (long)hasmin));
            LOG.statistics((Statistic)new DoubleStatistic(GriDBSCAN.class.getName() + ".redundancy", (double)tcount / (double)size));
            LOG.statistics((Statistic)new DoubleStatistic(GriDBSCAN.class.getName() + ".relative-cost", savings));
            return hasmin;
        }

        protected int expandCluster(DBIDRef seed, int clusterid, WritableIntegerDataStore clusterids, ModifiableDoubleDBIDList neighbors, ArrayModifiableDBIDs activeSet, RangeSearcher<DBIDRef> rq, FiniteProgress pprog) {
            assert (activeSet.size() == 0);
            int clustersize = 1 + this.processCorePoint(seed, (DoubleDBIDList)neighbors, clusterid, clusterids, activeSet);
            LOG.incrementProcessed((AbstractProgress)pprog);
            DBIDVar id = DBIDUtil.newVar();
            while (!activeSet.isEmpty()) {
                activeSet.pop(id);
                rq.getRange((Object)id, this.epsilon, neighbors.clear());
                if (neighbors.size() >= this.minpts) {
                    clustersize += this.processCorePoint((DBIDRef)id, (DoubleDBIDList)neighbors, clusterid, clusterids, activeSet);
                }
                LOG.incrementProcessed((AbstractProgress)pprog);
            }
            return clustersize;
        }

        protected int processCorePoint(DBIDRef seed, DoubleDBIDList newneighbors, int clusterid, WritableIntegerDataStore clusterids, ArrayModifiableDBIDs activeSet) {
            clusterids.putInt(seed, clusterid);
            int clustersize = 0;
            DoubleDBIDListIter it = newneighbors.iter();
            while (it.valid()) {
                block8: {
                    block7: {
                        int oldassign;
                        block6: {
                            oldassign = clusterids.intValue((DBIDRef)it);
                            if (oldassign != 0) break block6;
                            if (it.doubleValue() > 0.0) {
                                activeSet.add((DBIDRef)it);
                            }
                            break block7;
                        }
                        if (oldassign != 1) break block8;
                    }
                    ++clustersize;
                    clusterids.putInt((DBIDRef)it, -clusterid);
                }
                it.advance();
            }
            return clustersize;
        }

        protected void mergeClusterInformation(ModifiableDBIDs cellids, WritableIntegerDataStore temporary, WritableDataStore<Assignment> clusterids) {
            FiniteProgress mprog = LOG.isVerbose() ? new FiniteProgress("Collecting result", cellids.size(), LOG) : null;
            DBIDMIter id = cellids.iter();
            while (id.valid()) {
                Assignment oclus;
                int nclus = temporary.intValue((DBIDRef)id);
                if (nclus > 1) {
                    Core core = this.cores[nclus];
                    assert (core.num > 1);
                    oclus = (Assignment)clusterids.get((DBIDRef)id);
                    if (oclus == null) {
                        clusterids.put((DBIDRef)id, (Object)core);
                    } else if (oclus instanceof Core) {
                        core.mergeWith((Core)oclus);
                    } else if (oclus instanceof Border) {
                        core.mergeWith(((Border)oclus).core);
                        clusterids.put((DBIDRef)id, (Object)core);
                    } else {
                        int m2;
                        int m;
                        assert (oclus instanceof MultiBorder);
                        if (LOG.isDebuggingFinest()) {
                            LOG.debugFinest((CharSequence)("Multi-Merge: " + nclus + " - " + oclus + " -> " + core));
                        }
                        int n = m = (m = core.num) < (m2 = ((MultiBorder)oclus).getCore().num) ? m : m2;
                        assert (m > 1);
                        for (Border b : ((MultiBorder)oclus).cs) {
                            this.cores[b.core.num].num = m;
                        }
                        core.num = m;
                        clusterids.put((DBIDRef)id, (Object)core);
                    }
                } else if (nclus < 0) {
                    Border border = this.borders[-nclus];
                    oclus = (Assignment)clusterids.get((DBIDRef)id);
                    if (oclus == null) {
                        clusterids.put((DBIDRef)id, (Object)border);
                    } else if (oclus instanceof Core) {
                        ((Core)oclus).mergeWith(border.core);
                    } else if (oclus instanceof Border) {
                        if (((Border)oclus).core.num != border.core.num) {
                            clusterids.put((DBIDRef)id, (Object)new MultiBorder((Border)oclus, border));
                        }
                    } else {
                        assert (oclus instanceof MultiBorder);
                        clusterids.put((DBIDRef)id, (Object)((MultiBorder)oclus).update(border));
                    }
                } else assert (nclus == 1);
                LOG.incrementProcessed((AbstractProgress)mprog);
                id.advance();
            }
            LOG.ensureCompleted(mprog);
        }

        protected Clustering<Model> buildResult(DBIDs ids, int clusterid) {
            FiniteProgress pprog = LOG.isVerbose() ? new FiniteProgress("Building final result", ids.size(), LOG) : null;
            ModifiableDBIDs[] clusters = new ModifiableDBIDs[clusterid];
            ArrayModifiableDBIDs noise = DBIDUtil.newArray();
            DBIDIter it = ids.iter();
            while (it.valid()) {
                Assignment cids = (Assignment)this.clusterids.get((DBIDRef)it);
                if (cids == null) {
                    noise.add((DBIDRef)it);
                } else {
                    if (cids instanceof MultiBorder) {
                        cids = ((MultiBorder)cids).getCore();
                    } else if (cids instanceof Border) {
                        cids = ((Border)cids).core;
                    }
                    assert (cids instanceof Core);
                    Core co = (Core)cids;
                    while (this.cores[co.num].num != co.num) {
                        co.num = this.cores[co.num].num;
                        co = this.cores[co.num];
                    }
                    ModifiableDBIDs clu = clusters[co.num];
                    if (clu == null) {
                        clu = clusters[co.num] = DBIDUtil.newArray();
                    }
                    clu.add((DBIDRef)it);
                }
                LOG.incrementProcessed((AbstractProgress)pprog);
                it.advance();
            }
            LOG.ensureCompleted(pprog);
            this.clusterids.destroy();
            Clustering<Model> result = new Clustering<Model>();
            Metadata.of(result).setLongName("DBSCAN Clustering");
            for (int i = 2; i < clusters.length; ++i) {
                if (clusters[i] == null) continue;
                result.addToplevelCluster(new Cluster<ClusterModel>((DBIDs)clusters[i], ClusterModel.CLUSTER));
            }
            if (noise.size() > 0) {
                result.addToplevelCluster(new Cluster<ClusterModel>((DBIDs)noise, true, ClusterModel.CLUSTER));
            }
            return result;
        }
    }
}

