/*
 * Decompiled with CFR 0.152.
 */
package elki.clustering.hierarchical.birch;

import elki.clustering.hierarchical.birch.BIRCHAbsorptionCriterion;
import elki.clustering.hierarchical.birch.BIRCHDistance;
import elki.clustering.hierarchical.birch.ClusteringFeature;
import elki.clustering.hierarchical.birch.DiameterCriterion;
import elki.clustering.hierarchical.birch.VarianceIncreaseDistance;
import elki.data.NumberVector;
import elki.database.ids.DBIDIter;
import elki.database.ids.DBIDRef;
import elki.database.ids.DBIDs;
import elki.database.relation.Relation;
import elki.logging.Logging;
import elki.logging.progress.AbstractProgress;
import elki.logging.progress.FiniteProgress;
import elki.utilities.datastructures.iterator.Iter;
import elki.utilities.documentation.Reference;
import elki.utilities.documentation.References;
import elki.utilities.io.FormatUtil;
import elki.utilities.optionhandling.OptionID;
import elki.utilities.optionhandling.Parameterizer;
import elki.utilities.optionhandling.constraints.CommonConstraints;
import elki.utilities.optionhandling.constraints.GreaterEqualConstraint;
import elki.utilities.optionhandling.constraints.ParameterConstraint;
import elki.utilities.optionhandling.parameterization.Parameterization;
import elki.utilities.optionhandling.parameters.DoubleParameter;
import elki.utilities.optionhandling.parameters.IntParameter;
import elki.utilities.optionhandling.parameters.ObjectParameter;
import java.util.ArrayList;
import java.util.Arrays;

@References(value={@Reference(authors="T. Zhang, R. Ramakrishnan, M. Livny", title="BIRCH: An Efficient Data Clustering Method for Very Large Databases", booktitle="Proc. 1996 ACM SIGMOD International Conference on Management of Data", url="https://doi.org/10.1145/233269.233324", bibkey="DBLP:conf/sigmod/ZhangRL96"), @Reference(authors="T. Zhang, R. Ramakrishnan, M. Livny", title="BIRCH: A New Data Clustering Algorithm and Its Applications", booktitle="Data Min. Knowl. Discovery", url="https://doi.org/10.1023/A:1009783824328", bibkey="DBLP:journals/datamine/ZhangRL97")})
public class CFTree {
    public static final Logging LOG = Logging.getLogger(CFTree.class);
    BIRCHDistance distance;
    BIRCHAbsorptionCriterion absorption;
    double thresholdsq;
    int capacity;
    TreeNode root = null;
    int leaves;

    public CFTree(BIRCHDistance distance, BIRCHAbsorptionCriterion absorption, double threshold, int capacity) {
        this.distance = distance;
        this.absorption = absorption;
        this.thresholdsq = threshold * threshold;
        this.capacity = capacity;
    }

    public void insert(NumberVector nv) {
        int dim = nv.getDimensionality();
        if (this.root == null) {
            ClusteringFeature leaf = new ClusteringFeature(dim);
            leaf.addToStatistics(nv);
            this.root = new TreeNode(dim, this.capacity);
            this.root.children[0] = leaf;
            this.root.addToStatistics(nv);
            ++this.leaves;
            return;
        }
        TreeNode other = this.insert(this.root, nv);
        if (other != null) {
            TreeNode newnode = new TreeNode(dim, this.capacity);
            newnode.children[0] = this.root;
            newnode.addToStatistics(newnode.children[0]);
            newnode.children[1] = other;
            newnode.addToStatistics(newnode.children[1]);
            this.root = newnode;
        }
    }

    protected void rebuildTree() {
        int dim = this.root.getDimensionality();
        double t = this.estimateThreshold(this.root) / (double)this.leaves;
        this.thresholdsq = (t *= t) > this.thresholdsq ? t : this.thresholdsq;
        LOG.debug((CharSequence)("New squared threshold: " + this.thresholdsq));
        LeafIterator iter = new LeafIterator(this.root);
        assert (iter.valid());
        ClusteringFeature first = iter.get();
        this.leaves = 0;
        this.root = new TreeNode(dim, this.capacity);
        this.root.children[0] = first;
        this.root.addToStatistics(first);
        ++this.leaves;
        iter.advance();
        while (iter.valid()) {
            TreeNode other = this.insert(this.root, iter.get());
            if (other != null) {
                TreeNode newnode = new TreeNode(dim, this.capacity);
                newnode.children[0] = this.root;
                newnode.addToStatistics(newnode.children[0]);
                newnode.children[1] = other;
                newnode.addToStatistics(newnode.children[1]);
                this.root = newnode;
            }
            iter.advance();
        }
    }

    private double estimateThreshold(TreeNode current) {
        ClusteringFeature[] children = current.children;
        double total = 0.0;
        if (!(children[0] instanceof TreeNode)) {
            ClusteringFeature ci;
            if (children[1] == null) {
                return 0.0;
            }
            double[] best = new double[children.length];
            Arrays.fill(best, Double.POSITIVE_INFINITY);
            int[] besti = new int[children.length];
            for (int i = 0; i < children.length && (ci = children[i]) != null; ++i) {
                double bi = best[i];
                int bestii = besti[i];
                for (int j = i + 1; j < children.length && children[j] != null; ++j) {
                    double dist = this.distance.squaredDistance(ci, children[j]);
                    if (dist < bi) {
                        bi = dist;
                        bestii = j;
                    }
                    if (!(dist < best[j])) continue;
                    best[j] = dist;
                    besti[j] = i;
                }
                double t = this.absorption.squaredCriterion(ci, children[bestii]);
                total += t > 0.0 ? Math.sqrt(t) : 0.0;
            }
        } else {
            assert (children[0] instanceof TreeNode) : "Node is neither child nor inner?";
            for (int i = 0; i < children.length && children[i] != null; ++i) {
                total += this.estimateThreshold((TreeNode)children[i]);
            }
        }
        return total;
    }

    private TreeNode insert(TreeNode node, NumberVector nv) {
        ClusteringFeature cf;
        ClusteringFeature[] cfs = node.children;
        assert (cfs[0] != null) : "Unexpected empty node!";
        ClusteringFeature best = cfs[0];
        double bestd = this.distance.squaredDistance(nv, best);
        for (int i = 1; i < cfs.length && (cf = cfs[i]) != null; ++i) {
            double d2 = this.distance.squaredDistance(nv, cf);
            if (!(d2 < bestd)) continue;
            best = cf;
            bestd = d2;
        }
        if (!(best instanceof TreeNode)) {
            if (this.absorption.squaredCriterion(best, nv) <= this.thresholdsq) {
                best.addToStatistics(nv);
                node.addToStatistics(nv);
                return null;
            }
            best = new ClusteringFeature(nv.getDimensionality());
            best.addToStatistics(nv);
            ++this.leaves;
            if (this.add(node.children, best)) {
                node.addToStatistics(nv);
                return null;
            }
            return this.split(node, best);
        }
        assert (best instanceof TreeNode) : "Node is neither child nor inner?";
        TreeNode newchild = this.insert((TreeNode)best, nv);
        if (newchild == null || this.add(node.children, newchild)) {
            node.addToStatistics(nv);
            return null;
        }
        return this.split(node, newchild);
    }

    public ClusteringFeature findLeaf(NumberVector nv) {
        if (this.root == null) {
            throw new IllegalStateException("CFTree not yet built.");
        }
        return this.findLeaf(this.root, nv);
    }

    private ClusteringFeature findLeaf(TreeNode node, NumberVector nv) {
        ClusteringFeature cf;
        ClusteringFeature[] cfs = node.children;
        assert (cfs[0] != null) : "Unexpected empty node!";
        ClusteringFeature best = cfs[0];
        double bestd = this.distance.squaredDistance(nv, best);
        for (int i = 1; i < cfs.length && (cf = cfs[i]) != null; ++i) {
            double d2 = this.distance.squaredDistance(nv, cf);
            if (!(d2 < bestd)) continue;
            best = cf;
            bestd = d2;
        }
        return best instanceof TreeNode ? this.findLeaf((TreeNode)best, nv) : best;
    }

    private TreeNode split(TreeNode node, ClusteringFeature newchild) {
        int j;
        int capacity = node.children.length;
        assert (node.children[capacity - 1] != null) : "Node to split is not empty!";
        TreeNode newn = new TreeNode(node.getDimensionality(), capacity);
        int size = capacity + 1;
        int m1 = -1;
        int m2 = -1;
        double maxd = Double.NEGATIVE_INFINITY;
        double[][] dists = new double[size][size];
        for (int i = 0; i < capacity; ++i) {
            ClusteringFeature ci = node.children[i];
            for (int j2 = i + 1; j2 < capacity; ++j2) {
                double d = this.distance.squaredDistance(ci, node.children[j2]);
                dists[j2][i] = d;
                dists[i][j2] = d;
                double d2 = d;
                if (!(d2 > maxd)) continue;
                maxd = d2;
                m1 = i;
                m2 = j2;
            }
            double d = this.distance.squaredDistance(ci, newchild);
            dists[capacity][i] = d;
            dists[i][capacity] = d;
            double d3 = d;
            if (!(d3 > maxd)) continue;
            maxd = d3;
            m1 = i;
            m2 = capacity;
        }
        node.resetStatistics();
        newn.resetStatistics();
        int si = 0;
        int sj = 0;
        double[] d1s = dists[m1];
        double[] d2s = dists[m2];
        for (int i = 0; i < capacity; ++i) {
            double d1 = d1s[i];
            double d2 = d2s[i];
            if (i == m1 || i != m2 && (d1 < d2 || d1 == d2 && si <= sj)) {
                int n = si++;
                ClusteringFeature clusteringFeature = node.children[i];
                node.children[n] = clusteringFeature;
                node.addToStatistics(clusteringFeature);
                continue;
            }
            int n = sj++;
            ClusteringFeature clusteringFeature = node.children[i];
            newn.children[n] = clusteringFeature;
            newn.addToStatistics(clusteringFeature);
        }
        double d1 = d1s[capacity];
        double d2 = d2s[capacity];
        if (capacity != m2 && (d1 < d2 || d1 == d2 && si <= sj)) {
            int n = si++;
            ClusteringFeature clusteringFeature = newchild;
            node.children[n] = clusteringFeature;
            node.addToStatistics(clusteringFeature);
        } else {
            int n = sj++;
            ClusteringFeature clusteringFeature = newchild;
            newn.children[n] = clusteringFeature;
            newn.addToStatistics(clusteringFeature);
        }
        for (j = si; j < capacity; ++j) {
            node.children[j] = null;
        }
        for (j = sj; j < capacity; ++j) {
            assert (newn.children[j] == null);
        }
        return newn;
    }

    private TreeNode insert(TreeNode node, ClusteringFeature nleaf) {
        ClusteringFeature cf;
        ClusteringFeature[] cfs = node.children;
        assert (cfs[0] != null) : "Unexpected empty node!";
        ClusteringFeature best = cfs[0];
        double bestd = this.distance.squaredDistance(nleaf, best);
        for (int i = 1; i < cfs.length && (cf = cfs[i]) != null; ++i) {
            double d2 = this.distance.squaredDistance(nleaf, cf);
            if (!(d2 < bestd)) continue;
            best = cf;
            bestd = d2;
        }
        assert (best != nleaf);
        if (!(best instanceof TreeNode)) {
            if (this.absorption.squaredCriterion(best, nleaf) <= this.thresholdsq) {
                best.addToStatistics(nleaf);
                node.addToStatistics(nleaf);
                return null;
            }
            ++this.leaves;
            if (this.add(node.children, nleaf)) {
                node.addToStatistics(nleaf);
                return null;
            }
            return this.split(node, nleaf);
        }
        assert (best instanceof TreeNode) : "Node is neither child nor inner?";
        TreeNode newchild = this.insert((TreeNode)best, nleaf);
        if (newchild == null || this.add(node.children, newchild)) {
            node.addToStatistics(nleaf);
            return null;
        }
        return this.split(node, newchild);
    }

    private boolean add(ClusteringFeature[] children, ClusteringFeature child) {
        for (int i = 0; i < children.length; ++i) {
            if (children[i] != null) continue;
            children[i] = child;
            return true;
        }
        return false;
    }

    public LeafIterator leafIterator() {
        return new LeafIterator(this.root);
    }

    protected StringBuilder printDebug(StringBuilder buf, ClusteringFeature n, int d) {
        FormatUtil.appendSpace((StringBuilder)buf, (int)d).append(n.n);
        for (int i = 0; i < n.getDimensionality(); ++i) {
            buf.append(' ').append(n.centroid(i));
        }
        buf.append(" - ").append(n.n).append('\n');
        if (n instanceof TreeNode) {
            ClusteringFeature[] children = ((TreeNode)n).children;
            for (int i = 0; i < children.length; ++i) {
                ClusteringFeature c = children[i];
                if (c == null) continue;
                this.printDebug(buf, c, d + 1);
            }
        }
        return buf;
    }

    public static class Factory {
        BIRCHDistance distance;
        BIRCHAbsorptionCriterion absorption;
        double threshold;
        int branchingFactor;
        double maxleaves;

        public Factory(BIRCHDistance distance, BIRCHAbsorptionCriterion absorption, double threshold, int branchingFactor, double maxleaves) {
            this.distance = distance;
            this.absorption = absorption;
            this.threshold = threshold;
            this.branchingFactor = branchingFactor;
            this.maxleaves = maxleaves;
        }

        public CFTree newTree(DBIDs ids, Relation<? extends NumberVector> relation) {
            CFTree tree = new CFTree(this.distance, this.absorption, this.threshold, this.branchingFactor);
            double max = this.maxleaves <= 1.0 ? this.maxleaves * (double)ids.size() : this.maxleaves;
            FiniteProgress prog = LOG.isVerbose() ? new FiniteProgress("Building tree", relation.size(), LOG) : null;
            DBIDIter it = relation.iterDBIDs();
            while (it.valid()) {
                tree.insert((NumberVector)relation.get((DBIDRef)it));
                if ((double)tree.leaves > max) {
                    if (LOG.isVerbose()) {
                        LOG.verbose((CharSequence)"Compacting CF-tree.");
                    }
                    tree.rebuildTree();
                }
                LOG.incrementProcessed((AbstractProgress)prog);
                it.advance();
            }
            LOG.ensureCompleted(prog);
            return tree;
        }

        public static class Par
        implements Parameterizer {
            public static final OptionID DISTANCE_ID = new OptionID("cftree.distance", "Distance function to use for node assignment.");
            public static final OptionID ABSORPTION_ID = new OptionID("cftree.absorption", "Absorption criterion to use.");
            public static final OptionID THRESHOLD_ID = new OptionID("cftree.threshold", "Threshold for adding points to existing nodes in the CF-Tree.");
            public static final OptionID BRANCHING_ID = new OptionID("cftree.branching", "Maximum branching factor of the CF-Tree");
            public static final OptionID MAXLEAVES_ID = new OptionID("cftree.maxleaves", "Maximum number of leaves (if less than 1, the values is assumed to be relative)");
            BIRCHDistance distance;
            BIRCHAbsorptionCriterion absorption;
            double threshold = 0.0;
            int branchingFactor;
            double maxleaves;

            public void configure(Parameterization config) {
                new ObjectParameter(DISTANCE_ID, BIRCHDistance.class, VarianceIncreaseDistance.class).grab(config, x -> {
                    this.distance = x;
                });
                new ObjectParameter(ABSORPTION_ID, BIRCHAbsorptionCriterion.class, DiameterCriterion.class).grab(config, x -> {
                    this.absorption = x;
                });
                ((DoubleParameter)((DoubleParameter)new DoubleParameter(THRESHOLD_ID).addConstraint((ParameterConstraint)CommonConstraints.GREATER_EQUAL_ZERO_DOUBLE)).setOptional(true)).grab(config, x -> {
                    this.threshold = x;
                });
                ((IntParameter)((IntParameter)new IntParameter(BRANCHING_ID).addConstraint((ParameterConstraint)new GreaterEqualConstraint(2))).setDefaultValue((Object)64)).grab(config, x -> {
                    this.branchingFactor = x;
                });
                ((DoubleParameter)((DoubleParameter)new DoubleParameter(MAXLEAVES_ID).addConstraint((ParameterConstraint)CommonConstraints.GREATER_THAN_ZERO_DOUBLE)).setDefaultValue((Object)0.05)).grab(config, x -> {
                    this.maxleaves = x;
                });
            }

            public Factory make() {
                return new Factory(this.distance, this.absorption, this.threshold, this.branchingFactor, this.maxleaves);
            }
        }
    }

    public static class TreeNode
    extends ClusteringFeature {
        ClusteringFeature[] children;

        public TreeNode(int dim, int capacity) {
            super(dim);
            this.children = new ClusteringFeature[capacity];
        }
    }

    public static class LeafIterator
    implements Iter {
        private ArrayList<ClusteringFeature> queue = new ArrayList();
        private ClusteringFeature current;

        private LeafIterator(TreeNode root) {
            this.queue.add(root);
            this.advance();
        }

        public boolean valid() {
            return this.current != null;
        }

        public ClusteringFeature get() {
            return this.current;
        }

        public Iter advance() {
            this.current = null;
            while (!this.queue.isEmpty()) {
                ClusteringFeature f = this.queue.remove(this.queue.size() - 1);
                if (!(f instanceof TreeNode)) {
                    this.current = f;
                    break;
                }
                for (ClusteringFeature c : ((TreeNode)f).children) {
                    if (c == null) break;
                    this.queue.add(c);
                }
            }
            return this;
        }
    }
}

