/*
 * Decompiled with CFR 0.152.
 */
package elki.clustering.subspace;

import elki.clustering.em.EM;
import elki.clustering.em.models.EMClusterModel;
import elki.clustering.em.models.MultivariateGaussianModel;
import elki.clustering.subspace.SubspaceClusteringAlgorithm;
import elki.data.Cluster;
import elki.data.Clustering;
import elki.data.NumberVector;
import elki.data.Subspace;
import elki.data.VectorUtil;
import elki.data.model.SubspaceModel;
import elki.data.type.TypeInformation;
import elki.data.type.TypeUtil;
import elki.database.datastore.DataStoreUtil;
import elki.database.datastore.WritableDataStore;
import elki.database.ids.ArrayModifiableDBIDs;
import elki.database.ids.DBIDArrayMIter;
import elki.database.ids.DBIDIter;
import elki.database.ids.DBIDMIter;
import elki.database.ids.DBIDRef;
import elki.database.ids.DBIDUtil;
import elki.database.ids.DBIDs;
import elki.database.ids.HashSetModifiableDBIDs;
import elki.database.ids.ModifiableDBIDs;
import elki.database.ids.SetDBIDs;
import elki.database.relation.Relation;
import elki.database.relation.RelationUtil;
import elki.logging.Logging;
import elki.logging.progress.FiniteProgress;
import elki.logging.progress.MutableProgress;
import elki.logging.progress.StepProgress;
import elki.logging.statistics.DoubleStatistic;
import elki.logging.statistics.Statistic;
import elki.math.MathUtil;
import elki.math.MeanVariance;
import elki.math.linearalgebra.CovarianceMatrix;
import elki.math.linearalgebra.VMath;
import elki.math.statistics.distribution.ChiSquaredDistribution;
import elki.math.statistics.distribution.PoissonDistribution;
import elki.result.Metadata;
import elki.utilities.Priority;
import elki.utilities.datastructures.BitsUtil;
import elki.utilities.documentation.Reference;
import elki.utilities.documentation.Title;
import elki.utilities.io.FormatUtil;
import elki.utilities.optionhandling.OptionID;
import elki.utilities.optionhandling.Parameterizer;
import elki.utilities.optionhandling.constraints.CommonConstraints;
import elki.utilities.optionhandling.constraints.ParameterConstraint;
import elki.utilities.optionhandling.parameterization.Parameterization;
import elki.utilities.optionhandling.parameters.DoubleParameter;
import elki.utilities.optionhandling.parameters.IntParameter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;

@Title(value="P3C: A Robust Projected Clustering Algorithm.")
@Reference(authors="Gabriela Moise, J\u00f6rg Sander, Martin Ester", title="P3C: A Robust Projected Clustering Algorithm", booktitle="Proc. Sixth International Conference on Data Mining (ICDM '06)", url="https://doi.org/10.1109/ICDM.2006.123", bibkey="DBLP:conf/icdm/MoiseSE06")
@Priority(value=190)
public class P3C
implements SubspaceClusteringAlgorithm<SubspaceModel> {
    private static final Logging LOG = Logging.getLogger(P3C.class);
    protected double poissonThreshold;
    protected int maxEmIterations;
    protected double emDelta;
    protected int minClusterSize;
    protected double alpha = 0.001;

    public P3C(double alpha, double poissonThreshold, int maxEmIterations, double emDelta, int minClusterSize) {
        this.alpha = alpha;
        this.poissonThreshold = poissonThreshold;
        this.maxEmIterations = maxEmIterations;
        this.emDelta = emDelta;
        this.minClusterSize = minClusterSize;
    }

    public TypeInformation[] getInputTypeRestriction() {
        return TypeUtil.array((TypeInformation[])new TypeInformation[]{TypeUtil.NUMBER_VECTOR_FIELD});
    }

    public Clustering<SubspaceModel> run(Relation<? extends NumberVector> relation) {
        StepProgress stepProgress;
        int dim = RelationUtil.dimensionality(relation);
        StepProgress stepProgress2 = stepProgress = LOG.isVerbose() ? new StepProgress(8) : null;
        if (stepProgress != null) {
            stepProgress.beginStep(1, "Grid-partitioning data.", LOG);
        }
        int binCount = (int)Math.ceil(1.0 + MathUtil.log2((double)relation.size()));
        SetDBIDs[][] partitions = this.partitionData(relation, binCount);
        if (stepProgress != null) {
            stepProgress.beginStep(2, "Searching for non-uniform bins in support histograms.", LOG);
        }
        long[][] markers = new long[dim][];
        for (int d = 0; d < dim; ++d) {
            int bestBin;
            SetDBIDs[] parts = partitions[d];
            if (parts == null) continue;
            markers[d] = BitsUtil.zero((int)binCount);
            long[] marked = markers[d];
            for (int card = 0; card < dim - 1 && (bestBin = this.chiSquaredUniformTest(parts, marked, card)) >= 0; ++card) {
                BitsUtil.setI((long[])marked, (int)bestBin);
            }
            if (!LOG.isDebugging()) continue;
            LOG.debug((CharSequence)("Marked bins in dim " + d + ": " + BitsUtil.toString((long[])marked, (int)binCount)));
        }
        if (stepProgress != null) {
            stepProgress.beginStep(3, "Merging marked bins to 1-signatures.", LOG);
        }
        ArrayList<Signature> signatures = this.constructOneSignatures(partitions, markers);
        if (stepProgress != null) {
            stepProgress.beginStep(4, "Computing cluster cores from merged p-signatures.", LOG);
        }
        ArrayList<Signature> clusterCores = this.mergeClusterCores(binCount, signatures);
        if (stepProgress != null) {
            stepProgress.beginStep(5, "Pruning redundant cluster cores.", LOG);
        }
        clusterCores = this.pruneRedundantClusterCores(clusterCores);
        if (LOG.isVerbose()) {
            LOG.verbose((CharSequence)("Number of cluster cores found: " + clusterCores.size()));
        }
        if (clusterCores.isEmpty()) {
            LOG.setCompleted(stepProgress);
            Clustering<SubspaceModel> c = new Clustering<SubspaceModel>();
            Metadata.of(c).setLongName("P3C Clustering");
            c.addToplevelCluster(new Cluster(relation.getDBIDs(), true));
            return c;
        }
        if (stepProgress != null) {
            stepProgress.beginStep(5, "Refining cluster cores to clusters via EM.", LOG);
        }
        HashSetModifiableDBIDs noise = DBIDUtil.newHashSet();
        WritableDataStore probClusterIGivenX = DataStoreUtil.makeStorage((DBIDs)relation.getDBIDs(), (int)10, double[].class);
        int k = clusterCores.size();
        ArrayList<MultivariateGaussianModel> models = new ArrayList<MultivariateGaussianModel>(k);
        this.computeFuzzyMembership(relation, clusterCores, (ModifiableDBIDs)noise, (WritableDataStore<double[]>)probClusterIGivenX, models, dim);
        EM.recomputeCovarianceMatrices(relation, (WritableDataStore<double[]>)probClusterIGivenX, models, 0.0);
        this.assignUnassigned(relation, (WritableDataStore<double[]>)probClusterIGivenX, models, (ModifiableDBIDs)noise);
        double emNew = EM.assignProbabilitiesToInstances(relation, models, (WritableDataStore<double[]>)probClusterIGivenX, null);
        for (int it = 1; it <= this.maxEmIterations || this.maxEmIterations < 0; ++it) {
            double emOld = emNew;
            EM.recomputeCovarianceMatrices(relation, (WritableDataStore<double[]>)probClusterIGivenX, models, 0.0);
            emNew = EM.assignProbabilitiesToInstances(relation, models, (WritableDataStore<double[]>)probClusterIGivenX, null);
            if (LOG.isStatistics()) {
                LOG.statistics((Statistic)new DoubleStatistic(this.getClass().getName() + ".iteration-" + it + ".logexpectation", emNew));
            }
            if (emNew - emOld <= this.emDelta) break;
        }
        if (stepProgress != null) {
            stepProgress.beginStep(6, "Generating hard clustering.", LOG);
        }
        ArrayList<ClusterCandidate> clusterCandidates = this.hardClustering((WritableDataStore<double[]>)probClusterIGivenX, clusterCores, relation.getDBIDs());
        if (stepProgress != null) {
            stepProgress.beginStep(7, "Looking for outliers and moving them to the noise set.", LOG);
        }
        this.findOutliers(relation, models, clusterCandidates, (ModifiableDBIDs)noise);
        if (stepProgress != null) {
            stepProgress.beginStep(8, "Removing empty clusters.", LOG);
        }
        Iterator<ClusterCandidate> it = clusterCandidates.iterator();
        while (it.hasNext()) {
            ClusterCandidate cand = it.next();
            int size = cand.ids.size();
            if (size >= this.minClusterSize) continue;
            if (size > 0) {
                noise.addDBIDs((DBIDs)cand.ids);
            }
            it.remove();
        }
        if (LOG.isVerbose()) {
            LOG.verbose((CharSequence)("Number of clusters remaining: " + clusterCandidates.size()));
        }
        if (stepProgress != null) {
            stepProgress.beginStep(9, "Generating final result.", LOG);
        }
        Clustering<SubspaceModel> result = new Clustering<SubspaceModel>();
        Metadata.of(result).setLongName("P3C Clustering");
        for (int cluster = 0; cluster < clusterCandidates.size(); ++cluster) {
            ClusterCandidate candidate = clusterCandidates.get(cluster);
            CovarianceMatrix cvm = CovarianceMatrix.make(relation, (DBIDs)candidate.ids);
            result.addToplevelCluster(new Cluster<SubspaceModel>((DBIDs)candidate.ids, new SubspaceModel(new Subspace(candidate.dimensions), cvm.getMeanVector())));
        }
        LOG.verbose((CharSequence)("Noise size: " + noise.size()));
        if (noise.size() > 0) {
            result.addToplevelCluster(new Cluster((DBIDs)noise, true));
        }
        LOG.ensureCompleted((FiniteProgress)stepProgress);
        return result;
    }

    private ArrayList<Signature> constructOneSignatures(SetDBIDs[][] partitions, long[][] markers) {
        int dim = partitions.length;
        ArrayList<Signature> signatures = new ArrayList<Signature>();
        for (int d = 0; d < dim; ++d) {
            SetDBIDs[] parts = partitions[d];
            if (parts == null) continue;
            long[] marked = markers[d];
            int start = BitsUtil.nextSetBit((long[])marked, (int)0);
            while (start >= 0) {
                int end = BitsUtil.nextClearBit((long[])marked, (int)(start + 1));
                end = end == -1 ? dim : end;
                int[] signature = new int[dim << 1];
                Arrays.fill(signature, -1);
                signature[d << 1] = start;
                signature[(d << 1) + 1] = end - 1;
                HashSetModifiableDBIDs sids = this.unionDBIDs((DBIDs[])parts, start, end);
                if (LOG.isDebugging()) {
                    LOG.debug((CharSequence)("1-signature: " + d + " " + start + "-" + (end - 1)));
                }
                signatures.add(new Signature(signature, (DBIDs)sids));
                start = end < dim ? BitsUtil.nextSetBit((long[])marked, (int)(end + 1)) : -1;
            }
        }
        return signatures;
    }

    private ArrayList<Signature> mergeClusterCores(int binCount, ArrayList<Signature> signatures) {
        MutableProgress mergeProgress = LOG.isVerbose() ? new MutableProgress("Merging signatures", signatures.size(), LOG) : null;
        int[] firstdim = new int[signatures.size()];
        for (int i = 0; i < signatures.size(); ++i) {
            firstdim[i] = signatures.get(i).getFirstDim();
        }
        LOG.debug((CharSequence)("First dimensions: " + FormatUtil.format((int[])firstdim)));
        ArrayList<Signature> clusterCores = new ArrayList<Signature>(signatures);
        for (int i = 0; i < clusterCores.size(); ++i) {
            Signature parent = clusterCores.get(i);
            int end = parent.getFirstDim();
            for (int j = 0; j < signatures.size() && firstdim[j] < end; ++j) {
                Signature onesig = signatures.get(j);
                Signature merge = this.mergeSignatures(parent, onesig, binCount);
                if (merge == null) continue;
                clusterCores.add(merge);
                onesig.prune = true;
                parent.prune = true;
            }
            if (mergeProgress == null) continue;
            mergeProgress.setTotal(clusterCores.size());
            mergeProgress.incrementProcessed(LOG);
        }
        if (mergeProgress != null) {
            mergeProgress.setProcessed(mergeProgress.getTotal(), LOG);
        }
        return clusterCores;
    }

    private ArrayList<Signature> pruneRedundantClusterCores(ArrayList<Signature> clusterCores) {
        ArrayList<Signature> retain = new ArrayList<Signature>(clusterCores.size());
        block0: for (Signature clusterCore : clusterCores) {
            if (clusterCore.prune) continue;
            for (int k = 0; k < clusterCores.size(); ++k) {
                Signature other = clusterCores.get(k);
                if (other != clusterCore && other.isSuperset(clusterCore)) continue block0;
            }
            if (LOG.isDebugging()) {
                LOG.debug((CharSequence)("Retained cluster core: " + clusterCore));
            }
            retain.add(clusterCore);
        }
        return retain;
    }

    private SetDBIDs[][] partitionData(Relation<? extends NumberVector> relation, int bins) {
        int dim = RelationUtil.dimensionality(relation);
        SetDBIDs[][] partitions = new SetDBIDs[dim][bins];
        ArrayModifiableDBIDs ids = DBIDUtil.newArray((DBIDs)relation.getDBIDs());
        DBIDArrayMIter iter = ids.iter();
        VectorUtil.SortDBIDsBySingleDimension sorter = new VectorUtil.SortDBIDsBySingleDimension(relation, 0);
        for (int d = 0; d < dim; ++d) {
            sorter.setDimension(d);
            ids.sort((Comparator)sorter);
            iter.seek(0);
            double min = ((NumberVector)relation.get((DBIDRef)iter)).doubleValue(d);
            iter.seek(ids.size() - 1);
            double delta = (((NumberVector)relation.get((DBIDRef)iter)).doubleValue(d) - min) / (double)bins;
            if (delta > 0.0) {
                SetDBIDs[] dimparts = partitions[d];
                double split = min + delta;
                HashSetModifiableDBIDs pids = DBIDUtil.newHashSet();
                dimparts[0] = pids;
                int i = 0;
                iter.seek(0);
                while (iter.valid()) {
                    double v = ((NumberVector)relation.get((DBIDRef)iter)).doubleValue(d);
                    if (v <= split || i == dimparts.length - 1) {
                        pids.add((DBIDRef)iter);
                    } else {
                        split += delta;
                        pids = DBIDUtil.newHashSet();
                        dimparts[++i] = pids;
                    }
                    iter.advance();
                }
                ++i;
                while (i < dimparts.length) {
                    dimparts[i] = pids;
                    ++i;
                }
                continue;
            }
            partitions[d] = null;
        }
        return partitions;
    }

    protected HashSetModifiableDBIDs unionDBIDs(DBIDs[] parts, int start, int end) {
        int sum = 0;
        for (int i = start; i < end; ++i) {
            sum += parts[i].size();
        }
        HashSetModifiableDBIDs sids = DBIDUtil.newHashSet((int)sum);
        for (int i = start; i < end; ++i) {
            sids.addDBIDs(parts[i]);
        }
        return sids;
    }

    private int chiSquaredUniformTest(SetDBIDs[] parts, long[] marked, int card) {
        int binCount;
        int max = 0;
        int maxpos = -1;
        MeanVariance mv = new MeanVariance();
        for (int i = 0; i < parts.length; ++i) {
            if (BitsUtil.get((long[])marked, (int)i)) continue;
            int binSupport = parts[i].size();
            mv.put((double)binSupport);
            if (binSupport <= max) continue;
            max = binSupport;
            maxpos = i;
        }
        if (mv.getCount() < 1.0 || !(mv.getPopulationVariance() > 0.0)) {
            return -1;
        }
        double chiSquare = mv.getPopulationVariance() / mv.getMean();
        double test = ChiSquaredDistribution.cdf((double)chiSquare, (double)Math.max(1, (binCount = parts.length - card) - card - 1));
        return 1.0 - this.alpha < test ? maxpos : -1;
    }

    private void computeFuzzyMembership(Relation<? extends NumberVector> relation, ArrayList<Signature> clusterCores, ModifiableDBIDs unassigned, WritableDataStore<double[]> probClusterIGivenX, List<MultivariateGaussianModel> models, int dim) {
        int n = relation.size();
        double pweight = 1.0 / (double)n;
        int k = clusterCores.size();
        double[] clusterWeights = new double[k];
        DBIDIter iter = relation.iterDBIDs();
        while (iter.valid()) {
            int count = 0;
            double[] weights = new double[k];
            for (int cluster = 0; cluster < k; ++cluster) {
                if (!clusterCores.get((int)cluster).ids.contains((DBIDRef)iter)) continue;
                weights[cluster] = 1.0;
                ++count;
            }
            if (count > 0) {
                VMath.timesEquals((double[])weights, (double)(1.0 / (double)count));
                VMath.plusTimesEquals((double[])clusterWeights, (double[])weights, (double)pweight);
            } else {
                unassigned.add((DBIDRef)iter);
            }
            probClusterIGivenX.put((DBIDRef)iter, (Object)weights);
            iter.advance();
        }
        for (int i = 0; i < k; ++i) {
            models.add(new MultivariateGaussianModel(clusterWeights[i], new double[dim]));
        }
    }

    private void assignUnassigned(Relation<? extends NumberVector> relation, WritableDataStore<double[]> probClusterIGivenX, List<MultivariateGaussianModel> models, ModifiableDBIDs unassigned) {
        if (unassigned.size() == 0) {
            return;
        }
        int k = models.size();
        double pweight = 1.0 / (double)relation.size();
        for (EMClusterModel eMClusterModel : models) {
            eMClusterModel.setWeight(eMClusterModel.getWeight() * (double)(relation.size() - unassigned.size()) * pweight);
        }
        DBIDMIter iter = unassigned.iter();
        while (iter.valid()) {
            NumberVector numberVector = (NumberVector)relation.get((DBIDRef)iter);
            int bestCluster = -1;
            MultivariateGaussianModel bestModel = null;
            double minDistance = Double.POSITIVE_INFINITY;
            int c = 0;
            for (MultivariateGaussianModel model : models) {
                double distance = model.mahalanobisDistance(numberVector);
                if (distance < minDistance) {
                    minDistance = distance;
                    bestCluster = c;
                    bestModel = model;
                }
                ++c;
            }
            double[] weights = new double[k];
            weights[bestCluster] = 1.0;
            if (bestModel == null) {
                throw new IllegalStateException("No models?");
            }
            bestModel.setWeight(bestModel.getWeight() + pweight);
            probClusterIGivenX.put((DBIDRef)iter, (Object)weights);
            iter.advance();
        }
        unassigned.clear();
    }

    private ArrayList<ClusterCandidate> hardClustering(WritableDataStore<double[]> probClusterIGivenX, List<Signature> clusterCores, DBIDs dbids) {
        int k = clusterCores.size();
        ArrayList<ClusterCandidate> candidates = new ArrayList<ClusterCandidate>();
        for (Signature sig : clusterCores) {
            candidates.add(new ClusterCandidate(sig));
        }
        DBIDIter iter = dbids.iter();
        while (iter.valid()) {
            double[] probs = (double[])probClusterIGivenX.get((DBIDRef)iter);
            int bestCluster = 0;
            double bestProbability = probs[0];
            for (int c = 1; c < k; ++c) {
                if (!(probs[c] > bestProbability)) continue;
                bestCluster = c;
                bestProbability = probs[c];
            }
            candidates.get((int)bestCluster).ids.add((DBIDRef)iter);
            iter.advance();
        }
        return candidates;
    }

    private void findOutliers(Relation<? extends NumberVector> relation, List<MultivariateGaussianModel> models, ArrayList<ClusterCandidate> clusterCandidates, ModifiableDBIDs noise) {
        Iterator<MultivariateGaussianModel> it = models.iterator();
        int c = 0;
        while (it.hasNext()) {
            MultivariateGaussianModel model = it.next();
            ClusterCandidate candidate = clusterCandidates.get(c);
            int dof = BitsUtil.cardinality((long[])candidate.dimensions);
            double threshold = ChiSquaredDistribution.quantile((double)(1.0 - this.alpha), (double)dof);
            DBIDMIter iter = candidate.ids.iter();
            while (iter.valid()) {
                double distance = model.mahalanobisDistance((NumberVector)relation.get((DBIDRef)iter));
                if (distance >= threshold) {
                    noise.add((DBIDRef)iter);
                    iter.remove();
                }
                iter.advance();
            }
            ++c;
        }
    }

    protected Signature mergeSignatures(Signature first, Signature second, int numBins) {
        int d2 = -1;
        for (int i = 0; i < second.spec.length; i += 2) {
            if (second.spec[i] < 0) continue;
            assert (d2 == -1) : "Merging with non-1-signature?!?";
            d2 = i;
        }
        assert (d2 >= 0) : "Merging with empty signature?";
        if (first.spec[d2] >= 0) {
            return null;
        }
        ModifiableDBIDs intersection = DBIDUtil.intersection((DBIDs)first.ids, (DBIDs)second.ids);
        int support = intersection.size();
        double width = ((double)(second.spec[d2 + 1] - second.spec[d2]) + 1.0) / (double)numBins;
        double expect = (double)first.ids.size() * width;
        if ((double)support <= expect || support < this.minClusterSize) {
            return null;
        }
        double test = PoissonDistribution.rawProbability((double)support, (double)expect);
        if (this.poissonThreshold <= test) {
            return null;
        }
        int[] spec = (int[])first.spec.clone();
        spec[d2] = second.spec[d2];
        spec[d2 + 1] = second.spec[d2 + 1];
        Signature newsig = new Signature(spec, (DBIDs)intersection);
        if (LOG.isDebugging()) {
            LOG.debug((CharSequence)newsig.toString());
        }
        return newsig;
    }

    public static class Par
    implements Parameterizer {
        public static final OptionID ALPHA_THRESHOLD_ID = new OptionID("p3c.alpha", "The significance level for uniform testing in the initial binning step.");
        public static final OptionID POISSON_THRESHOLD_ID = new OptionID("p3c.threshold", "The threshold value for the poisson test used when merging signatures.");
        public static final OptionID MAX_EM_ITERATIONS_ID = new OptionID("p3c.em.maxiter", "The maximum number of iterations for the EM step. Use -1 to run until delta convergence.");
        public static final OptionID EM_DELTA_ID = new OptionID("p3c.em.delta", "The change delta for the EM step below which to stop.");
        public static final OptionID MIN_CLUSTER_SIZE_ID = new OptionID("p3c.minsize", "The minimum size of a cluster, otherwise it is seen as noise (this is a cheat, it is not mentioned in the paper).");
        protected double alpha;
        protected double poissonThreshold;
        protected int maxEmIterations;
        protected double emDelta;
        protected int minClusterSize;

        public void configure(Parameterization config) {
            ((DoubleParameter)((DoubleParameter)new DoubleParameter(ALPHA_THRESHOLD_ID, 0.001).addConstraint((ParameterConstraint)CommonConstraints.GREATER_THAN_ZERO_DOUBLE)).addConstraint((ParameterConstraint)CommonConstraints.LESS_THAN_HALF_DOUBLE)).grab(config, x -> {
                this.alpha = x;
            });
            ((DoubleParameter)((DoubleParameter)new DoubleParameter(POISSON_THRESHOLD_ID, 1.0E-4).addConstraint((ParameterConstraint)CommonConstraints.GREATER_THAN_ZERO_DOUBLE)).addConstraint((ParameterConstraint)CommonConstraints.LESS_THAN_HALF_DOUBLE)).grab(config, x -> {
                this.poissonThreshold = x;
            });
            ((IntParameter)new IntParameter(MAX_EM_ITERATIONS_ID, 20).addConstraint((ParameterConstraint)CommonConstraints.GREATER_EQUAL_MINUSONE_INT)).grab(config, x -> {
                this.maxEmIterations = x;
            });
            ((DoubleParameter)new DoubleParameter(EM_DELTA_ID, 1.0E-5).addConstraint((ParameterConstraint)CommonConstraints.GREATER_THAN_ZERO_DOUBLE)).grab(config, x -> {
                this.emDelta = x;
            });
            ((IntParameter)new IntParameter(MIN_CLUSTER_SIZE_ID, 1).addConstraint((ParameterConstraint)CommonConstraints.GREATER_EQUAL_ONE_INT)).grab(config, x -> {
                this.minClusterSize = x;
            });
        }

        public P3C make() {
            return new P3C(this.alpha, this.poissonThreshold, this.maxEmIterations, this.emDelta, this.minClusterSize);
        }
    }

    private static class ClusterCandidate {
        public final long[] dimensions;
        public final ModifiableDBIDs ids;

        public ClusterCandidate(Signature clusterCore) {
            this.dimensions = BitsUtil.zero((int)(clusterCore.spec.length >> 1));
            for (int i = 0; i < clusterCore.spec.length; i += 2) {
                BitsUtil.setI((long[])this.dimensions, (int)(i >> 1));
            }
            this.ids = DBIDUtil.newArray((int)clusterCore.ids.size());
        }
    }

    private static class Signature {
        int[] spec;
        DBIDs ids;
        boolean prune = false;

        private Signature(int[] spec, DBIDs ids) {
            this.spec = spec;
            this.ids = ids;
        }

        public boolean isSuperset(Signature other) {
            for (int i = 1; i < this.spec.length; i += 2) {
                if (this.spec[i - 1] == other.spec[i - 1] && this.spec[i] == other.spec[i - 1] || other.spec[i - 1] == -1) continue;
                return false;
            }
            return true;
        }

        public int getFirstDim() {
            for (int i = 0; i < this.spec.length; i += 2) {
                if (this.spec[i] < 0) continue;
                return i >>> 1;
            }
            return -1;
        }

        public String toString() {
            int p = 0;
            for (int i = 0; i < this.spec.length; i += 2) {
                if (this.spec[i] < 0) continue;
                ++p;
            }
            StringBuilder buf = new StringBuilder(1000).append(p).append("-signature: ");
            for (int i = 1; i < this.spec.length; i += 2) {
                if (this.spec[i - 1] < 0) continue;
                buf.append(i >>> 1).append(':').append(this.spec[i - 1]).append('-').append(this.spec[i]).append(' ');
            }
            return buf.append(" size: ").append(this.ids.size()).toString();
        }
    }
}

